﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nn/fs/fs_Result.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/audio/audio_AudioRendererApi-os.win32.h>
#endif

#include <memory> // std::unique_ptr
#include <string>

namespace
{

const std::string g_AudioTestFileName = "/testAudio_VoiceVolumeOperation.wav";
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_TestData[1024 * 1024];

char g_WorkBuffer[1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WorkBuffer2[1024 * 1024];

char g_RecorderBuffer[16 * 1024 * 1024];

const auto VolumeMax = 1.0f;
const auto VolumeMin = 0.0f;
const auto MixVolumeMax = 0.95f;
const auto MixVolumeMin = 0.0f;

}

void PrepareSourceVoice( nnt::audio::util::MinimalRenderer* renderer, nn::audio::VoiceType &voice, nn::audio::WaveBuffer &waveBuffer);
std::size_t GenerateTestWave(void** data, int sampleRate, int frequency, int sampleCount);

TEST(VoiceVolumeOperation, Success)
{
    std::string mountPath = "";
    nnt::audio::util::GetMountPath(mountPath);

    // Test environment
    auto params = nnt::audio::util::GetAudioRendererParameterForWaveComparison();

    const std::string TargetFile = nnt::audio::util::GetResultDataDirectory() + g_AudioTestFileName;
    const std::string GoldenFile = g_RefernerceDataFolder + g_AudioTestFileName;
    const nn::TimeSpan CheckDuration = nn::TimeSpan::FromSeconds(1);
    const int32_t RecoderAudioFrameCount = 100;
    const int32_t ChannelCount = 2;
    const int32_t BitRate = 16;
    const size_t SendReturnBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecoderAudioFrameCount, ChannelCount);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::os::Event event(nn::os::EventClearMode_AutoClear);
    nn::audio::SetFastRenderingMode(true, &event);
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

    // Prepare Test Components
    /////////////////////////////////////////////////////////////////////////////
    nnt::audio::util::SimpleFsUtility fsUtil;
    nnt::audio::util::MinimalRenderer renderer(g_WorkBuffer, sizeof(g_WorkBuffer));
    nnt::audio::util::Recorder recorder;
    nnt::audio::util::WaveFile testWav;

    //fsUtil->InitializeFileSystem(MountName, MountPath);
    ASSERT_TRUE(fsUtil.InitializeFileSystem(g_MountName.c_str(), mountPath.c_str()).IsSuccess());
    renderer.Initialize(params);

    int64_t estimatedFileSize = nnt::audio::util::WaveFile::GetRequiredFileSize(params.sampleRate, ChannelCount, BitRate, CheckDuration);
    nn::Result rt = fsUtil.PrepareNewFile(TargetFile.c_str(), estimatedFileSize);
    ASSERT_TRUE(rt.IsSuccess()) << "Failed to create new file.";
    testWav.Open(TargetFile.c_str(), params.sampleRate, ChannelCount, BitRate);


    // Recorder Setup
    //////////////////////////////////////////////////////////////////////////
    nn::audio::MemoryPoolType pool, sourcePool;
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &pool, g_WorkBuffer2, sizeof(g_WorkBuffer2)));
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(renderer.GetConfig(), &sourcePool, g_TestData, sizeof(g_TestData)));
    RequestAttachMemoryPool(&pool);
    RequestAttachMemoryPool(&sourcePool);
    nn::mem::StandardAllocator allocator(g_WorkBuffer2, sizeof(g_WorkBuffer2));

    // Setup AuxType
    nn::audio::AuxType aux;
    auto bufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&params, RecoderAudioFrameCount, ChannelCount);
    auto sendBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    auto returnBuffer = static_cast<int32_t*>(allocator.Allocate(bufferSize, nn::audio::BufferAlignSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::audio::AddAux(renderer.GetConfig(), &aux, renderer.GetFinalMix(), sendBuffer,  returnBuffer, bufferSize));
    int8_t inputIndex[ChannelCount] = {0, 1};
    int8_t outputIndex[ChannelCount] = {0, 1};
    nn::audio::SetAuxInputOutput(&aux, inputIndex, outputIndex, ChannelCount);

    // Attach AuxType to Recorder & WaveFile.
    ASSERT_TRUE(recorder.Prepare(&params, &aux, &testWav, SendReturnBufferSize, CheckDuration, g_RecorderBuffer, sizeof(g_RecorderBuffer)).IsSuccess());

    // Prepare Target Voice.
    //////////////////////////////////////////////////////////////////////////
    nn::audio::VoiceType voice;
    nn::audio::WaveBuffer waveBuffer;
    PrepareSourceVoice(&renderer, voice, waveBuffer);

    // start running
    //////////////////////////////////////////////////////////////////////////
    recorder.Start();
    renderer.Start();

    // keep playing at least for recording period.
    auto count = 0;
    ASSERT_TRUE(VolumeMax <= nn::audio::VoiceType::GetVolumeMax());
    ASSERT_TRUE(VolumeMin >= nn::audio::VoiceType::GetVolumeMin());
    while (recorder.Record())
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::audio::TriggerRendering();
        event.Wait();
#endif
        renderer.Wait();
        if (++count % 20 == 0)
        {
            if (count == 20)
            {
                nn::audio::SetVoiceMixVolume(&voice, renderer.GetFinalMix(), MixVolumeMin, 0, 0);
                nn::audio::SetVoiceMixVolume(&voice, renderer.GetFinalMix(), MixVolumeMin, 0, 1);
            }
            else if (count == 40)
            {
                nn::audio::SetVoiceMixVolume(&voice, renderer.GetFinalMix(), MixVolumeMax, 0, 0);
                nn::audio::SetVoiceMixVolume(&voice, renderer.GetFinalMix(), MixVolumeMax, 0, 1);
            }
            else
            {
                const auto Volume = nn::audio::GetVoiceVolume(&voice) == VolumeMin ? VolumeMax : VolumeMin;
                nn::audio::SetVoiceVolume(&voice, Volume);
            }
        }
        renderer.Update();
    }

    // finish running
    //////////////////////////////////////////////////////////////////////////
    renderer.Stop();
    recorder.Stop();
    testWav.Close();

    allocator.Free(sendBuffer);
    allocator.Free(returnBuffer);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // TODO: it's too difficult on Windows to ensure the timing; We should use FastRendering
    nnt::audio::util::SimpleVerify(TargetFile, GoldenFile);
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)
}

std::size_t GenerateTestWave(void* data, int sampleRate, int frequency, int sampleCount)
{
    // Generating square wave
    const int WaveLength = sampleRate / frequency;
    const int16_t Amplitude = std::numeric_limits<int16_t>::max();
    int count = 0;

    int16_t* p = static_cast<int16_t*>(data);
    for (int i = 0; i < sampleCount; ++i)
    {
        p[i] = static_cast<int16_t>(count < (WaveLength / 2) ? Amplitude : - Amplitude);
        if (++count == WaveLength)
        {
            count = 0;
        }
    }
    return sampleCount * sizeof(int16_t);
}

void PrepareSourceVoice( nnt::audio::util::MinimalRenderer* renderer, nn::audio::VoiceType &voice, nn::audio::WaveBuffer &waveBuffer)
{
    const int sineSampleRate = 48000;
    const int sineFrequency = 440;
    const int sineSampleCount = 48000;
    std::size_t size = GenerateTestWave(g_TestData, sineSampleRate, sineFrequency, sineSampleCount);
    nn::audio::AcquireVoiceSlot(renderer->GetConfig(), &voice, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(renderer->GetConfig(), &voice, renderer->GetFinalMix());

    waveBuffer.buffer = g_TestData;
    waveBuffer.size = size;
    waveBuffer.startSampleOffset = 0;
    waveBuffer.endSampleOffset = sineSampleCount;
    waveBuffer.loop = 0;
    waveBuffer.isEndOfStream = false;

    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceVolume(&voice, VolumeMax);
    nn::audio::SetVoiceMixVolume(&voice, renderer->GetFinalMix(), MixVolumeMax, 0, 0);
    nn::audio::SetVoiceMixVolume(&voice, renderer->GetFinalMix(), MixVolumeMax, 0, 1);
}



