﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <memory>

#include <nnt.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#include <nn/audio.h>
#include <nn/mem.h>

namespace
{
struct StdAllocatorDeleter
{
    static void SetupDeleter(nn::mem::StandardAllocator* pAllocator)
    {
        m_Allocator = pAllocator;
    }
    static nn::mem::StandardAllocator* m_Allocator;
    void operator()(void* ptr) const
    {
        NN_ABORT_UNLESS_NOT_NULL(m_Allocator);
        m_Allocator->Free(ptr);
    }
};

nn::mem::StandardAllocator* StdAllocatorDeleter::m_Allocator = nullptr;
using StdUniquePtr = std::unique_ptr<void, StdAllocatorDeleter>;

struct TestContext
{
    std::atomic_bool                running;
    nnt::audio::util::ScopedAudioRenderer* pRenderer;
    nn::os::SystemEvent             event;
    nn::os::Event                   syncEvent;
    nn::mem::StandardAllocator*     pAllocator;

    TestContext()
        : pRenderer { nullptr }
        , syncEvent{ nn::os::EventClearMode_AutoClear }
        , pAllocator{ nullptr }
        {
            running = false;
            syncEvent.Clear();
        }
};

void UpdaterFunc(void* args)
{
    auto pContext = reinterpret_cast<TestContext*>(args);
    while(pContext->running)
    {
        pContext->event.Wait();
        pContext->syncEvent.Signal();
        nn::audio::RequestUpdateAudioRenderer(pContext->pRenderer->GetHandle(), &pContext->pRenderer->GetConfig());
    }
}

void AccessorFunc(void* args)
{
    auto pContext = reinterpret_cast<TestContext*>(args);
    nn::audio::VoiceType voice[20];

    nn::audio::MemoryPoolType pool;
    StdUniquePtr poolBuf{ pContext->pAllocator->Allocate(nn::os::MemoryPageSize, nn::os::MemoryPageSize) };
    ASSERT_TRUE(poolBuf != nullptr);
    memset(poolBuf.get(), 0, nn::os::MemoryPageSize);
    ASSERT_TRUE(nn::audio::AcquireMemoryPool(&pContext->pRenderer->GetConfig(), &pool, poolBuf.get(), nn::os::MemoryPageSize));
    ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&pool));

    nn::audio::WaveBuffer wb;
    wb.buffer = poolBuf.get();
    wb.size = nn::os::MemoryPageSize;
    wb.startSampleOffset = 0;
    wb.endSampleOffset = nn::os::MemoryPageSize / 4;
    wb.loop = false;
    wb.isEndOfStream = true;
    wb.pContext = nullptr;
    wb.contextSize = 0;

    while(pContext->running)
    {
        pContext->syncEvent.Wait();
        for (auto&& v : voice)
        {
            ASSERT_TRUE(nn::audio::AcquireVoiceSlot(&pContext->pRenderer->GetConfig(), &v, 48000, 2, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
            // wait for SIGLO-52638 fixed.
            // nn::audio::AppendWaveBuffer(&v, &wb);
            // nn::audio::SetVoicePlayState(&v, nn::audio::VoiceType::PlayState_Play);
            nn::audio::ReleaseVoiceSlot(&pContext->pRenderer->GetConfig(), &v);
        }
    }
}



} // namespace


void TestMain(nn::mem::StandardAllocator& allocator)
{
    TestContext context;
    context.pAllocator = &allocator;

    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.sampleCount = 240;
    parameter.mixBufferCount = 2;
    parameter.voiceCount = 24;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 0;
    parameter.performanceFrameCount = 0;

    nnt::audio::util::ScopedAudioRenderer sar{ parameter, &context.event };
    sar.Start();
    context.pRenderer = &sar;

    size_t stackSize = nn::os::MemoryPageSize * 4;
    StdUniquePtr updaterStack{ allocator.Allocate(stackSize, nn::os::MemoryPageSize) };
    StdUniquePtr accessorStack{ allocator.Allocate(stackSize, nn::os::MemoryPageSize) };
    ASSERT_TRUE(updaterStack != nullptr);
    ASSERT_TRUE(accessorStack != nullptr);


    nn::os::ThreadType updaterThread;
    nn::os::ThreadType accessorThread;

    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread(&updaterThread,
                             UpdaterFunc,
                             &context,
                             updaterStack.get(),
                             stackSize,
                             nn::os::DefaultThreadPriority,
                             0));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::os::CreateThread(&accessorThread,
                             AccessorFunc,
                             &context,
                             accessorStack.get(),
                             stackSize,
                             nn::os::DefaultThreadPriority,
                             1));


    context.running = true;
    nn::os::StartThread(&updaterThread);
    nn::os::StartThread(&accessorThread);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    context.running = false;

    nn::os::WaitThread(&updaterThread);
    nn::os::WaitThread(&accessorThread);
    nn::os::DestroyThread(&updaterThread);
    nn::os::DestroyThread(&accessorThread);
}

TEST(DISABLED_VoiceAllocation, Success)
{
    size_t bufferSize = 1 * 1024 * 1024;
    void* buffer = malloc(bufferSize);
    ASSERT_TRUE(buffer != nullptr);

    nn::mem::StandardAllocator allocator;
    allocator.Initialize(buffer, bufferSize);
    StdAllocatorDeleter::SetupDeleter(&allocator);

    TestMain(allocator);

    allocator.Finalize();
    if (buffer)
    {
        free(buffer);
        buffer = nullptr;
    }
}


