﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <memory>
#include <string>

#include <nnt.h>
#include <nn/audio.h>
#include <nn/fs/fs_Result.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/os/os_SdkThread.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief
 */
TEST(RunAudioRendererWithBusySystemCore, Success)
{
    // Set up a thread that the system core becomes busy.
    nn::os::ThreadType busyThread;
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t busyThreadStack[4096];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&busyThread,
                        [](void* arg)
                        {
                            for(int i = 0; i < 3; ++i)
                            {
                                const auto beginTicks = nn::os::GetSystemTick();
                                do
                                {
                                    // do nothing
                                } while(nn::os::GetSystemTick() - beginTicks < nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(1000 * 3)));

                                // Sleep
                                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                            }
                        },
                        nullptr,
                        busyThreadStack,
                        sizeof(busyThreadStack),
                        nn::os::HighestSystemThreadPriority,
                        3));
    nn::os::StartThread(&busyThread);

    // set up AudioRenderer
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.mixBufferCount = 16;
    parameter.voiceCount =24;

    nn::os::SystemEvent systemEvent;

    nnt::audio::util::ScopedAudioRenderer sar(parameter, &systemEvent);

    {
        nn::audio::VoiceType voice;
        nn::audio::FinalMixType finalMix;
        EXPECT_TRUE(nn::audio::AcquireVoiceSlot(&sar.GetConfig(), &voice, 48000, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0));
        EXPECT_TRUE(nn::audio::AcquireFinalMix(&sar.GetConfig(), &finalMix, 1));
        nn::audio::SetVoiceDestination(&sar.GetConfig(), &voice, &finalMix);

        nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);

        nn::audio::MemoryPoolType memoryPool;
        NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t pcmBuffer[nn::audio::MemoryPoolType::SizeGranularity];

        ASSERT_TRUE(nn::audio::AcquireMemoryPool(&sar.GetConfig(), &memoryPool, pcmBuffer, sizeof(pcmBuffer)));
        ASSERT_TRUE(nn::audio::RequestAttachMemoryPool(&memoryPool));

        nn::audio::WaveBuffer waveBuffer;
        waveBuffer.buffer = pcmBuffer;
        waveBuffer.size = parameter.sampleCount * sizeof(pcmBuffer[0]);
        waveBuffer.startSampleOffset = 0;
        waveBuffer.endSampleOffset = static_cast<int32_t>(waveBuffer.size / sizeof(pcmBuffer[0]));
        waveBuffer.loop = 0;
        waveBuffer.isEndOfStream = false;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer);
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(sar.GetHandle(), &sar.GetConfig()));
    }

    // Wait for the busy thread.
    nn::os::WaitThread(&busyThread);
    nn::os::DestroyThread(&busyThread);
}
#endif
