﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::max
#include <cstdlib>    // std::aligned_alloc
#include <cstring>    // std::memcmp
#include <memory>
#include <string>
#include <vector>

#include <nnt.h>
#include <nnt/audioUtil/testAudio_Constants.h>
#include <nnt/audioUtil/testAudio_Util.h>

#include <nn/audio.h>
#include <nn/mem.h>

#include "../../../../Programs/Eris/Sources/Libraries/audio/common/audio_BitArray.h"


template <typename StorageType>
class BitArrayTest : public ::testing::Test
{
public:
    using ba_t = nn::audio::common::BitArray<StorageType>;
};

typedef ::testing::Types<uint64_t, uint32_t, uint16_t, uint8_t> StorageTypeSet;
TYPED_TEST_CASE(BitArrayTest, StorageTypeSet);

TYPED_TEST(BitArrayTest, Success)
{
    using ba_t = typename TestFixture::ba_t;

    for (uint32_t bitCount : { 1, 2,
                             7, 8, 9,
                             15, 16, 17,
                             31, 32, 33,
                             63, 64, 65,
                             127, 128, 129,
                             64 * 64, 64 * 64 + 1 })
    {
        size_t reqSize = ba_t::CalcBufferSize(bitCount);
        auto buf = std::unique_ptr<uint8_t[]>(new uint8_t[reqSize]);

        ba_t ba;
        ba.Initialize(buf.get(), reqSize);

        // after initialized
        for (uint32_t i = 0; i < static_cast<uint32_t>(bitCount); ++i)
        {
            EXPECT_TRUE(ba[i] == false);
        }
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any() == false);
        EXPECT_TRUE(ba.None());
        EXPECT_TRUE(ba.Count() == 0);
        EXPECT_TRUE(ba.Size() >= bitCount);

        // simple set/reset
        ba.Set(bitCount / 2);
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any());
        EXPECT_TRUE(ba.None() == false);
        EXPECT_TRUE(ba.Count() == 1);
        ba.Reset(bitCount / 2);
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any() == false);
        EXPECT_TRUE(ba.None());
        EXPECT_TRUE(ba.Count() == 0);
        EXPECT_TRUE(ba.Size() >= bitCount);

        // set with "false"
        ba.Set(bitCount / 2);
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any());
        EXPECT_TRUE(ba.None() == false);
        EXPECT_TRUE(ba.Count() == 1);
        ba.Set(bitCount / 2, false);
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any() == false);
        EXPECT_TRUE(ba.None());
        EXPECT_TRUE(ba.Count() == 0);
        EXPECT_TRUE(ba.Size() >= bitCount);

        // all bit operation
        ba.Set();
        EXPECT_TRUE(ba.All());
        EXPECT_TRUE(ba.Any());
        EXPECT_TRUE(ba.None() == false);
        EXPECT_TRUE(ba.Count() == ba.Size());
        ba.Reset();
        EXPECT_TRUE(ba.All() == false);
        EXPECT_TRUE(ba.Any() == false);
        EXPECT_TRUE(ba.None());
        EXPECT_TRUE(ba.Count() == 0);

        // 最低でも 2 bit が必要なチェック
        if (bitCount >= 2)
        {
            // boundary check
            EXPECT_TRUE(ba.Set(0)[0]);
            EXPECT_TRUE(ba.Reset(0)[0] == false);
            EXPECT_TRUE(ba.Flip(0)[0]);
            EXPECT_TRUE(ba.Set(bitCount - 1)[bitCount - 1]);
            EXPECT_TRUE(ba.Reset(bitCount - 1)[bitCount - 1] == false);
            EXPECT_TRUE(ba.Flip(bitCount - 1)[bitCount - 1]);
            EXPECT_TRUE(ba.Count() == 2);
        }

        // 適当に選んだ要素が正しく set されるかのチェック
        ba.Reset();
        std::vector<uint32_t> checkPos = { 12, bitCount, 5, 22, 0 };
        if (ba.Size() > *std::max_element(checkPos.begin(), checkPos.end()))
        {
            std::vector<uint32_t> resultPos = {};
            for (auto pos : checkPos)
            {
                ba.Set(pos);
            }
            for (uint32_t i = 0; i < ba.Size(); ++i)
            {
                if (ba[i])
                {
                    resultPos.push_back(i);
                }
            }

            std::sort(checkPos.begin(), checkPos.end());
            EXPECT_TRUE(checkPos == resultPos);
        }
    }
}
