﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/util/util_BytePtr.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/audio/audio_AudioIn.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include "../../../../../Programs/Eris/Sources/Libraries/audio/common/audio_AudioInPrivate.h"
#include "testAudio_ScopedAudioIn.h"

namespace {

const size_t g_BufferSize = nn::audio::AudioInBuffer::SizeGranularity;
const size_t g_DataSize = 1024;
NN_AUDIO_ALIGNAS_AUDIO_IN_BUFFER_ALIGN char g_Buffer[g_BufferSize];

}

/**
 * @brief       InitializeAudioInParameter() の正常系テストです。
 */
TEST(InitializeAudioInParameter, Success)
{
    nn::audio::AudioInParameter parameter;

    nn::audio::InitializeAudioInParameter(&parameter);
    EXPECT_EQ(parameter.sampleRate, 0);
}

/**
 * @brief       ListAudioIns() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ListAudioIns, PreCondition)
{
    nn::audio::AudioInInfo audioInInfos[nn::audio::AudioInCountMax];

    // outAudioIns == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ListAudioIns(nullptr, sizeof(audioInInfos) / sizeof(*audioInInfos)), "");

    // count < 0
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ListAudioIns(audioInInfos, -1), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       OpenAudioIn() の正常系テストです。
 */
TEST(OpenAudioIn, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    int sampleRate;

    // With default
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
        sampleRate = GetAudioInSampleRate(audioIn.Get());
        EXPECT_EQ(
            nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
    }

    // With non-default sample rate
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        parameter.sampleRate = sampleRate;
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
        EXPECT_EQ(
            nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
    }

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    // With non-default sample reate on NX
    {
        int sampleRates[] = { 48000 };
        for(auto sampleRate : sampleRates)
        {
            nn::audio::AudioInParameter parameter;
            nn::audio::InitializeAudioInParameter(&parameter);
            parameter.sampleRate = sampleRate;
            ScopedAudioIn audioIn;
            NNT_EXPECT_RESULT_SUCCESS(
                nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
            EXPECT_EQ(
                nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
        }
    }
#endif

    // TODO: 2 つ同時にオープン出来るようになったら再度有効にする
#if 0
    // 同時に 2 つオープンする
    {
        nn::audio::AudioInParameter parameter0, parameter1;
        nn::audio::InitializeAudioInParameter(&parameter0);
        nn::audio::InitializeAudioInParameter(&parameter1);
        parameter1.sampleRate = sampleRate;
        ScopedAudioIn audioIn0, audioIn1;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(audioIn0.Get(), "BuiltInHeadset", parameter0));
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(audioIn1.Get(), "BuiltInHeadset", parameter1));
    }
#endif
}

/**
 * @brief       OpenAudioIn() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpenAudioIn, PreCondition)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioIn audioIn;

    // With default
    {
        // pAudioIn == nullptr
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioIn(nullptr, "BuiltInHeadset", parameter), "");

        // name == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioIn(&audioIn, nullptr, parameter), "");
    }

    // With non-default sample rate
    {
        // pAudioIn == nullptr
        nn::audio::AudioInParameter parameter0;
        nn::audio::InitializeAudioInParameter(&parameter0);
        parameter0.sampleRate = 48000;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioIn(nullptr, "BuiltInHeadset", parameter0), "");

        // name == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioIn(&audioIn, nullptr, parameter0), "");

        // sampleRate < 0
        nn::audio::AudioInParameter parameter1;
        nn::audio::InitializeAudioInParameter(&parameter1);
        parameter1.sampleRate = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenAudioIn(&audioIn, "BuiltInHeadset", parameter1), "");
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       OpenAudioIn() のエラー処理テストです。
 */
TEST(OpenAudioIn, Error)
{
    const char InvalidAudioInName[] = "____InvalidAudioInName0123456789";  // おおよそ ありえない であろう名前
    const int InvalidSampleRate = 1;  // おおよそ ありえない であろうサンプルレート

    nn::audio::AudioIn audioIn;

    // OpenAudioIn(AudioIn*, const char[], int)
    {
        nn::audio::AudioInParameter parameter0;
        nn::audio::InitializeAudioInParameter(&parameter0);
        parameter0.sampleRate = 48000;
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultNotFound, nn::audio::OpenAudioIn(&audioIn, InvalidAudioInName, parameter0));

        nn::audio::AudioInParameter parameter1;
        nn::audio::InitializeAudioInParameter(&parameter1);
        parameter1.sampleRate = InvalidSampleRate;
        nn::audio::AudioInInfo audioInInfo;
        int count = nn::audio::ListAudioIns(&audioInInfo, 1);
        EXPECT_LE(count, 1);
        EXPECT_GE(count, 0);
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultInvalidSampleRate, nn::audio::OpenAudioIn(&audioIn, "BuiltInHeadset", parameter1));
    }
}

/**
 * @brief       CloseAudioIn() の正常系テストです。
 */
TEST(CloseAudioIn, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    nn::audio::CloseAudioIn(audioIn.Get());
}

/**
 * @brief       CloseAudioIn() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(CloseAudioIn, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::CloseAudioIn(nullptr), "");

    // GetAudioInState(pAudioIn) == nn::audio::AudioInState_Started
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Started);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::CloseAudioIn(audioIn.Get()), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       StartAudioIn() の正常系テストです。
 */
TEST(StartAudioIn, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Started);
}

/**
 * @brief       StartAudioIn() の事前条件違反テストです。
 */
TEST(StartAudioIn, PreCondition)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;

#if defined(NN_SDK_BUILD_RELEASE)
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));

    //  Precondition violation: Start AudioIn when it is AudioInState_Started.
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOperationFailed, nn::audio::StartAudioIn(audioIn.Get()));
#else
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StartAudioIn(nullptr), "");

    // GetAudioInState(pAudioIn) == nn::audio::AudioInState_Started
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Started);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StartAudioIn(audioIn.Get()), "");
#endif
}

/**
 * @brief       StopAudioIn() の正常系テストです。
 */
TEST(StopAudioIn, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));
    nn::audio::StopAudioIn(audioIn.Get());
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
}

/**
 * @brief       StopAudioIn() の事前条件違反テストです。
 */
TEST(StopAudioIn, PreCondition)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;

#if defined(NN_SDK_BUILD_RELEASE)
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);

    //  Precondition violation: Stop AudioIn when it is AudioInState_Stopped.
    nn::audio::StopAudioIn(audioIn.Get());

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));
    nn::audio::StopAudioIn(audioIn.Get());

    //  Precondition violation: Stop AudioIn when it is AudioInState_Stopped.
    nn::audio::StopAudioIn(audioIn.Get());
#else
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StopAudioIn(nullptr), "");

    // GetAudioInState(pAudioIn) == nn::audio::AudioInState_Stopped
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    EXPECT_EQ(
        nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::StopAudioIn(audioIn.Get()), "");
#endif  // !defined(NN_SDK_BUILD_RELEASE)
}

/**
 * @brief       GetAudioInState() の正常系テストです。
 */
TEST(GetAudioInState, Success)
{
}

/**
 * @brief       GetAudioInState() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioInState, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioInState(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioInName() の正常系テストです。
 */
TEST(GetAudioInNameTest, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    EXPECT_STREQ(nn::audio::GetAudioInName(audioIn.Get()), "BuiltInHeadset");
#else
    EXPECT_STRNE(nn::audio::GetAudioInName(audioIn.Get()), "");
#endif
}

/**
 * @brief       GetAudioInName() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioInName, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioInName(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioInSampleRate() の正常系テストです。
 */
TEST(GetAudioInSampleRate, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    EXPECT_GT(
        nn::audio::GetAudioInSampleRate(audioIn.Get()), 0);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    EXPECT_EQ(
        nn::audio::GetAudioInSampleRate(audioIn.Get()), 48000);
#endif
}

/**
 * @brief       GetAudioInSampleRate() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioInSampleRate, PreCondition)
{
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioInSampleRate(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioInChannelCount() の正常系テストです。
 */
TEST(GetAudioInChannelCount, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    EXPECT_GT(
        nn::audio::GetAudioInChannelCount(audioIn.Get()), 0);
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    EXPECT_EQ(
        nn::audio::GetAudioInChannelCount(audioIn.Get()), 2);
#endif
}

/**
 * @brief       GetAudioInChannelCount() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioInChannelCount, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioInChannelCount(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetAudioInSampleFormat() の正常系テストです。
 */
TEST(GetAudioInSampleFormat, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));
    EXPECT_NE(
        nn::audio::GetAudioInSampleFormat(audioIn.Get()), nn::audio::SampleFormat_Invalid);
}

/**
 * @brief       GetAudioInSampleFormat() の事前条件テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetAudioInSampleFormat, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetAudioInSampleFormat(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       AppendAudioInBuffer() の正常系テストです。
 */
TEST(AppendAudioInBuffer, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));

    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, g_DataSize);
    for(int i = 0; i < nn::audio::AudioInBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer));
        EXPECT_EQ(nn::audio::GetAudioInBufferCount(audioIn.Get()), i + 1);
    }

    EXPECT_FALSE(nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer));
}

/**
 * @brief       AppendAudioInBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(AppendAudioInBuffer, PreCondition)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);

    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));

    nn::audio::AudioInBuffer audioInBuffer;

    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioInBuffer(nullptr, &audioInBuffer), "");

    // pAudioInBuffer == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioInBuffer(audioIn.Get(), nullptr), "");

    // pAudioInBuffer.buffer == nullptr
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, g_DataSize);
    audioInBuffer.buffer = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer), "");

    // GetAudioInBufferDataPointer(pAudioInBuffer) is multiple of nn::audio::BufferAlignSize
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, g_DataSize - 1);
    audioInBuffer.buffer = g_Buffer + 1;
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer), "");

    // pAudioInBuffer->size % (nn::audio::GetAudioInChannelCount(pAudioIn) * nn::audio::GetSampleByteSize(nn::audio::GetAudioInSampleFormat(pAudioIn))) == 0
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, 1);
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       GetReleasedAudioInBufferTest() の正常系テストです。
 */
TEST(GetReleasedAudioInBufferTest, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), "BuiltInHeadset", parameter));

    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, g_DataSize);
    nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);
    EXPECT_EQ(nn::audio::GetAudioInBufferCount(audioIn.Get()), 1);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));

    nn::audio::AudioInBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        pReleased = nn::audio::GetReleasedAudioInBuffer(audioIn.Get());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }

    EXPECT_EQ(pReleased, &audioInBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);
}

/**
 * @brief       GetReleasedAudioInBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(GetReleasedAudioInBufferTest, PreCondition)
{
    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::GetReleasedAudioInBuffer(nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       ContainsAudioInBuffer() の正常系テストです。
 */
TEST(ContainsAudioInBufferTest, Success)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn ain;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(ain.Get(), "BuiltInHeadset", parameter));

    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, g_DataSize);

    EXPECT_FALSE(nn::audio::ContainsAudioInBuffer(ain.Get(), &audioInBuffer));

    nn::audio::AppendAudioInBuffer(ain.Get(), &audioInBuffer);

    EXPECT_TRUE(nn::audio::ContainsAudioInBuffer(ain.Get(), &audioInBuffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(ain.Get()));

    nn::audio::AudioInBuffer* pReleased = nullptr;
    while (pReleased == nullptr)
    {
        pReleased = nn::audio::GetReleasedAudioInBuffer(ain.Get());
    }

    EXPECT_FALSE(nn::audio::ContainsAudioInBuffer(ain.Get(), &audioInBuffer));
}

/**
 * @brief       ContainsAudioInBuffer() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(ContainsAudioInBufferTest, PreCondition)
{
    nn::audio::AudioIn audioIn;
    nn::audio::AudioInBuffer audioInBuffer;

    // pAudioIn == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ContainsAudioInBuffer(nullptr, &audioInBuffer), "");
    // pAudioInBuffer == nullptr
    EXPECT_DEATH_IF_SUPPORTED(
        nn::audio::ContainsAudioInBuffer(&audioIn, nullptr), "");
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

// TODO: 複数個の AudioIn が扱えるようになったら有効にする
#if 0
/**
 * @brief       AudioIn によるマルチスレッドでの再生の正常系テストです。
 */
TEST(PlaybackAudioInMultiThread, Success)
{
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack1[32768];
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t threadStack2[32768];
    nn::os::ThreadType thread1, thread2;

    const auto threadFunc = [](void* args)
    {
        NN_UNUSED(args);

        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        nn::audio::AudioIn audioIn;
        nn::os::SystemEvent systemEventOut;

        for(int i = 0; i < 10; ++i)
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(&audioIn, &systemEventOut, "BuiltInHeadset", parameter));
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(&audioIn));

            nn::audio::AudioInBuffer audioInBuffer;
            // Set mininum buffer size
            nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, sizeof(g_Buffer), nn::audio::GetAudioInChannelCount(&audioIn) * nn::audio::GetSampleByteSize(nn::audio::GetAudioInSampleFormat(&audioIn)));
            nn::audio::AppendAudioInBuffer(&audioIn, &audioInBuffer);

            const auto beginTick = nn::os::GetSystemTick();
            while(nn::os::GetSystemTick() - beginTick < nn::os::ConvertToTick(nn::TimeSpan::FromMilliSeconds(100)))
            {
                if(nn::audio::GetReleasedAudioInBuffer(&audioIn) != nullptr)
                {
                    nn::audio::AppendAudioInBuffer(&audioIn, &audioInBuffer);
                }

                // 偶数回目の時は SystemEvent がシグナル状態になるのを待つパターンでテスト
                if(i % 2 == 0)
                {
                    systemEventOut.Wait();
                }
            }

            nn::audio::StopAudioIn(&audioIn);
            nn::audio::CloseAudioIn(&audioIn);
        }
    };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread1, threadFunc, nullptr, threadStack1, sizeof(threadStack1), nn::os::DefaultThreadPriority));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&thread2, threadFunc, nullptr, threadStack2, sizeof(threadStack2), nn::os::DefaultThreadPriority));
    nn::os::SetThreadCoreMask(&thread1, 1, 1 << 1);
    nn::os::SetThreadCoreMask(&thread2, 2, 1 << 2);
    nn::os::StartThread(&thread1);
    nn::os::StartThread(&thread2);
    nn::os::WaitThread(&thread1);
    nn::os::WaitThread(&thread2);
    nn::os::DestroyThread(&thread1);
    nn::os::DestroyThread(&thread2);
}
#endif

/**
 * @brief       AudioInBuffer の操作を行う API の事前条件テストです。
 */
TEST(AudioInBuffer, Precondition)
{
    uint8_t* pBuffer = reinterpret_cast<uint8_t*>(nn::audio::AudioInBuffer::AddressAlignment);
    const size_t bufferSize = nn::audio::AudioInBuffer::SizeGranularity * 2;
    const auto channelCount = 2;
    const auto sampleSize = sizeof(int16_t);
    const auto sampleCount = 100;
    const size_t dataSize = channelCount * sampleSize * sampleCount;
    nn::audio::AudioInBuffer inBuffer;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(nullptr, pBuffer, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, nullptr, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, nullptr, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, pBuffer + 1, bufferSize, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, pBuffer, bufferSize + 1, dataSize), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, pBuffer, bufferSize, bufferSize + 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInBufferInfo(&inBuffer, pBuffer, bufferSize, 0), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioInBufferDataPointer(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioInBufferDataSize(nullptr), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioInBufferBufferSize(nullptr), "");
}

/**
 * @brief       AudioInBuffer の操作を行う API の正常系テストです。
 */
TEST(AudioInBuffer, Success)
{
    void* pBuffer = reinterpret_cast<void*>(nn::audio::AudioInBuffer::AddressAlignment);
    const size_t bufferSize = nn::audio::AudioInBuffer::SizeGranularity * 2;
    const auto channelCount = 2;
    const auto sampleSize = sizeof(int16_t);
    const auto sampleCount = 100;
    const size_t dataSize = channelCount * sampleSize * sampleCount;
    nn::audio::AudioInBuffer outBuffer;

    nn::audio::SetAudioInBufferInfo(&outBuffer, pBuffer, bufferSize, dataSize);
    EXPECT_EQ(pBuffer, nn::audio::GetAudioInBufferDataPointer(&outBuffer));
    EXPECT_EQ(dataSize, nn::audio::GetAudioInBufferDataSize(&outBuffer));
    EXPECT_EQ(bufferSize, nn::audio::GetAudioInBufferBufferSize(&outBuffer));
}

TEST(GetAudioInBufferCount, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioInBufferCount(nullptr), "");
}

TEST(GetAudioInBufferCount, Success)
{
    nn::audio::AudioInParameter parameter;

    // Setup AudioIn
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn ain;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(ain.Get(), &systemEvent, "BuiltInHeadset", parameter));

    // Check initial count
    EXPECT_EQ(0, nn::audio::GetAudioInBufferCount(ain.Get()));

    // Append 1 AudioInBuffer
    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    nn::audio::AppendAudioInBuffer(ain.Get(), &audioInBuffer); // 43 ms

    // Check AudioIn's buffer count is 1
    EXPECT_EQ(1, nn::audio::GetAudioInBufferCount(ain.Get()));

    // Start AudioIn
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(ain.Get()));

    // Wait for consuming AudioInBuffer by AudioIn
    const auto RetryCountMax = 10; // 500 ms
    int i;
    for(i = 0; i < RetryCountMax; ++i)
    {
        const auto CurrentBufferCount = nn::audio::GetAudioInBufferCount(ain.Get());

        EXPECT_TRUE(CurrentBufferCount == 1 || CurrentBufferCount == 0);

        if(CurrentBufferCount == 0)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));
    }

    if(i == RetryCountMax - 1)
    {
        ASSERT_TRUE(false) << "AudioInBufferCount is not 0.";
    }

    // Check AudioInBufferCount is not changed even though GetReleasedAudioInBuffer() is called
    EXPECT_NE(nullptr, nn::audio::GetReleasedAudioInBuffer(ain.Get()));
    EXPECT_EQ(0, nn::audio::GetAudioInBufferCount(ain.Get()));
}
