﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>

#include <nn/util/util_BytePtr.h>
#include <nn/audio/audio_Applet.h>
#include <nn/audio/audio_Debugger.h>
#include <nn/applet/applet_Apis.h>
#include <nn/audio/audio_AudioIn.h>
#include <nnt/audioUtil/testAudio_Util.h>
#include "../../../../../Programs/Eris/Sources/Libraries/audio/common/audio_AudioInPrivate.h"
#include "testAudio_ScopedAudioIn.h"

#if !defined(NN_SDK_BUILD_RELEASE)
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    EXPECT_DEATH_IF_SUPPORTED(func, "")
#else
#define NN_TEST_EXPECT_DEATH_OR_RESULT(func, expectedResult) \
    NNT_EXPECT_RESULT_FAILURE(expectedResult, func)
#endif

namespace {

const size_t g_BufferSize = nn::audio::AudioInBuffer::SizeGranularity;
NN_AUDIO_ALIGNAS_AUDIO_IN_BUFFER_ALIGN char g_Buffer[g_BufferSize];

bool FindUacDevice(nn::audio::AudioInInfo* pAudioInInfo) NN_NOEXCEPT
{
    nn::audio::AudioInInfo audioInInfos[nn::audio::AudioInCountMax];
    const auto Count = nn::audio::ListAudioIns(&audioInInfos[0], NN_ARRAY_SIZE(audioInInfos));

    for(int i = 0; i < Count; ++i)
    {
        if(strcmp(audioInInfos[i].name, "BuiltInHeadset") != 0)
        {
            *pAudioInInfo = audioInInfos[i];
            return true;
        }
    }

    return false;
}

}

/**
 * @brief       ListAudioIns() の正常系テストです。
 */
TEST(ListAudioInsPrivate, Success)
{
    int counts[] = { 0, 1, nn::audio::AudioInCountMax };
    for (auto count : counts)
    {
        nn::audio::AudioInInfo audioInInfos[nn::audio::AudioInCountMax];
        std::memset(audioInInfos, 0xFF, sizeof(audioInInfos));

        int listedCount = nn::audio::ListAudioIns(audioInInfos, count);
        EXPECT_GE(listedCount, 0);
        EXPECT_LE(listedCount, count);

        for (int i = 0; i < listedCount; ++i)
        {
            EXPECT_GT(strnlen(audioInInfos[i].name, nn::audio::AudioIn::NameLength), 0u);
        }

        // 外部仕様では特に規定していないがサーバ側で 0 クリアするのでその確認
        for (int i = listedCount; i < count; ++i)
        {
            EXPECT_EQ(strnlen(audioInInfos[i].name, nn::audio::AudioIn::NameLength), 0u);
        }
    }
}

/**
 * @brief       複数回 StartAudioIn() / StopAudioIn() を呼び出す正常系テストです。
 */
TEST(MultipleStartStopAudioIn, Success)
{
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));

    for(int i = 0; i < 16; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::StartAudioIn(audioIn.Get()));
        nn::audio::StopAudioIn(audioIn.Get());
    }
}

TEST(ReleasedAudioInBufferSystemEventTest, Success)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn ain;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioIn(ain.Get(), &systemEvent, parameter));
    EXPECT_NE(systemEvent.GetReadableHandle(), nn::os::InvalidNativeHandle);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(ain.Get()), nullptr);

    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));
    nn::audio::AppendAudioInBuffer(ain.Get(), &audioInBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(ain.Get()), nullptr);
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(ain.Get()));
    nn::audio::AudioInBuffer* pReleased = nullptr;

    systemEvent.Wait();

    pReleased = nn::audio::GetReleasedAudioInBuffer(ain.Get());
    EXPECT_EQ(pReleased, &audioInBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(ain.Get()), nullptr);
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief       Debugger 向け API の正常系テストです
 */
TEST(SuspendAndResumeForDebug, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioInsForDebug(appletResourceUserId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioInsForDebug(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(SuspendForDebug, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));

    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);

    {  // Open, Start, Suspend
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioInsForDebug(appletResourceUserId));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), &audioInBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioInsForDebug(appletResourceUserId));
    }
    {  // Open, Suspend, Start
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioInsForDebug(appletResourceUserId));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), &audioInBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioInsForDebug(appletResourceUserId));
    }
    {  // Suspend, Open, Start
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioInsForDebug(appletResourceUserId));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        if (nnt::audio::util::IsFirmwareSettingEnabled("suspend_for_debugger_enabled"))
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);
        }
        else
        {
            EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), &audioInBuffer);
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioInsForDebug(appletResourceUserId));
    }
    {  // AudioIn is stopped and closed without resuming
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioInsForDebug(appletResourceUserId));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioInsForDebug(appletResourceUserId));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       API call test in suspended status
 */
TEST(Suspend, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));

    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);

    {  // Open, Start, Suspend
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));

        // wait for AudioIn suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // Open, Suspend, Start
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));

        // wait for AudioIn suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // Suspend, Open, Start
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));

        // wait for AudioIn suspended without fail (TODO: this should be removed)
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        nn::audio::AudioInBuffer audioInBuffer;
        nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
        nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500));

        EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    {  // AudioIn is stopped and closed without resuming
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::StartAudioIn(audioIn.Get()));
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));
    }
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioIns(appletResourceUserId, nn::TimeSpan::FromSeconds(0)));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

/**
 * @brief       AppletManager 向け API の正常系テストです
 */
TEST(SuspendAndResume, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));

    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    nn::audio::AudioInBuffer audioInBuffer;

    int channelCount = nn::audio::GetAudioInChannelCount(audioIn.Get());
    int sampleRate = nn::audio::GetAudioInSampleRate(audioIn.Get());
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioInSampleFormat(audioIn.Get());
    const int bufferLengthInMilliseconds = 100; // StartAudioIn() -> RequestSuspendAudioIns() が完了するよりも十分に大きな時間
    const int millisecondsPerSecond = 1000;
    const int frameRate = millisecondsPerSecond / bufferLengthInMilliseconds;                             // 20fps
    const int frameSampleCount = sampleRate / frameRate;  // 50msecs (in samples)
    const size_t dataSize = frameSampleCount * channelCount * nn::audio::GetSampleByteSize(sampleFormat);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioInBuffer::SizeGranularity) + nn::audio::AudioInBuffer::SizeGranularity;

    auto sampleBuffer = malloc(bufferSize);
    NN_ABORT_UNLESS_NOT_NULL(sampleBuffer);
    nn::audio::SetAudioInBufferInfo(&audioInBuffer,
                                    nn::util::BytePtr(sampleBuffer).AlignUp(nn::audio::AudioInBuffer::AddressAlignment).Get(),
                                    bufferSize,
                                    dataSize);
    nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestSuspendAudioIns(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(500)); // bufferLengthInMilliseconds よりも十分に大きな時間

    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RequestResumeAudioIns(appletResourceUserId, nn::TimeSpan::FromMilliSeconds(1)));

    auto pReleased = nn::audio::GetReleasedAudioInBuffer(audioIn.Get());
    while( pReleased == nullptr )
    {
        pReleased = nn::audio::GetReleasedAudioInBuffer(audioIn.Get());
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
    EXPECT_EQ(pReleased, &audioInBuffer);
    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));

    if (sampleBuffer)
    {
        free(sampleBuffer);
        sampleBuffer = nullptr;
    }
}

TEST(AppendManyAudioInBufferTest, Success)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn aout;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioIn(aout.Get(), &systemEvent, parameter));
    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(aout.Get()));

    // 1. Append nn::audio::AudioInBufferCountMax -> Release nn::audio::AudioInBufferCountMax
    for (int i = 0; i < nn::audio::AudioInBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioInBuffer(aout.Get(), &audioInBuffer));
    }

    for (int releasedBufferCount = 0; releasedBufferCount < nn::audio::AudioInBufferCountMax;)
    {
        systemEvent.Wait();

        while (nn::audio::GetReleasedAudioInBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }

    // 2. Append nn::audio::AudioInBufferCountMax -> Release nn::audio::AudioInBufferCountMax / 2
    //    -> Append nn::audio::AudioInBufferCountMax / 2 -> Release nn::audio::AudioInBufferCountMax
    for (int i = 0; i < nn::audio::AudioInBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioInBuffer(aout.Get(), &audioInBuffer));
    }

    for (int releasedBufferCount = 0; releasedBufferCount < nn::audio::AudioInBufferCountMax / 2;)
    {
        systemEvent.Wait();

        while (nn::audio::GetReleasedAudioInBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }

    for (int i = 0; i < nn::audio::AudioInBufferCountMax / 2; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioInBuffer(aout.Get(), &audioInBuffer));
    }

    for (int releasedBufferCount = 0; releasedBufferCount < nn::audio::AudioInBufferCountMax;)
    {
        systemEvent.Wait();

        while (nn::audio::GetReleasedAudioInBuffer(aout.Get()) != nullptr)
        {
            ++releasedBufferCount;
        }
    }
}

void ThreadFunc(void*)
{
}

TEST(AppendAfterStartStopTest, Success)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    nn::audio::AudioIn audioIn;
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenDefaultAudioIn(&audioIn, &systemEvent, parameter));
    nn::audio::AudioInBuffer audioInBuffer;
    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, sizeof(g_Buffer), sizeof(g_Buffer));

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(&audioIn));

    nn::audio::StopAudioIn(&audioIn);

    // 1. Append nn::audio::AudioOutBufferCountMax -> Release nn::audio::AudioOutBufferCountMax
    for(int i = 0; i < nn::audio::AudioInBufferCountMax; ++i)
    {
        EXPECT_TRUE(nn::audio::AppendAudioInBuffer(&audioIn, &audioInBuffer));
    }

    nn::audio::CloseAudioIn(&audioIn);


    //Replaced thread creation with os call to make sure g_Buffer is not mapped
    #if defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_FALSE(nn::os::IsMemoryLocked(g_Buffer, sizeof(g_Buffer)));
    #endif
}

/**
 * @brief       AppletManager 向け API の正常系テストです
 */
TEST(AppletVolume, Success)
{
    auto appletResourceUserId = nn::applet::GetAppletResourceUserId();
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(appletResourceUserId));
    nn::audio::AudioInInfo audioInInfo;
    int count = nn::audio::ListAudioIns(&audioInInfo, 1);
    EXPECT_LE(count, 1);
    EXPECT_GE(count, 0);
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn audioIn;
    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(audioIn.Get(), nn::audio::common::DeviceAudioIn, parameter));

    EXPECT_EQ(nn::audio::GetReleasedAudioInBuffer(audioIn.Get()), nullptr);

    nn::audio::AudioInBuffer audioInBuffer;

    nn::audio::SetAudioInBufferInfo(&audioInBuffer, g_Buffer, g_BufferSize, sizeof(g_Buffer));
    nn::audio::AppendAudioInBuffer(audioIn.Get(), &audioInBuffer);

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::StartAudioIn(audioIn.Get()));

    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioInsProcessMasterVolume(appletResourceUserId, 0.0f, nn::TimeSpan::FromMilliSeconds(0)));
    float volume = 1.0f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::GetAudioInsProcessMasterVolume(&volume, appletResourceUserId));
    EXPECT_EQ(volume, 0.0f);

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioInsProcessMasterVolume(appletResourceUserId, 1.0f, nn::TimeSpan::FromMilliSeconds(10)));
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::GetAudioInsProcessMasterVolume(&volume, appletResourceUserId));
    EXPECT_EQ(volume, 1.0f);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(appletResourceUserId));
}

TEST(RegisterApplet, Success)
{
    auto id = nn::applet::GetAppletResourceUserId();
    float volume = 1.0f;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Actually register. (Lookup Count = 1)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::RegisterAppletResourceUserId(id)); // Only the lookup count goes up. (Lookup Count = 2)
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Only the lookup count goes down. (Lookup Count = 1)

    nn::audio::SetAudioInsProcessMasterVolume(id, 0.5f, 0);
    nn::audio::GetAudioInsProcessMasterVolume(&volume, id);

    EXPECT_TRUE(volume == 0.5f);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::UnregisterAppletResourceUserId(id)); // Actually cancel registration. (Lookup Count = 0)
}

/**
 * @brief       AppletManager 向け API の事前条件テストです
 */
TEST(SuspendAndResume, Precondition)
{
    auto id = nn::applet::GetAppletResourceUserId();

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioIns(id, nn::TimeSpan::FromMilliSeconds(0 - 1)) , "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::RequestSuspendAudioIns(id, nn::TimeSpan::FromMilliSeconds(10 + 1)) , "");
}

TEST(RegisterApplet, Failure)
{
    auto id = nn::applet::AppletResourceUserId::GetInvalidId();
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::UnregisterAppletResourceUserId(id));

    float volume = 0.0f;
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::GetAudioInsProcessMasterVolume(&volume, id));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::SetAudioInsProcessMasterVolume(id, 0.5f, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestSuspendAudioIns(id, nn::TimeSpan::FromMilliSeconds(10)));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultAppletResourceUserIdNotFound, nn::audio::RequestResumeAudioIns(id, nn::TimeSpan::FromMilliSeconds(10)));
}

#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

TEST(OpenDuplicateAudioInTest, Error)
{
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    nn::audio::AudioInInfo audioInInfos[nn::audio::AudioInCountMax];
    std::memset(audioInInfos, 0xFF, sizeof(audioInInfos));

    int listedCount = nn::audio::ListAudioIns(audioInInfos, nn::audio::AudioInCountMax);
    EXPECT_GE(listedCount, 0);
    EXPECT_LE(listedCount, nn::audio::AudioInCountMax);
    {
        ScopedAudioIn ain;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(ain.Get(), parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenDefaultAudioIn(ain.Get(), parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedAudioIn ain;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(ain.Get(), &event, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenDefaultAudioIn(ain.Get(), &event, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
    {
        ScopedAudioIn ain;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(ain.Get(), nn::audio::common::DeviceAudioIn, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenAudioIn(ain.Get(), nn::audio::common::DeviceAudioIn, parameter), nn::audio::ResultAlreadyOpen);
    }
    {
        ScopedAudioIn ain;
        nn::os::SystemEvent event;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(ain.Get(), &event, nn::audio::common::DeviceAudioIn, parameter));
        NN_TEST_EXPECT_DEATH_OR_RESULT(
            nn::audio::OpenAudioIn(ain.Get(), &event, nn::audio::common::DeviceAudioIn, parameter), nn::audio::ResultAlreadyOpen);
        nn::os::DestroySystemEvent(event.GetBase());
    }
}

#if defined(NN_BUILD_CONFIG_SPEC_NX) && (defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX))
TEST(GetAudioInDeviceGain, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioInDeviceGain(nullptr), "");
}

TEST(GetAudioInDeviceGain, Success)
{
    {
        // API returns nn::audio::AudioInDeviceGainMax when AudioIn is connecting to BuiltInHeadset for now.
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        ScopedAudioIn ain;
        nn::os::SystemEvent systemEvent;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(ain.Get(), &systemEvent, "BuiltInHeadset", parameter));
        EXPECT_EQ(nn::audio::GetAudioInDeviceGain(ain.Get()), nn::audio::AudioInDeviceGainMax);
    }

    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        nn::audio::AudioIn audioIn;
        nn::audio::AudioInInfo audioInInfo;
        nn::os::SystemEvent systemEvent;

        if(FindUacDevice(&audioInInfo))
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(&audioIn, &systemEvent, audioInInfo.name, parameter));
            const auto gain = nn::audio::GetAudioInDeviceGain(&audioIn);
            NN_LOG("USB microphones is connected. name: (%s) gain:%.2f \n", audioInInfo.name, gain);
            EXPECT_LE(gain, nn::audio::AudioInDeviceGainMax);
            EXPECT_GE(gain, nn::audio::AudioInDeviceGainMin);

            nn::audio::CloseAudioIn(&audioIn);
        }
        else
        {
            NN_LOG("USB microphones is not connected.\n");
        }
    }
}

TEST(SetAudioInDeviceGain, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInDeviceGain(nullptr, nn::audio::AudioInDeviceGainMin), "");

    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);
    ScopedAudioIn ain;
    nn::os::SystemEvent systemEvent;

    NNT_EXPECT_RESULT_SUCCESS(
        nn::audio::OpenAudioIn(ain.Get(), &systemEvent, "BuiltInHeadset", parameter));
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInDeviceGain(ain.Get(), nn::audio::AudioInDeviceGainMin - 1), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioInDeviceGain(ain.Get(), nn::audio::AudioInDeviceGainMax + 1), "");
}

TEST(SetAudioInDeviceGain, Success)
{
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        ScopedAudioIn ain;
        nn::os::SystemEvent systemEvent;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(ain.Get(), &systemEvent, "BuiltInHeadset", parameter));

        // API can not set gain parameter when AudioIn is connecting to BuiltinHeadset for now
        NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultNotSupported, nn::audio::SetAudioInDeviceGain(ain.Get(), nn::audio::AudioInDeviceGainMin));
    }

    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        nn::audio::AudioIn audioIn;
        nn::audio::AudioInInfo audioInInfo;
        nn::os::SystemEvent systemEvent;
        if(FindUacDevice(&audioInInfo))
        {
            NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenAudioIn(&audioIn, &systemEvent, audioInInfo.name, parameter));
            NN_LOG("USB microphones is connected. name: (%s)\n", audioInInfo.name);

            const auto Gain1 = nn::audio::AudioInDeviceGainMax / 2.f;
            if(nn::audio::SetAudioInDeviceGain(&audioIn, Gain1).IsSuccess())
            {
                EXPECT_EQ(Gain1, nn::audio::GetAudioInDeviceGain(&audioIn));
            }
            else
            {
                EXPECT_EQ(nn::audio::AudioInDeviceGainMax, nn::audio::GetAudioInDeviceGain(&audioIn));
            }

            nn::audio::StartAudioIn(&audioIn);

            const auto Gain2 = nn::audio::AudioInDeviceGainMax / 3.f;
            if(nn::audio::SetAudioInDeviceGain(&audioIn, Gain2).IsSuccess())
            {
                EXPECT_EQ(Gain2, nn::audio::GetAudioInDeviceGain(&audioIn));
            }
            else
            {
                EXPECT_EQ(nn::audio::AudioInDeviceGainMax, nn::audio::GetAudioInDeviceGain(&audioIn));
            }

            nn::audio::StopAudioIn(&audioIn);
            nn::audio::CloseAudioIn(&audioIn);
        }
        else
        {
            NN_LOG("USB microphones is not connected.\n");
        }
    }
}
#endif

// A test process aborts if this test case is success.
#if 0
TEST(OpenAudioIn, SystemEventLeak)
{
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioInParameter parameter;
    nn::audio::InitializeAudioInParameter(&parameter);

    for(int i = 0; i < 10000; ++i)
    {
        ScopedAudioIn ain;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenAudioIn(ain.Get(), &systemEvent, "BuiltInHeadset", parameter));
    }
}
#endif

#include <nn/audio/audio_AudioInApi.private.h>

extern "C" void nnMain()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_SPEC_NX)
    nn::audio::SetAudioInEnabled(true);
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_SPEC_NX)

    auto argc = nnt::GetHostArgc();
    auto argv = nnt::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);

    const auto result = RUN_ALL_TESTS();
    ::nnt::Exit(result);
}
