﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/audio/audio_AudioIn.h>

#include "testAudio_ScopedAudioIn.h"

/**
 * @brief       OpenDefaultAudioIn() の正常系テストです。
 */
TEST(OpenDefaultAudioIn, Success)
{
    int sampleRate;

    // With default
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(audioIn.Get(), parameter));
        sampleRate = GetAudioInSampleRate(audioIn.Get());
        EXPECT_EQ(
            nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
        nn::audio::CloseAudioIn(audioIn.Get());
    }

    // With non-default sample rate
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        parameter.sampleRate = sampleRate;
        ScopedAudioIn audioIn;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(audioIn.Get(), parameter));
        EXPECT_EQ(
            nn::audio::GetAudioInState(audioIn.Get()), nn::audio::AudioInState_Stopped);
        nn::audio::CloseAudioIn(audioIn.Get());
    }

    // TODO: 2 つ同時にオープン出来るようになったら再度有効にする
#if 0
    // 同時に 2 つオープンする
    {
        nn::audio::AudioInParameter parameter0, parameter1;
        nn::audio::InitializeAudioInParameter(&parameter0);
        nn::audio::InitializeAudioInParameter(&parameter1);
        parameter1.sampleRate = sampleRate;
        ScopedAudioIn audioIn0, audioIn1;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(audioIn0.Get(), parameter0));
        NNT_EXPECT_RESULT_SUCCESS(
            nn::audio::OpenDefaultAudioIn(audioIn1.Get(), parameter1));
    }
#endif
}

/**
 * @brief       OpenDefaultAudioIn() の事前条件違反テストです。
 */
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpenDefaultAudioInTest, PreCondition)
{
    nn::audio::AudioIn audioIn;

    // With default
    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        // pAudioIn == nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioIn(nullptr, parameter), "");
    }

    // With non-default sample rate
    {
        // pAudioIn == nullptr
        nn::audio::AudioInParameter parameter0;
        nn::audio::InitializeAudioInParameter(&parameter0);
        parameter0.sampleRate = 48000;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioIn(nullptr, parameter0), "");

        // sampleRate < 0
        nn::audio::AudioInParameter parameter1;
        nn::audio::InitializeAudioInParameter(&parameter1);
        parameter1.sampleRate = -1;
        EXPECT_DEATH_IF_SUPPORTED(
            nn::audio::OpenDefaultAudioIn(&audioIn, parameter1), "");
    }
}
#endif  // !defined(NN_SDK_BUILD_RELEASE)

/**
 * @brief       OpenDefaultAudioIn() のエラー処理テストです。
 */
TEST(OpenDefaultAudioIn, Error)
{
    const int InvalidSampleRate = 1;  // おおよそ ありえない であろうサンプルレート

    nn::audio::AudioIn audioIn;

    {
        nn::audio::AudioInParameter parameter;
        nn::audio::InitializeAudioInParameter(&parameter);
        parameter.sampleRate = InvalidSampleRate;
        NNT_EXPECT_RESULT_FAILURE(
            nn::audio::ResultInvalidSampleRate, nn::audio::OpenDefaultAudioIn(&audioIn, parameter));
    }
}
