﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <limits>

#include <nn/nn_Log.h>
#include <nn/audio.h>
#include <nnt/audioUtil/testAudio_Util.h>

namespace nn { namespace audio {
int GetActiveChannelCount() NN_NOEXCEPT;
}}


TEST(ListAudioDeviceName, Success)
{
    const int nameCount = 10;
    nn::audio::AudioDeviceName device[nameCount];
    memset(device, 0, sizeof(device));

    const char* DeviceNames[] = { "AudioStereoJackOutput", "AudioBuiltInSpeakerOutput", "AudioTvOutput", "AudioUsbDeviceOutput" };

    auto count = nn::audio::ListAudioDeviceName(device, nameCount);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    EXPECT_EQ(count, NN_ARRAY_SIZE(DeviceNames));
#endif

    if (count > 0)
    {
        for (auto i = 0; i < count; ++i)
        {
            EXPECT_STREQ(device[i].name, DeviceNames[i]);
            std::cout << i << " : " << device[i].name << std::endl;
        }
    }
    else if (count == 0)
    {
        EXPECT_STREQ(device[0].name, "");
    }
}

TEST(ListAudioDeviceName, Precondition)
{
    const int nameCount = 10;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::ListAudioDeviceName(nullptr, nameCount), "");
}

TEST(SetGetAudioDeviceOutputVolume, Success)
{
    const int nameCount = 10;
    nn::audio::AudioDeviceName device[nameCount];
    memset(device, 0, sizeof(device));

    auto count = nn::audio::ListAudioDeviceName(device, nameCount);
#ifndef NN_BUILD_CONFIG_OS_WIN32
    EXPECT_GT(count, 0);
#endif
    if (count > 0)
    {
        for (auto i = 0; i < count; ++i)
        {
            EXPECT_STRNE(device[i].name, "");
            std::cout << i << " : " << device[i].name << std::endl;
        }
    }
    else if (count == 0)
    {
        EXPECT_STREQ(device[0].name, "");
    }

    // Note:
    // Do not run test for AudioUsbDeviceOutput in this testcase because we would like to test it on TEST(SetGetAudioDeviceOutputVolume, AudioUsbDeviceOutput).
    auto volume = nn::audio::GetAudioDeviceOutputVolume(&device[0]);
    EXPECT_EQ(volume, 1.0f); // default val
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[1]);
    EXPECT_EQ(volume, 1.0f); // default val
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[2]);
    EXPECT_EQ(volume, 1.0f); // default val

    float targetVal = 0.5;
    nn::audio::SetAudioDeviceOutputVolume(&device[0], targetVal);
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[0]);

#ifndef NN_BUILD_CONFIG_OS_WIN32
    EXPECT_EQ(volume, targetVal);
#endif
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[1]);
    EXPECT_EQ(volume, 1.0f); // default val
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[2]);
    EXPECT_EQ(volume, 1.0f); // default val

#ifndef NN_BUILD_CONFIG_OS_WIN32
    targetVal = 0.4;
    nn::audio::SetAudioDeviceOutputVolume(&device[1], targetVal);
    volume = nn::audio::GetAudioDeviceOutputVolume(&device[1]);
    EXPECT_EQ(volume, targetVal);
#endif

    // boundary tests before opening audio instance
    nn::audio::SetAudioDeviceOutputVolume(&device[0], nn::audio::AudioDeviceOutputVolumeMin);
    nn::audio::SetAudioDeviceOutputVolume(&device[0], nn::audio::AudioDeviceOutputVolumeMax);

    // boundary tests after opening audio instance
    nn::audio::AudioOut audioOut;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::OpenDefaultAudioOut(&audioOut, parameter));
    nn::audio::SetAudioDeviceOutputVolume(&device[0], nn::audio::AudioDeviceOutputVolumeMin);
    nn::audio::SetAudioDeviceOutputVolume(&device[0], nn::audio::AudioDeviceOutputVolumeMax);
    nn::audio::CloseAudioOut(&audioOut);

    // reset volume for next test loop
    nn::audio::SetAudioDeviceOutputVolume(&device[0], 1.0f);
    nn::audio::SetAudioDeviceOutputVolume(&device[1], 1.0f);
    nn::audio::SetAudioDeviceOutputVolume(&device[2], 1.0f);
}

#if defined(NN_BUILD_CONFIG_SPEC_NX) && defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(SetGetAudioDeviceOutputVolume, InvalidDevice)
{
    const int nameCount = 10;
    nn::audio::AudioDeviceName device[nameCount];
    memset(device, 0, sizeof(device));
    auto count = nn::audio::ListAudioDeviceName(device, nameCount);

    for(int i = 0; i < count; ++i)
    {
        EXPECT_EQ(nn::audio::GetAudioDeviceOutputVolume(&device[i]), 1.0f);
    }

    nn::audio::AudioDeviceName invalidDevice;
    strcpy(invalidDevice.name, "InvalidDevice");
    nn::audio::SetAudioDeviceOutputVolume(&invalidDevice, 0.f);
    nn::audio::GetAudioDeviceOutputVolume(&invalidDevice);

    for(int i = 0; i < count; ++i)
    {
        EXPECT_EQ(nn::audio::GetAudioDeviceOutputVolume(&device[i]), 1.0f);
    }
}

TEST(SetGetAudioDeviceOutputVolume, UsbOutputDevice)
{
    static bool isAlreadyTested;

    if(isAlreadyTested)
    {
        NN_LOG("This testcase can not run more than twice.\n");
        return;
    }

    const int nameCount = 10;
    nn::audio::AudioDeviceName device[nameCount];
    memset(device, 0, sizeof(device));
    auto count = nn::audio::ListAudioDeviceName(device, nameCount);
    ASSERT_GE(count, 3);
    EXPECT_STREQ(device[0].name, "AudioStereoJackOutput");
    EXPECT_STREQ(device[3].name, "AudioUsbDeviceOutput");

    // Step1: App does not call SetAudioDeviceOutputVolume for AudioUsbDeviceOutput.
    const auto TargetValForStep1 = 0.5f;
    nn::audio::SetAudioDeviceOutputVolume(&device[0], TargetValForStep1);
    {
        const auto VolumeForStereoJack = nn::audio::GetAudioDeviceOutputVolume(&device[0]);
        EXPECT_EQ(VolumeForStereoJack, TargetValForStep1);
        const auto VolumeForUsbDevice = nn::audio::GetAudioDeviceOutputVolume(&device[3]);
        EXPECT_EQ(VolumeForUsbDevice, TargetValForStep1);
    }

    // Step2: App calls SetAudioDeviceOutputVolume for AudioUsbDeviceOutput.
    const float TargetValForStep2 = 0.6f;
    nn::audio::SetAudioDeviceOutputVolume(&device[3], TargetValForStep2);
    {
        const auto VolumeForUsbDevice = nn::audio::GetAudioDeviceOutputVolume(&device[3]);
        EXPECT_EQ(VolumeForUsbDevice, TargetValForStep2);
        const auto VolumeForStereoJack = nn::audio::GetAudioDeviceOutputVolume(&device[0]);
        EXPECT_EQ(VolumeForStereoJack, TargetValForStep1);
    }

    isAlreadyTested = true;
}
#endif

TEST(SetGetAudioDeviceOutputVolume, Precondition)
{
    nnt::audio::util::ScopedAudioRenderer sar;
    sar.Start();

    nn::audio::AudioDeviceName device;

    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioDeviceOutputVolume(nullptr, 0.f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioDeviceOutputVolume(&device, nn::audio::AudioDeviceOutputVolumeMin - 0.1f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::SetAudioDeviceOutputVolume(&device, nn::audio::AudioDeviceOutputVolumeMax + 0.1f), "");
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetAudioDeviceOutputVolume(nullptr), "");
}

TEST(SetAudioDeviceMapping, Success)
{
    const int nameCount = 10;
    nn::audio::AudioDeviceName device[nameCount];
    memset(device, 0, sizeof(device));
    auto count = nn::audio::ListAudioDeviceName(device, nameCount);

    if (count > 0)
    {
        for (auto i = 0; i < count; ++i)
        {
            EXPECT_STRNE(device[i].name, "");
            std::cout << i << " : " << device[i].name << std::endl;
        }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NNT_EXPECT_RESULT_SUCCESS(nn::audio::SetAudioDeviceMapping("MainAudioOut", &device[0]));
#else  // defined(NN_BUILD_CONFIG_OS_WIN32)
        NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultNotSupported, nn::audio::SetAudioDeviceMapping("MainAudioOut", &device[0]));
#endif  // defined(NN_BUILD_CONFIG_OS_WIN32)
    }
    else if (count == 0)
    {
        EXPECT_STREQ(device[0].name, "");
    }
}

#if defined(NN_BUILD_CONFIG_SPEC_NX) && defined(NN_BUILD_CONFIG_OS_WIN)
TEST(GetActiveAudioDeviceName, Success)
{
    nn::audio::AudioDeviceName blankName;
    memset(blankName.name, 0, sizeof(blankName));

    nn::audio::AudioDeviceName deviceName;
    memset(deviceName.name, 0, sizeof(deviceName));
    nn::audio::GetActiveAudioDeviceName(&deviceName);
    EXPECT_STREQ(deviceName.name, blankName.name);
}

TEST(AcquireAudioDeviceSwitchNotification, Success)
{
    NN_PRAGMA_PUSH_WARNINGS
    NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS
    nn::os::SystemEvent systemEvent;
    nn::audio::AcquireAudioDeviceSwitchNotification(&systemEvent);
    EXPECT_DEATH_IF_SUPPORTED(systemEvent.Wait(), "");
    NN_PRAGMA_POP_WARNINGS
}

TEST(AcquireAudioDeviceNotificationForOutput, Success)
{
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultNotSupported, nn::audio::AcquireAudioDeviceNotificationForOutput(&systemEvent));
    EXPECT_DEATH_IF_SUPPORTED(systemEvent.Wait(), "");
}

TEST(AcquireAudioDeviceNotificationForInput, Success)
{
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultNotSupported, nn::audio::AcquireAudioDeviceNotificationForInput(&systemEvent));
    EXPECT_DEATH_IF_SUPPORTED(systemEvent.Wait(), "");
}
#elif defined(NN_BUILD_CONFIG_SPEC_NX) && defined(NN_BUILD_CONFIG_OS_HORIZON)
TEST(GetActiveAudioDeviceName, Success)
{
    nn::audio::AudioDeviceName deviceName;
    nn::audio::GetActiveAudioDeviceName(&deviceName);

    EXPECT_STRNE(deviceName.name, "");
    EXPECT_TRUE(std::strcmp(deviceName.name, "AudioTvOutput") == 0 ||
                std::strcmp(deviceName.name, "AudioStereoJackOutput") == 0 ||
                std::strcmp(deviceName.name, "AudioBuiltInSpeakerOutput") == 0 ||
                std::strcmp(deviceName.name, "AudioUsbDeviceOutput") == 0);
}

TEST(GetActiveAudioDeviceName, Precondition)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::GetActiveAudioDeviceName(nullptr), "");
}

TEST(AcquireAudioDeviceSwitchNotification, Success)
{
    NN_PRAGMA_PUSH_WARNINGS
    NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS
    nn::os::SystemEvent systemEvent;
    nn::audio::AcquireAudioDeviceSwitchNotification(&systemEvent);
    NN_PRAGMA_POP_WARNINGS
}

TEST(AcquireAudioDeviceSwitchNotification, Precondition)
{
    NN_PRAGMA_PUSH_WARNINGS
    NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS
    nn::os::SystemEvent systemEvent;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireAudioDeviceSwitchNotification(nullptr), "");
    nn::audio::AcquireAudioDeviceSwitchNotification(&systemEvent);
//    nn::audio::AcquireAudioDeviceSwitchNotification(&systemEvent);
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireAudioDeviceSwitchNotification(&systemEvent), "");
    NN_PRAGMA_POP_WARNINGS
}

TEST(AcquireAudioDeviceNotificationForOutput, Success)
{
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AcquireAudioDeviceNotificationForOutput(&systemEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOperationFailed, nn::audio::AcquireAudioDeviceNotificationForOutput(&systemEvent));
}

TEST(AcquireAudioDeviceNotificationForOutput, Precondition)
{
    nn::os::SystemEvent systemEvent;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireAudioDeviceNotificationForOutput(nullptr), "");
}

TEST(AcquireAudioDeviceNotificationForInput, Success)
{
    nn::os::SystemEvent systemEvent;
    NNT_EXPECT_RESULT_SUCCESS(nn::audio::AcquireAudioDeviceNotificationForInput(&systemEvent));
    NNT_EXPECT_RESULT_FAILURE(nn::audio::ResultOperationFailed, nn::audio::AcquireAudioDeviceNotificationForInput(&systemEvent));
}

TEST(AcquireAudioDeviceNotificationForInput, Precondition)
{
    nn::os::SystemEvent systemEvent;
    EXPECT_DEATH_IF_SUPPORTED(nn::audio::AcquireAudioDeviceNotificationForInput(nullptr), "");
}

TEST(GetActiveChannelCount, Success)
{
    auto ch = nn::audio::GetActiveChannelCount();
    std::cout << "ch count:" << ch << std::endl;
    EXPECT_TRUE(ch == 1 || ch == 2 || ch == 6);
}

TEST(GetActiveAudioDeviceChannelCountForOutput, Success)
{
    auto ch = nn::audio::GetActiveAudioDeviceChannelCountForOutput();
    std::cout << "ch count:" << ch << std::endl;
    EXPECT_TRUE(ch == 1 || ch == 2 || ch == 6);
}

#endif
