﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>
#include <limits>
#include <algorithm>
#include <string>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/audio.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>

#include <nn/settings/settings_DebugPad.h>

namespace {

    const int RenderRate = 32000;
    const int RenderCount = (RenderRate / 200);

    const int SeCount = 2;

    const char Title[] = "AudioDelayUpdateTest";

    const char* g_SeFileNames[SeCount] =
    {
        "asset:/AudioCommon/SampleSe1.adpcm",
        "asset:/AudioCommon/SampleSe3.adpcm",
    };

    const int parameterSetCount = 3;

    struct DelayTimeParameters
    {
        nn::TimeSpan delayTime;
        std::string timeString;
    };

    DelayTimeParameters delayTimeParameters[parameterSetCount] =
    {
        {
            nn::TimeSpan::FromMilliSeconds(10),
            "10ms",
        },
        {
            nn::TimeSpan::FromMilliSeconds(400),
            "400ms",
        },
        {
            nn::TimeSpan::FromMilliSeconds(1000),
            "1000ms",
        }
    };

    struct DelayFloatParameters
    {
        float delayFloat;
        std::string floatString;
    };

    DelayFloatParameters delayFloatParameters[parameterSetCount] =
    {
        {
            0.3f,
            "0.3f",
        },
        {
            0.7f,
            "0.7f",
        },
        {
            1.0f,
            "1.0f",
        }
    };

    NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[12 * 1024 * 1024];

    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_WaveBufferAllocator;
    nn::mem::StandardAllocator g_EffectBufferAllocator;


    enum DelayParameter { TIME, IN_GAIN, FEEDBACK_GAIN, DRY_GAIN, CHANNEL_SPREAD, LOW_PASS };

    struct AudioEffectStatus
    {
        int delayTimeIndex;
        std::string delayTime;
        int delayInGainIndex;
        std::string delayInGain;
        int delayFeedbackGainIndex;
        std::string delayFeedbackGain;
        int delayDryGainIndex;
        std::string delayDryGain;
        int delayChannelSpreadIndex;
        std::string delayChannelSpread;
        int delayLowPassIndex;
        std::string delayLowPassAmount;
        bool enabled;
        std::string enabledString;
    };

    char* g_MountRomCacheBuffer = NULL;

}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA = nn::hid::KeyboardKey::A::Index;
    map.buttonB = nn::hid::KeyboardKey::B::Index;
    map.buttonX = nn::hid::KeyboardKey::X::Index;
    map.buttonY = nn::hid::KeyboardKey::Y::Index;
    map.buttonL = nn::hid::KeyboardKey::L::Index;
    map.buttonR = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NN_LOG("---------------------------------------------- \n");
    NN_LOG("%s Sample\n", Title);
    NN_LOG("---------------------------------------------- \n");
    NN_LOG("[A]             StartSound     (SampleSe1)     \n");
    NN_LOG("[B]             StartSound     (SampleSe3)     \n");
    NN_LOG("[X]             Update Delay Time              \n");
    NN_LOG("[Y]             Update Delay In Gain           \n");
    NN_LOG("[L]             Update Delay Feedback Gain     \n");
    NN_LOG("[R]             Update Delay Dry Gain          \n");
    NN_LOG("[Left]          Update Delay Channel Spread    \n");
    NN_LOG("[Right]         Update Delay Low Pass Amount   \n");
    NN_LOG("[Down]          Disable Delay                  \n");
    NN_LOG("[Up]            Enable Delay                   \n");
    NN_LOG("[Start/Space]   Exit                           \n");
    NN_LOG("---------------------------------------------- \n");
}

int UpdateDelayParameter(nn::audio::DelayType* delay, int updateMode, std::string& message, DelayParameter delayParameter)
{
    DelayTimeParameters* delayTimeParameterSet = nullptr;
    DelayFloatParameters* delayFloatParameterSet = nullptr;

    switch (delayParameter)
    {
    case TIME:
        delayTimeParameterSet = &delayTimeParameters[updateMode];
        if (delayTimeParameterSet != nullptr) {
            message = delayTimeParameterSet->timeString;
            nn::audio::SetDelayTime(delay, delayTimeParameterSet->delayTime);
        }
        break;
    case IN_GAIN:
        delayFloatParameterSet = &delayFloatParameters[updateMode];
        if (delayFloatParameterSet != nullptr) {
            message = delayFloatParameterSet->floatString;
            nn::audio::SetDelayInGain(delay, delayFloatParameterSet->delayFloat);
        }
        break;
    case FEEDBACK_GAIN:
        delayFloatParameterSet = &delayFloatParameters[updateMode];
        if (delayFloatParameterSet != nullptr) {
            message = delayFloatParameterSet->floatString;
            nn::audio::SetDelayFeedbackGain(delay, delayFloatParameterSet->delayFloat);
        }
        break;
    case DRY_GAIN:
        delayFloatParameterSet = &delayFloatParameters[updateMode];
        if (delayFloatParameterSet != nullptr) {
            message = delayFloatParameterSet->floatString;
            nn::audio::SetDelayDryGain(delay, delayFloatParameterSet->delayFloat);
        }
        break;
    case CHANNEL_SPREAD:
        delayFloatParameterSet = &delayFloatParameters[updateMode];
        if (delayFloatParameterSet != nullptr) {
            message = delayFloatParameterSet->floatString;
            nn::audio::SetDelayChannelSpread(delay, delayFloatParameterSet->delayFloat);
        }
        break;
    case LOW_PASS:
        delayFloatParameterSet = &delayFloatParameters[updateMode];
        if (delayFloatParameterSet != nullptr) {
            message = delayFloatParameterSet->floatString;
            nn::audio::SetDelayLowPassAmount(delay, delayFloatParameterSet->delayFloat);
        }
        break;
    default:
        break;
    }

    return (updateMode + 1) % 3;
}

bool EnableDelay(nn::audio::DelayType* delay, bool enabled, std::string& message)
{
    if (!enabled) {
        nn::audio::SetDelayEnabled(delay, true);
        message = "enabled";
    }
    return true;
}

bool DisableDelay(nn::audio::DelayType* delay, bool enabled, std::string& message)
{
    if (enabled) {
        nn::audio::SetDelayEnabled(delay, false);
        message = "disabled";
    }
    return false;
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    InitializeFileSystem();
    InitializeHidDevices();

    AudioEffectStatus sampleStatus = { 0 };

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 4;
    parameter.voiceCount = 12;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 1;
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2] = { 0, 1 };

    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
        );

    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    nn::Result result = nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 4);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
    rt = RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    // Delay エフェクトを用意します。
    nn::audio::DelayType delay;
    nn::TimeSpan delayTimeMax = nn::TimeSpan::FromMilliSeconds(1100);

    size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, channelCount);
    void* delayBuffer = g_EffectBufferAllocator.Allocate(delaySize);
    NN_ABORT_UNLESS_NOT_NULL(delayBuffer);

    // Delay を FianlMix に追加します。
    result = nn::audio::AddDelay(&config, &delay, delayBuffer, delaySize, &finalMix, delayTimeMax, channelCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // Delay エフェクトへの入力・出力バッファを設定します。
    int delayStartIndex = 0;

    nn::audio::SetDelayInputOutput(&delay, mainBus, mainBus, sizeof(mainBus));
    DelayTimeParameters* delayTimeParameterSet = &delayTimeParameters[delayStartIndex];
    DelayFloatParameters* delayFloatParameterSet = &delayFloatParameters[delayStartIndex];
    nn::audio::SetDelayTime(&delay, delayTimeParameterSet->delayTime);
    nn::audio::SetDelayChannelSpread(&delay, delayFloatParameterSet->delayFloat);
    nn::audio::SetDelayDryGain(&delay, delayFloatParameterSet->delayFloat);
    nn::audio::SetDelayFeedbackGain(&delay, delayFloatParameterSet->delayFloat);
    nn::audio::SetDelayInGain(&delay, delayFloatParameterSet->delayFloat);
    nn::audio::SetDelayLowPassAmount(&delay, delayFloatParameterSet->delayFloat);
    nn::audio::SetDelayEnabled(&delay, true);

    sampleStatus.delayTimeIndex = delayStartIndex;
    sampleStatus.delayTime = delayTimeParameterSet->timeString;
    sampleStatus.delayChannelSpreadIndex = delayStartIndex;
    sampleStatus.delayChannelSpread = delayFloatParameterSet->floatString;
    sampleStatus.delayDryGainIndex = delayStartIndex;
    sampleStatus.delayDryGain = delayFloatParameterSet->floatString;
    sampleStatus.delayFeedbackGainIndex = delayStartIndex;
    sampleStatus.delayFeedbackGain = delayFloatParameterSet->floatString;
    sampleStatus.delayInGainIndex = delayStartIndex;
    sampleStatus.delayInGain = delayFloatParameterSet->floatString;
    sampleStatus.delayLowPassIndex = delayStartIndex;
    sampleStatus.delayLowPassAmount = delayFloatParameterSet->floatString;
    sampleStatus.enabled = true;
    sampleStatus.enabledString = "enabled";

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // SE を読み込みます。
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo) * SeCount, nn::audio::BufferAlignSize));
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        std::size_t dataSeSize = ReadAdpcmFile(&header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i].sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i].parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i].sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i].loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, mainBus[0]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, mainBus[1]);
    }

    PrintUsage();

    int64_t prevPadSamplingNumber = 0;
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];

    bool running = true;

    while (running)
    {
        systemEvent.Wait();

        // DebugPad の入力を取得します。
        // debugPad の状態を表す構造体を用意します。
        nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

        // 直前から現在までの Pad 入力の更新回数を取得します。
        int64_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
        if (padSamplingCount >= nn::hid::DebugPadStateCountMax)
        {
            padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
        }
        nn::hid::DebugPadButtonSet padButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

        // 現在までの Pad 入力の更新回数を取得します。
        prevPadSamplingNumber = debugPadState[0].samplingNumber;

        // 各 Effect のパラメータを更新します。
        {
            // Delay のパラメータを更新します。
            if (padButtonDown.Test<::nn::hid::DebugPadButton::X>())
            {
                sampleStatus.delayTimeIndex = UpdateDelayParameter(&delay, sampleStatus.delayTimeIndex, sampleStatus.delayTime, TIME);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Y>())
            {
                sampleStatus.delayInGainIndex = UpdateDelayParameter(&delay, sampleStatus.delayInGainIndex, sampleStatus.delayInGain, IN_GAIN);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::L>())
            {
                sampleStatus.delayFeedbackGainIndex = UpdateDelayParameter(&delay, sampleStatus.delayFeedbackGainIndex, sampleStatus.delayFeedbackGain, FEEDBACK_GAIN);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::R>())
            {
                sampleStatus.delayDryGainIndex = UpdateDelayParameter(&delay, sampleStatus.delayDryGainIndex, sampleStatus.delayDryGain, DRY_GAIN);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Left>())
            {
                sampleStatus.delayChannelSpreadIndex = UpdateDelayParameter(&delay, sampleStatus.delayChannelSpreadIndex, sampleStatus.delayChannelSpread, CHANNEL_SPREAD);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Right>())
            {
                sampleStatus.delayLowPassIndex = UpdateDelayParameter(&delay, sampleStatus.delayLowPassIndex, sampleStatus.delayLowPassAmount, LOW_PASS);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Up>())
            {
                sampleStatus.enabled = EnableDelay(&delay, sampleStatus.enabled, sampleStatus.enabledString);
            }
            if (padButtonDown.Test<::nn::hid::DebugPadButton::Down>())
            {
                sampleStatus.enabled = DisableDelay(&delay, sampleStatus.enabled, sampleStatus.enabledString);
            }


            if (padButtonDown.IsAnyOn())
            {
                if (!sampleStatus.enabled) {
                    NN_LOG("Delay disabled \n");
                }
                else {
                    NN_LOG("Delay Time: %s -- In Gain: %s -- FeedBack Gain: %s -- Dry Gain: %s -- Channel Spread: %s -- Low Pass: %s \n",
                        sampleStatus.delayTime.c_str(),
                        sampleStatus.delayInGain.c_str(),
                        sampleStatus.delayFeedbackGain.c_str(),
                        sampleStatus.delayDryGain.c_str(),
                        sampleStatus.delayChannelSpread.c_str(),
                        sampleStatus.delayLowPassAmount.c_str());
                }
            }
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if (debugPadState[0].buttons.Test(i) && !debugPadState[padSamplingCount].buttons.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]) != nullptr)
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // 各エフェクトに施したパラメータ変更をオーディオレンダラに反映します。
        // 反映するまでは、パラメータ変更は有効になりません。
        // nn::audio::RequestUpdateAudioRenderer() の呼び出し中は、各エフェクトへのアクセスは避けてください。
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        //  サンプルを終了する
        if (padButtonDown.Test<::nn::hid::DebugPadButton::Start>())
        {
            running = false;
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
    }
    if (header)
    {
        g_WaveBufferAllocator.Free(header);
        header = nullptr;
    }
    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
    if (delayBuffer)
    {
        g_EffectBufferAllocator.Free(delayBuffer);
        delayBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
