﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/fs.h>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/util/util_FormatString.h>
#include <nn/audio/audio_Profiler.h>

namespace {

//!< メモリヒープのサイズ
NN_ALIGNAS(4096)    char g_ProfilerBuffer[32 * 1024 * 1024];

const uint64_t      DefaultSamplingFrequencyUsec = 300;
const uint32_t      DefaultMaxSamples            = 1024 * 512;
const uint32_t      DefaultRuntimeSeconds        = 15;
const uint32_t      MaxSamples                   = (sizeof(g_ProfilerBuffer) - sizeof(nn::audio::DspProfilerHeader)) / sizeof(nn::audio::DspProfilerSample);



struct ProfilerArgs
{
    const char*   outPath;
    uint64_t      samplingFrequency;
    uint32_t      maxSamples;
    uint32_t      maxRunTime;
};

void* AllocateForFileSystem(size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void DeallocateForFileSystem(void* addr, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    free(addr);
}

void PrintHelp()
{
    NN_LOG("Profiler Args:\n");
    NN_LOG("OutputPath {absolute path}\n");
    NN_LOG("\tSets the path for the profiler log. Required.\n");
    NN_LOG("Frequency {frequency in usec}\n");
    NN_LOG("\tSets the profiler sampling frequency in microseconds. Recommended range 100-1000, default is %lld\n", DefaultSamplingFrequencyUsec);
    NN_LOG("MaxSamples {number of samples}\n");
    NN_LOG("\tSets the maximum number of samples. Recommended range 100000-2000000, default is %d, max is %d\n", DefaultMaxSamples, MaxSamples);
    NN_LOG("MaxDuration {max runtime seconds}\n");
    NN_LOG("\tSets the maximimum runtime, profiler will stop after this time has elpased. Default is %d.\n", DefaultRuntimeSeconds);
    NN_LOG("\tIf set to 0, profiler will run until the buffer is full.");
}

int ParseArg(ProfilerArgs* outArg, char** const argv, int argc, int index)
{
    const char* arg = argv[index];

    if( strcmp(arg, "OutputPath") == 0 )
    {
        if( index == argc )
        {
            return -1;
        }
        outArg->outPath = argv[index + 1];
        return index + 1;
    }
    else if( strcmp(arg, "Frequency") == 0 )
    {
        if( index == argc )
        {
            return -1;
        }
        outArg->samplingFrequency = atol(argv[index + 1]);
        return index + 1;
    }
    else if( strcmp(arg, "MaxSamples") == 0 )
    {
        if( index == argc )
        {
            return -1;
        }
        outArg->maxSamples = atoi(argv[index + 1]);
        if( outArg->maxSamples > MaxSamples )
        {
            NN_LOG("MaxSamples argument (%d) is too large. Clamping to %d\n", outArg->maxSamples, MaxSamples);
            outArg->maxSamples = MaxSamples;
        }
        return index + 1;
    }
    else if( strcmp(arg, "MaxDuration") == 0 )
    {
        if( index == argc)
        {
            return -1;
        }
        outArg->maxRunTime = atoi(argv[index + 1]);
        return index + 1;
    }
    NN_LOG("Skipping %s\n", argv[index]);
    return -1;
}

//!< プログラム引数を返します。
bool ParseArgs(ProfilerArgs* outArg) NN_NOEXCEPT
{
    const int argc = ::nn::os::GetHostArgc();

    char** const argv = ::nn::os::GetHostArgv();
    for ( int i = 1; i < argc; ++i )
    {
        int result = ParseArg(outArg, argv, argc, i);
        if( result == -1 )
        {
            NN_LOG("\"%s\" Unxepected argument\n", argv[i]);
            PrintHelp();
            return false;
        }
        i = result;
    }
    if( outArg->outPath == nullptr )
    {
        NN_LOG("OutputPath argument is required\n");
        return false;
    }
    return true;
}

bool WriteProfileBuffer(nn::audio::DspProfilerHeader* header, ProfilerArgs* args, nn::audio::DspProfilerParameter* params)
{
    const int BlockSize = 4096;
    uint64_t offset = 0;
    char path[nn::fs::EntryNameLengthMax] = {0};
    static char tmp [BlockSize] = {0};

    nn::fs::FileHandle file;
    nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);
    nn::fs::MountHostRoot();
    params->numSamples = header->curSample;
    size_t bufferSize = nn::audio::GetBufferSizeForDspProfiler(params);

    nn::util::SNPrintf(path, sizeof(path), "%s", args->outPath);

    auto result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    auto notFound = nn::fs::ResultPathNotFound();
    if( result.GetDescription() == notFound.GetDescription() )
    {
        nn::fs::CreateFile(path, bufferSize);
        result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    }
    if( !result.IsSuccess() )
    {
        NN_LOG("Cannot write file at %s (result: mod=%d, desc=%d)\n", path, result.GetModule(), result.GetDescription());
        return false;
    }

    nn::fs::WriteOption writeFlags;
    writeFlags.flags = nn::fs::WriteOptionFlag_Flush;

    int i = 0;
    for(; i < bufferSize / BlockSize; ++i)
    {
        memcpy(tmp, &g_ProfilerBuffer[i * BlockSize], BlockSize);
        nn::fs::WriteFile(file, offset, &tmp[0], BlockSize, writeFlags);
        offset += BlockSize;
    }
    int remainder = bufferSize - (i * BlockSize);
    if( remainder != 0 )
    {
        memcpy(tmp, &g_ProfilerBuffer[i * BlockSize], remainder);
        nn::fs::WriteFile(file, offset, &tmp[0], remainder, writeFlags);
    }
    nn::fs::CloseFile(file);
    nn::fs::UnmountHostRoot();
    return true;
}

} // namespace

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );
    NN_ASSERT(result.IsSuccess(), "Cannot set memory heap size.");

    uintptr_t address;
    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess(), "Cannot allocate memory block." );

    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

void WaitForProfilerEnd(ProfilerArgs* args)
{
    uint64_t estimatedDurationUsec = args->samplingFrequency * args->maxSamples;
    uint64_t maxDurationUsec = args->maxRunTime * 1000 * 1000;
    nn::TimeSpan sleepDuration = nn::TimeSpan::FromMicroSeconds(estimatedDurationUsec);
    if( estimatedDurationUsec > maxDurationUsec )
    {
        sleepDuration = nn::TimeSpan::FromMicroSeconds(maxDurationUsec);
    }

    nn::os::SleepThread(sleepDuration);

    if( maxDurationUsec != 0 )
    {
        return;
    }

    nn::audio::DspProfilerHeader* header = reinterpret_cast<nn::audio::DspProfilerHeader*>(g_ProfilerBuffer);

    while( header->curSample != header->maxSamples )
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
}

void PrintArgs(ProfilerArgs* args)
{
    NN_LOG("Starting profiler.\n");
    NN_LOG("\tSampling frequency = %lld usec\n", args->samplingFrequency);
    NN_LOG("\tMaxDuration = %d %s\n", args->maxRunTime,
            args->maxRunTime == 0 ? "(Run until buffer is full)" : "sec");
    NN_LOG("\tMaxSamples = %d\n", args->maxSamples);
    NN_LOG("\tOutput location: %s\n", args->outPath);
}

extern "C" void nnMain()
{
    ProfilerArgs args;
    args.outPath           = nullptr;
    args.samplingFrequency = DefaultSamplingFrequencyUsec;
    args.maxSamples        = DefaultMaxSamples;
    args.maxRunTime        = DefaultRuntimeSeconds;
    bool result = ParseArgs(&args);

    if( !result )
    {
        return;
    }

    nn::os::SetMemoryAttribute(reinterpret_cast<uintptr_t>(g_ProfilerBuffer), sizeof(g_ProfilerBuffer), nn::os::MemoryAttribute_Uncached);
    nn::audio::DspProfilerParameter profilerParam;
    profilerParam.sampleFrequencyUsec = args.samplingFrequency;
    profilerParam.numSamples = args.maxSamples;

    PrintArgs(&args);

    profilerParam.buffer = g_ProfilerBuffer;

    nn::audio::StartDspProfiler(&profilerParam);

    WaitForProfilerEnd(&args);

    nn::audio::StopDspProfiler();

    nn::audio::DspProfilerHeader* header = reinterpret_cast<nn::audio::DspProfilerHeader*>(g_ProfilerBuffer);
    bool written = WriteProfileBuffer(header, &args, &profilerParam);
    if( written )
    {
        NN_LOG("Profile written to %s successfully\n", args.outPath);
    }
    return;
}
