﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>

namespace {

    const int MemoryHeapSize = 32 * 1024 * 1024;

    nnt::atk::util::FsCommonSetup   g_FsSetup;

    static char                 g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;

    class SubMixSetup : protected nnt::atk::util::AtkCommonSetup
    {
    public:
        struct SubMixParam
        {
            int srcBusCount;
            int srcChannelCount;
            int dstBusCount;
            int dstChannelCount;
        };

        void Initialize(nn::mem::StandardAllocator& allocator, const SubMixParam* pParam, int subMixCount) NN_NOEXCEPT
        {
            int totalChannelCount = 0;
            for(int i = 0; i < subMixCount; i++)
            {
                totalChannelCount += pParam[i].srcBusCount * pParam[i].srcChannelCount;
            }

            nn::atk::SoundSystem::SoundSystemParam param;
            param.enableCustomSubMix = true;
            param.subMixCount = subMixCount;
            param.subMixTotalChannelCount = totalChannelCount;

            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam( param );
            AtkCommonSetup::Initialize( initializeParam, allocator );

            m_SubMixCount = subMixCount;
            m_pSubMix = reinterpret_cast<nn::atk::SubMix*>( allocator.Allocate( sizeof(nn::atk::SubMix) * subMixCount ) );
            m_pSubMixBuffer = reinterpret_cast<void**>( allocator.Allocate( sizeof(void*) * subMixCount ) );
            for(int i = 0 ; i < subMixCount; i++)
            {
                new(&m_pSubMix[i]) nn::atk::SubMix();
                m_pSubMixBuffer[i] = nullptr;
            }
        }
        bool InitializeSubMix(nn::mem::StandardAllocator& allocator, const SubMixParam* pParam, int paramCount)
        {
            NN_UNUSED( paramCount );
            NN_SDK_ASSERT_LESS_EQUAL( paramCount, m_SubMixCount );

            for(int i = 0; i < m_SubMixCount; i++)
            {
                const size_t size = nn::atk::SubMix::GetRequiredMemorySize( pParam[i].srcBusCount, pParam[i].srcChannelCount, pParam[i].dstBusCount, pParam[i].dstChannelCount );
                void* buffer = allocator.Allocate( size );

                const bool result = m_pSubMix[i].Initialize( pParam[i].srcBusCount, pParam[i].srcChannelCount, pParam[i].dstBusCount, pParam[i].dstChannelCount, buffer, size );

                if( result == false )
                {
                    return false;
                }
            }
            return true;
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            FinalizeSoundArchivePlayer( allocator );
            FinalizeSoundDataManager( allocator );
            FinalizeSoundArchive( allocator );
            FinalizeSoundHeap( allocator );

            //  SubMix を削除します
            const auto WaitTime  = nn::TimeSpan::FromMilliSeconds( 5 );
            const auto TimeOut   = nn::TimeSpan::FromMilliSeconds( 5 * 1000 );
            nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 );

            for(int i = 0; i < m_SubMixCount;)
            {
                if( m_pSubMix[i].IsFinalizable() )
                {
                    m_pSubMix[i].Finalize();
                    if( m_pSubMixBuffer[i] != nullptr )
                    {
                        allocator.Free( m_pSubMixBuffer[i] );
                    }
                    i++;
                }
                else
                {
                    nn::os::SleepThread( WaitTime );
                    elapsed += WaitTime;
                    EXPECT_TRUE( elapsed < TimeOut );
                }
            }

            allocator.Free( m_pSubMixBuffer );
            allocator.Free( m_pSubMix );

            FinalizeSoundSystem( allocator );
        }

        void StartSound(nn::atk::SoundHandle& handle, uint32_t soundId, int subMixIndex) NN_NOEXCEPT
        {
            nn::atk::SoundArchivePlayer::StartInfo info;

            info.enableFlag |= nn::atk::SoundArchivePlayer::StartInfo::EnableFlagBit_OutputReceiver;
            info.pOutputReceiver = &GetSubMix( subMixIndex );

            AtkCommonSetup::GetSoundArchivePlayer().StartSound( &handle, soundId, &info );
        }
        nn::atk::SubMix& GetSubMix(int index) NN_NOEXCEPT
        {
            NN_SDK_ASSERT_RANGE( index, 0, m_SubMixCount );
            return m_pSubMix[index];
        }
        nn::atk::FinalMix& GetFinalMix() NN_NOEXCEPT
        {
            return nn::atk::SoundSystem::GetFinalMix();
        }

    private:
        //  clang の警告対策
        using AtkCommonSetup::Initialize;

        nn::atk::SubMix* m_pSubMix;
        void** m_pSubMixBuffer;
        int m_SubMixCount;
    };

    SubMixSetup  g_SubMixSetup;

    //  DriverCommand の枯渇を防ぐために Flush を行います
    void FlushCommandDriverCommand()
    {
        nn::atk::detail::DriverCommand::GetInstance().FlushCommand( false );
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    //  DEATH テストでのみ使用しているため、RELEASE では定義しません

    //  Effect が IsRemovable を返すまで待ちます
    void WaitForEffectRemovable(nn::atk::EffectBase& effect)
    {
        const auto WaitTime  = nn::TimeSpan::FromMilliSeconds( 5 );
        const auto TimeOut   = nn::TimeSpan::FromMilliSeconds( 1 * 1000 );
        nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 );

        while( !effect.IsRemovable() )
        {
            nn::os::SleepThread( WaitTime );

            elapsed += WaitTime;
            if( elapsed >= TimeOut )
            {
                break;
            }
        }
        EXPECT_LT( elapsed, TimeOut );
    }
#endif
}

#if !defined(NN_SDK_BUILD_RELEASE)

TEST(SubMix, SoundSystemInitializeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    //  subMixCount, subMixTotalChannelCount に負数を設定
    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableCustomSubMix = true;
    param.subMixCount = -1;
    param.subMixTotalChannelCount = 0;

    nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
    initializeParam.SetSoundSystemParam( param );

    nnt::atk::util::AtkCommonSetup atkSetup;
    EXPECT_DEATH_IF_SUPPORTED( atkSetup.Initialize( initializeParam, g_Allocator ), "" );

    param.subMixCount = 0;
    param.subMixTotalChannelCount = -1;
    initializeParam.SetSoundSystemParam( param );
    EXPECT_DEATH_IF_SUPPORTED( atkSetup.Initialize( initializeParam, g_Allocator ), "" );

    g_Allocator.Finalize();
}

TEST(SubMix, SubMixInitializeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    enum SubMixIndex
    {
        SubMixIndex_Main,
        SubMixIndex_DestinateToMain,
        SubMixIndex_Count,
    };
    SubMixSetup::SubMixParam subMixParam[SubMixIndex_Count] =
    {
        //  srcBus, srcChannel, dstBus, dstChannel
        { 4, 6, 1, 4 }, //  値は適当です
        { 1, 1, 1, 1 }, //  値は適当です
    };
    g_SubMixSetup.Initialize( g_Allocator, subMixParam, SubMixIndex_Count );

    const int BusOverCount = (nn::atk::AuxBus_Count + 1) + 1; // DefaultBusCount + 1
    const int ChannelOverCount = nn::audio::MixBufferCountMax + 1;
    nn::atk::SubMix& subMix = g_SubMixSetup.GetSubMix( SubMixIndex_Main );
    const SubMixSetup::SubMixParam& param = subMixParam[SubMixIndex_Main];

    // 未初期化、初期化済の OutputReceiver を用意(初期化済の方は FinalMix を使用)
    nn::atk::SubMix uninitializedOutputReceiver;
    nn::atk::FinalMix& initializedOutputReceiver = nn::atk::SoundSystem::GetFinalMix();

    //  GetRequiredMemorySize に不適切な値を設定
    {
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( 0, param.srcChannelCount, param.dstBusCount, param.dstChannelCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, 0, param.dstBusCount, param.dstChannelCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, 0, param.dstChannelCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( BusOverCount, 1, param.dstBusCount, param.dstChannelCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( 1, ChannelOverCount, param.dstBusCount, param.dstChannelCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, BusOverCount, 1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, 1, ChannelOverCount ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, &uninitializedOutputReceiver), "");
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( 0, param.srcChannelCount, &initializedOutputReceiver ), "");
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( param.srcBusCount, 0, &initializedOutputReceiver ), "");
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( BusOverCount, 1, &initializedOutputReceiver ), "");
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetRequiredMemorySize( 1, ChannelOverCount, &initializedOutputReceiver ), "");
    }

    //  Initialize に不適切な値を設定
    {
        const size_t size = subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount );
        void* buffer = g_Allocator.Allocate( size );

        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( 0, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, 0, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, 0, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, 0, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, nullptr, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, buffer, size - 1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount + 1, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount + 1, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount + 1, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount + 1, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( BusOverCount, 1, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( 1, ChannelOverCount, param.dstBusCount, param.dstChannelCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, BusOverCount, 1, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, 1, ChannelOverCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, reinterpret_cast<char*>(buffer) + 1, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, reinterpret_cast<char*>(buffer) + (nn::atk::SubMix::SubMixAlignment - 1), size ), "" );

        g_Allocator.Free( buffer );
    }

    //  Initialize に不適切な値を設定 (OutputReceiver を指定する関数)
    {
        const size_t size = subMix.GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, &initializedOutputReceiver );
        void* buffer = g_Allocator.Allocate( size );

        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( 0, param.srcChannelCount, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, 0, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, nullptr, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, &uninitializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, &initializedOutputReceiver, nullptr, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, &initializedOutputReceiver, buffer, size - 1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount + 1, param.srcChannelCount, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount + 1, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( BusOverCount, 1, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( 1, ChannelOverCount, &initializedOutputReceiver, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, &initializedOutputReceiver, reinterpret_cast<char*>(buffer) + 1, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.Initialize( param.srcBusCount, param.srcChannelCount, &initializedOutputReceiver, reinterpret_cast<char*>(buffer) + (nn::atk::SubMix::SubMixAlignment - 1), size ), "" );
        g_Allocator.Free( buffer );
    }

    //  Initialize 前に関数を呼ぶ
    {
        nn::atk::FinalMix& finalMix = g_SubMixSetup.GetFinalMix();

        EXPECT_DEATH_IF_SUPPORTED( subMix.IsBusMuted( 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.IsChannelMuted( 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.IsSubMixMuted(), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetBusMute( 0, false ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetSend( 0, 0, 0.0f ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetBusVolume( 0, 0.0f, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetChannelMute( 0, false ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetDestination( &finalMix ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetSubMixMute( false ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.SetSubMixVolume( 0.0f, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetBusCount(), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetChannelCount(), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetSend( 0, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetBusVolume( 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetChannelVolume( 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.GetSubMixVolume(), "" );
    }

    //  Effect の着脱関連
    {
        const size_t subMixSize = nn::atk::SubMix::GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount );
        void* subMixBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, subMixSize );
        const int AppendBusIndex = 0;
        const bool result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixSize );
        EXPECT_TRUE( result );

        nn::atk::EffectReverb effect;
        const size_t requiredSize = effect.GetRequiredMemSize();
        const size_t size = nn::util::align_up( requiredSize, nn::audio::MemoryPoolType::SizeGranularity );
        void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, size + 1, nn::audio::MemoryPoolType::AddressAlignment );
        nn::audio::MemoryPoolType memoryPool;
        nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, buffer, size );

        //  nn::atk::EffectAux と区別するために nullptr を reinterpret_cast します
        nn::atk::EffectReverb* nullptrReverb = nullptr;
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( nullptrReverb, AppendBusIndex, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( &effect, -1, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( &effect, param.srcBusCount, buffer, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( &effect, AppendBusIndex, nullptr, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( &effect, AppendBusIndex, reinterpret_cast<char*>( buffer ) + 1, size ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.AppendEffect( &effect, AppendBusIndex, buffer, requiredSize - 1 ), "" );

        EXPECT_DEATH_IF_SUPPORTED( subMix.RemoveEffect( nullptrReverb, AppendBusIndex ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.RemoveEffect( &effect, -1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.RemoveEffect( &effect, param.srcBusCount ), "" );

        EXPECT_DEATH_IF_SUPPORTED( subMix.ClearEffect( -1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( subMix.ClearEffect( param.srcBusCount ), "" );

        subMix.Finalize();

        nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
        g_Allocator.Free( buffer );
        g_Allocator.Free( subMixBuffer );
    }

    //  参照カウント関連
    {
        const size_t subMixBufferSize = nn::atk::SubMix::GetRequiredMemorySize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount );
        void* subMixBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, subMixBufferSize );
        bool result;

        //  Effect の着脱に関する参照カウント
        {
            const int BusIndex = 0;
            nn::atk::EffectReverb effect;
            const size_t effectBufferSize = nn::util::align_up( effect.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity );
            void* effectBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, effectBufferSize, nn::audio::MemoryPoolType::AddressAlignment );
            nn::audio::MemoryPoolType memoryPool;
            nn::atk::SoundSystem::AttachMemoryPool( &memoryPool, effectBuffer, effectBufferSize );

            result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixBufferSize );
            EXPECT_TRUE( result );
            result = subMix.AppendEffect( &effect, BusIndex, effectBuffer, effectBufferSize );
            EXPECT_TRUE( result );
            nnt::atk::util::WaitForProcessCommand();
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );

            //  RequestUpdateAudioRenderer 直後は Effect の IsRemovable が false を返すことがあるため、
            //  IsRemovable が true を返すまで待ちます
            WaitForEffectRemovable( effect );

            subMix.RemoveEffect( &effect, BusIndex );
            subMix.Finalize();

            result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixBufferSize );
            EXPECT_TRUE( result );
            result = subMix.AppendEffect( &effect, BusIndex, effectBuffer, effectBufferSize );
            EXPECT_TRUE( result );
            nnt::atk::util::WaitForProcessCommand();
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );

            WaitForEffectRemovable( effect );
            subMix.ClearEffect( BusIndex );
            subMix.Finalize();

            nn::atk::SoundSystem::DetachMemoryPool( &memoryPool );
            g_Allocator.Free( effectBuffer );
        }

        //  サウンドの再生に関する参照カウント
        {
            nn::atk::SoundHandle handle;
            result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixBufferSize );
            EXPECT_TRUE( result );
            g_SubMixSetup.StartSound( handle, STRM_MARIOKART, SubMixIndex_Main );
            nnt::atk::util::WaitForProcessCommand();
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );
            handle.Stop( 0 );
            nnt::atk::util::WaitForProcessCommand();
            subMix.Finalize();
        }

        //  SubMix の出力先設定に関する参照カウント
        {
            nn::atk::SubMix& destinateToMainSubMix = g_SubMixSetup.GetSubMix( SubMixIndex_DestinateToMain );
            const auto& destinateToMainSubMixParam = subMixParam[SubMixIndex_DestinateToMain];
            const size_t destinateToMainSubMixBufferSize = nn::atk::SubMix::GetRequiredMemorySize( destinateToMainSubMixParam.srcBusCount, destinateToMainSubMixParam.srcChannelCount, destinateToMainSubMixParam.dstBusCount, destinateToMainSubMixParam.dstChannelCount );
            void* destinateToMainSubMixBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, destinateToMainSubMixBufferSize, nn::audio::MemoryPoolType::AddressAlignment );

            result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixBufferSize );
            EXPECT_TRUE( result );
            result = destinateToMainSubMix.Initialize( destinateToMainSubMixParam.srcBusCount, destinateToMainSubMixParam.srcChannelCount, destinateToMainSubMixParam.dstBusCount, destinateToMainSubMixParam.dstChannelCount, destinateToMainSubMixBuffer, destinateToMainSubMixBufferSize );
            EXPECT_TRUE( result );

            destinateToMainSubMix.SetDestination( &subMix );
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );
            nnt::atk::util::WaitForProcessCommand();
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );
            destinateToMainSubMix.Finalize();
            subMix.Finalize();

            g_Allocator.Free( destinateToMainSubMixBuffer );
        }

        //  DriverCommand に関する参照カウント
        {
            nn::atk::FinalMix& finalMix = g_SubMixSetup.GetFinalMix();
            result = subMix.Initialize( param.srcBusCount, param.srcChannelCount, param.dstBusCount, param.dstChannelCount, subMixBuffer, subMixBufferSize );
            EXPECT_TRUE( result );
            subMix.SetDestination( &finalMix );
            EXPECT_DEATH_IF_SUPPORTED( subMix.Finalize(), "" );
            nnt::atk::util::WaitForProcessCommand();
            subMix.Finalize();
        }

        g_Allocator.Free( subMixBuffer );
    }

    g_SubMixSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} //NOLINT(impl/function_size)

#endif

TEST(SubMix, InitializeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    const int SubMixCount = 1;
    SubMixSetup::SubMixParam param =
    {
        //  srcBus, srcChannel, dstBus, dstChannel
        2, 2, 1, 1 //  値は適当です
    };
    bool result;

    g_SubMixSetup.Initialize( g_Allocator, &param, SubMixCount );

    //  ミックスバッファが足りずに初期化に失敗
    param.srcBusCount++;
    result = g_SubMixSetup.InitializeSubMix( g_Allocator, &param, SubMixCount );
    EXPECT_FALSE( result );
    param.srcBusCount--;

    param.srcChannelCount++;
    result = g_SubMixSetup.InitializeSubMix( g_Allocator, &param, SubMixCount );
    EXPECT_FALSE( result );
    param.srcChannelCount--;

    //  初期化に成功
    param.srcBusCount--;
    param.srcChannelCount--;
    result = g_SubMixSetup.InitializeSubMix( g_Allocator, &param, SubMixCount );
    EXPECT_TRUE( result );

    //  ミックスバッファは足りているが、サブミックスインスタンスが足りないために失敗
    nn::atk::SubMix additionalSubMix;
    const size_t bufferSize = additionalSubMix.GetRequiredMemorySize( 1, 1, 1, 1 );
    void* buffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, bufferSize );
    result = additionalSubMix.Initialize( 1, 1, 1, 1, buffer, bufferSize );
    EXPECT_FALSE( result );

    g_Allocator.Free( buffer );
    g_SubMixSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SubMix, ParameterTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    enum SubMixIndex
    {
        SubMixIndex_Main,
        SubMixIndex_Destination,
        SubMixIndex_Count
    };
    SubMixSetup::SubMixParam param[SubMixIndex_Count] =
    {
        //  srcBus, srcChannel, dstBus, dstChannel
        { 4, 6, 4, 6 }, //  値は適当です
        { 4, 6, 4, 6 }  //  値は適当です
    };
    bool result;

    g_SubMixSetup.Initialize( g_Allocator, param, SubMixIndex_Count );
    result = g_SubMixSetup.InitializeSubMix( g_Allocator, param, SubMixIndex_Count );
    EXPECT_TRUE( result );

    nn::atk::SubMix& subMix = g_SubMixSetup.GetSubMix( SubMixIndex_Main );
    const int busCount = subMix.GetBusCount();
    const int channelCount = subMix.GetChannelCount();
    subMix.SetDestination( &g_SubMixSetup.GetSubMix( SubMixIndex_Destination ) );

    //  MoveVolume 系の確認。MoveVolume であるため、何度かサウンドスレッドが回ることで値が変化します
    const float MoveVolumeTargetValue[] = { 0.0f, 1.0f, 0.5f, 1.0f, 0.0f, 1.0f };
    const int   MoveVolumeFadeFrame[] = { 0, 0, 1, 1, 6, 6 };
    for(int i = 0; i < sizeof(MoveVolumeTargetValue) / sizeof(MoveVolumeTargetValue[0]); i++)
    {
        if( i == 0 )
        {
            //  i == 0 のときはボリュームを設定します

            //  i == 0 のときは即時反映させる必要があります
            NN_SDK_ASSERT_EQUAL( MoveVolumeFadeFrame[i], 0 );

            const float volume = MoveVolumeTargetValue[i];
            const int fadeFrame = MoveVolumeFadeFrame[i];
            for(int bus = 0; bus < busCount; bus++)
            {
                subMix.SetBusVolume( bus, volume, fadeFrame );
                EXPECT_EQ( subMix.GetBusVolume( bus ), volume );
            }
            for(int channel = 0; channel < channelCount; channel++)
            {
                subMix.SetChannelVolume( channel, volume, fadeFrame );
                EXPECT_EQ( subMix.GetChannelVolume( channel ), volume );
            }
            {
                subMix.SetSubMixVolume( volume, fadeFrame );
                EXPECT_EQ( subMix.GetSubMixVolume(), volume );
            }
        }
        else
        {
            //  i > 0 のときは新しいボリュームを設定して動くさまを確認します
            const auto WaitTime = nn::TimeSpan::FromMilliSeconds( 5 );
            const auto TimeOut = nn::TimeSpan::FromMilliSeconds( 5000 );
            const float targetVolume = MoveVolumeTargetValue[i];
            const bool isVolumeUp = MoveVolumeTargetValue[i] >= MoveVolumeTargetValue[i - 1];
            const int fadeFrame = MoveVolumeFadeFrame[i];

            for(int bus = 0; bus < busCount; bus++)
            {
                float prvVolume = subMix.GetBusVolume( bus );
                subMix.SetBusVolume( bus, targetVolume, fadeFrame );

                //  fadeFrame かけてボリュームが変化します
                nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 );
                for(; elapsed < TimeOut; elapsed += WaitTime)
                {
                    const float curVolume = subMix.GetBusVolume( bus );
                    if( isVolumeUp )
                    {
                        EXPECT_GE( curVolume, prvVolume );
                    }
                    else
                    {
                        EXPECT_LE( curVolume, prvVolume );
                    }

                    if( curVolume == targetVolume )
                    {
                        //  MoveValue は設定した値をそのまま返すため、float の比較でも == で大丈夫です
                        break;
                    }

                    prvVolume = curVolume;
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                    //  vcmd 版では FrameProcess を呼ぶ必要があります
                    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
                    nn::os::SleepThread( WaitTime );
                }
                EXPECT_LT( elapsed, TimeOut );
            }

            for(int channel = 0; channel < channelCount; channel++)
            {
                float prvVolume = subMix.GetChannelVolume( channel );
                subMix.SetChannelVolume( channel, targetVolume, fadeFrame );

                //  fadeFrame かけてボリュームが変化します
                nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 );
                for(; elapsed < TimeOut; elapsed += WaitTime)
                {
                    const float curVolume = subMix.GetChannelVolume( channel );
                    if( isVolumeUp )
                    {
                        EXPECT_GE( curVolume, prvVolume );
                    }
                    else
                    {
                        EXPECT_LE( curVolume, prvVolume );
                    }

                    if( curVolume == targetVolume )
                    {
                        //  MoveValue は設定した値をそのまま返すため、float の比較でも == で大丈夫です
                        break;
                    }

                    prvVolume = curVolume;
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                    //  vcmd 版では FrameProcess を呼ぶ必要があります
                    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
                    nn::os::SleepThread( WaitTime );
                }
                EXPECT_LT( elapsed, TimeOut );
            }

            {
                float prvVolume = subMix.GetSubMixVolume();
                subMix.SetSubMixVolume( targetVolume, fadeFrame );

                //  fadeFrame かけてボリュームが変化します
                nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 );
                for(; elapsed < TimeOut; elapsed += WaitTime)
                {
                    const float curVolume = subMix.GetSubMixVolume();
                    if( isVolumeUp )
                    {
                        EXPECT_GE( curVolume, prvVolume );
                    }
                    else
                    {
                        EXPECT_LE( curVolume, prvVolume );
                    }

                    if( curVolume == targetVolume )
                    {
                        //  MoveValue は設定した値をそのまま返すため、float の比較でも == で大丈夫です
                        break;
                    }

                    prvVolume = curVolume;
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                    //  vcmd 版では FrameProcess を呼ぶ必要があります
                    nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
                    nn::os::SleepThread( WaitTime );
                }
                EXPECT_LT( elapsed, TimeOut );
            }
        }
    }

    //  Send 系の確認。SendVolume は設定後はすぐに Get に反映されます
    const float SendList[] = { 0.0f, 1.0f, 0.5f };
    const int dstBusCount = g_SubMixSetup.GetSubMix( SubMixIndex_Destination ).GetBusCount();
    for(const float send : SendList)
    {
        for(int srcBus = 0; srcBus < busCount; srcBus++)
        {
            for(int dstBus = 0; dstBus < dstBusCount; dstBus++)
            {
                subMix.SetSend( srcBus, dstBus, send );
                EXPECT_EQ( subMix.GetSend( srcBus, dstBus ), send );
            }
            //  SetSendVolume で発行された DriverCommand が Flush されないために
            //  DriverCommand が枯渇してしまっているため Flush を呼び出します
            FlushCommandDriverCommand();
        }
    }

    //  Mute 系の確認。Mute は設定後はすぐに Get に反映されます
    const bool MuteFlag[] = { false, true, false };
    for(const auto muteFlag : MuteFlag)
    {
        for(int bus = 0; bus < busCount; bus++)
        {
            subMix.SetBusMute( bus, muteFlag );
            EXPECT_EQ( subMix.IsBusMuted( bus ), muteFlag );
        }
        for(int channel = 0; channel < channelCount; channel++)
        {
            subMix.SetChannelMute( channel, muteFlag );
            EXPECT_EQ( subMix.IsChannelMuted( channel ), muteFlag );
        }
        subMix.SetSubMixMute( muteFlag );
        EXPECT_EQ( subMix.IsSubMixMuted(), muteFlag );
    }
    //  SetSubMixMute で発行された DriverCommand が Flush されないために
    //  DriverCommand が枯渇してしまっているため Flush を呼び出します
    FlushCommandDriverCommand();

    g_SubMixSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} //NOLINT(impl/function_size)
