﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>

namespace
{

bool IsLoopDetected(nn::atk::SoundHandle& handle, int64_t prevSamplePosition)
{
    nn::atk::StreamSoundHandle streamHandle(&handle);
    if (streamHandle.IsAttachedSound())
    {
        return streamHandle.GetPlayLoopCount() != 0;
    }
    nn::atk::WaveSoundHandle waveHandle(&handle);
    if (waveHandle.IsAttachedSound())
    {
        return waveHandle.GetPlaySamplePosition() > 0 && waveHandle.GetPlaySamplePosition() < prevSamplePosition;
    }
    return false;
}

int64_t GetPlaySamplePosition(nn::atk::SoundHandle& handle)
{
    nn::atk::StreamSoundHandle streamHandle(&handle);
    if (streamHandle.IsAttachedSound())
    {
        return streamHandle.GetPlaySamplePosition();
    }
    nn::atk::WaveSoundHandle waveHandle(&handle);
    if (waveHandle.IsAttachedSound())
    {
        return waveHandle.GetPlaySamplePosition();
    }

    return -1;
}

class StartInfoTester : public ::testing::Test
{
protected:
    virtual void SetUp() NN_OVERRIDE
    {
        m_pHeapMemory = malloc( MemoryHeapSize );
        m_Allocator.Initialize(m_pHeapMemory, MemoryHeapSize);
        m_FsSetup.Initialize();
        m_AtkSetup.Initialize(m_Allocator);
    }
    virtual void TearDown() NN_OVERRIDE
    {
        m_AtkSetup.Finalize(m_Allocator);
        m_FsSetup.Finalize();
        m_Allocator.Finalize();
        free(m_pHeapMemory);
    }

public:
    nn::atk::SoundArchivePlayer& GetSoundArchivePlayer()
    {
        return m_AtkSetup.GetSoundArchivePlayer();
    }

    void LoadData(uint32_t soundId)
    {
        m_AtkSetup.GetSoundDataManager().LoadData(soundId, &m_AtkSetup.GetSoundHeap());
    }

private:
    static const int MemoryHeapSize = 32 * 1024 * 1024;
    nnt::atk::util::FsCommonSetup   m_FsSetup;
    void* m_pHeapMemory;
    nn::mem::StandardAllocator  m_Allocator;
    nnt::atk::util::AtkCommonSetup m_AtkSetup;
};

}

TEST_F(StartInfoTester, LoopEnabledTest)
{
    enum TestConditionIndex
    {
        LoopEnabledFlagBit,
        LoopEnabled,
        SoundId,
        Expected
    };
    enum class TestState
    {
        Play,
        LoopDetected,
        Stop
    };
    const std::tuple<bool, bool, uint32_t, TestState> TestConditions[] =
    {
        {false, false, STRM_SHORTSWEEP_15MS, TestState::Stop},
        {true, true, STRM_SHORTSWEEP_15MS, TestState::LoopDetected},
        {true, false, STRM_SHORTSWEEP_15MS, TestState::Stop},
        {false, false, STRM_SHORTSWEEP_LOOP_15MS, TestState::LoopDetected},
        {true, true, STRM_SHORTSWEEP_LOOP_15MS, TestState::LoopDetected},
        {true, false, STRM_SHORTSWEEP_LOOP_15MS, TestState::Stop},
        {false, false, SE_SHORTSWEEP_15MS, TestState::Stop},
        {true, true, SE_SHORTSWEEP_15MS, TestState::LoopDetected},
        {true, false, SE_SHORTSWEEP_15MS, TestState::Stop},
        {false, false, SE_SHORTSWEEP_LOOP_15MS, TestState::LoopDetected},
        {true, true, SE_SHORTSWEEP_LOOP_15MS, TestState::LoopDetected},
        {true, false, SE_SHORTSWEEP_LOOP_15MS, TestState::Stop},
    };

    LoadData(SE_SHORTSWEEP_15MS);
    LoadData(SE_SHORTSWEEP_LOOP_15MS);
    int testIndex = 0;

    for(const auto& condition : TestConditions)
    {
        nn::atk::SoundStartable::StartInfo info;
        info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_LoopInfo;
        info.loopInfo.enableParameterFlag |= std::get<TestConditionIndex::LoopEnabledFlagBit>(condition) && nn::atk::SoundStartable::StartInfo::LoopInfo::EnableParameterFlagBit_LoopEnabled;
        info.loopInfo.isLoopEnabled = std::get<TestConditionIndex::LoopEnabled>(condition);

        auto& soundArchivePlayer = GetSoundArchivePlayer();
        nn::atk::SoundHandle handle;
        auto result = soundArchivePlayer.StartSound(&handle, std::get<TestConditionIndex::SoundId>(condition), &info);
        ASSERT_TRUE(result.IsSuccess());

        auto state = TestState::Play;
        int64_t prevSamplePosition = 0;
        while (state == TestState::Play)
        {
            if (!handle.IsAttachedSound())
            {
                state = TestState::Stop;
                break;
            }
            if (IsLoopDetected(handle, prevSamplePosition))
            {
                state = TestState::LoopDetected;
                break;
            }
            prevSamplePosition = GetPlaySamplePosition(handle);
            // アセットのループ間隔と SoundArchivePlayer の更新間隔が近いと、
            // 長時間テストが行われてしまうので、 SoundArchivePlayer の更新間隔をアセットのループ間隔と異なるようにする
            nnt::atk::util::UpdateAndWait(soundArchivePlayer, nn::TimeSpan::FromMilliSeconds(5));
        }

        handle.Stop(0);
        nnt::atk::util::UpdateAndWait(soundArchivePlayer);

        if (std::get<TestConditionIndex::Expected>(condition) != state)
        {
            // Show Failure Condition
            NN_LOG("Test %d failed. EnableFlagBit: %d\t LoopEnabled: %d\t SoundId: %x\t Expect: %d\n"
                , testIndex
                , std::get<TestConditionIndex::LoopEnabledFlagBit>(condition)
                , std::get<TestConditionIndex::LoopEnabled>(condition)
                , std::get<TestConditionIndex::SoundId>(condition)
                , std::get<TestConditionIndex::Expected>(condition));
        }
        EXPECT_EQ(std::get<TestConditionIndex::Expected>(condition), state);
        testIndex++;
    }
}
