﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup   g_AtkSetup;

static char                 g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

}

// TODO: 再生中のサウンド数などに関するテストの追加

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundPlayer, InvalidParamTest)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundPlayer& player = soundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

    EXPECT_DEATH_IF_SUPPORTED(player.SetEffectSend(static_cast<nn::atk::AuxBus>(-1), 1.0f), ".*" );
    EXPECT_DEATH_IF_SUPPORTED(player.SetEffectSend(nn::atk::AuxBus_Count, 1.0f), ".*" );

    EXPECT_DEATH_IF_SUPPORTED(player.SetOutputVolume(static_cast<nn::atk::OutputDevice>(-1), 0.0f), ".*");
    EXPECT_DEATH_IF_SUPPORTED(player.SetOutputVolume(nn::atk::OutputDevice_Count, 0.0f), ".*");

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundPlayer, SetParamTest)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundPlayer& player = soundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

    player.SetVolume(0.0f);
    EXPECT_EQ(0.0f, player.GetVolume());
    player.SetVolume(0.5f);
    EXPECT_EQ(0.5f, player.GetVolume());

    player.SetLowPassFilterFrequency(0.0f);
    EXPECT_EQ(0.0f, player.GetLowPassFilterFrequency());
    player.SetLowPassFilterFrequency(-1.0f);
    EXPECT_EQ(-1.0f, player.GetLowPassFilterFrequency());

    player.SetBiquadFilter(nn::atk::BiquadFilterType_Min, 1.0f);
    EXPECT_EQ(nn::atk::BiquadFilterType_Min, player.GetBiquadFilterType());
    EXPECT_EQ(1.0f, player.GetBiquadFilterValue());
    player.SetBiquadFilter(nn::atk::BiquadFilterType_Max, 0.0f);
    EXPECT_EQ(nn::atk::BiquadFilterType_Max, player.GetBiquadFilterType());
    EXPECT_EQ(0.0f, player.GetBiquadFilterValue());

    player.SetDefaultOutputLine(0u);
    EXPECT_EQ(0u, player.GetDefaultOutputLine());
    player.SetDefaultOutputLine(nn::atk::OutputLine_Main);
    EXPECT_EQ(nn::atk::OutputLine_Main, player.GetDefaultOutputLine());

    player.SetMainSend(-1.0f);
    EXPECT_EQ(-1.0f, player.GetMainSend());
    player.SetMainSend(0.0f);
    EXPECT_EQ(0.0f, player.GetMainSend());

    for(int i = 0; i < nn::atk::AuxBus_Count; ++i)
    {
        player.SetEffectSend(static_cast<nn::atk::AuxBus>(i), 1.0f);
        EXPECT_EQ(1.0f, player.GetEffectSend(static_cast<nn::atk::AuxBus>(i)));
        player.SetEffectSend(static_cast<nn::atk::AuxBus>(i), 0.0f);
        EXPECT_EQ(0.0f, player.GetEffectSend(static_cast<nn::atk::AuxBus>(i)));
    }

    for(int i = 0; i < nn::atk::OutputDevice_Count; ++i)
    {
        player.SetOutputVolume(static_cast<nn::atk::OutputDevice>(i), 0.0f);
        EXPECT_EQ(0.0f, player.GetOutputVolume(static_cast<nn::atk::OutputDevice>(i)));
        player.SetOutputVolume(static_cast<nn::atk::OutputDevice>(i), 1.0f);
        EXPECT_EQ(1.0f, player.GetOutputVolume(static_cast<nn::atk::OutputDevice>(i)));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST(SoundPlayer, InvalidSendTest)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    bool CustomSubMixCondition[2] = {false, true};
    for(bool isCustomSubMixEnabled : CustomSubMixCondition)
    {
        if(isCustomSubMixEnabled)
        {
            nn::atk::SoundSystem::SoundSystemParam param;
            param.enableCustomSubMix = true;
            param.busCountMax = nn::atk::OutputReceiver::BusCountMax / 2;
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam(param);
            g_AtkSetup.Initialize(initializeParam, g_Allocator);
        }
        else
        {
            g_AtkSetup.Initialize(g_Allocator);
        }

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundPlayer& player = soundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

        const int BusCount = isCustomSubMixEnabled ? nn::atk::OutputReceiver::BusCountMax / 2 : nn::atk::AuxBus_Count + 1;

        EXPECT_DEATH_IF_SUPPORTED(player.SetSend(-1, 1.0f), ".*");
        EXPECT_DEATH_IF_SUPPORTED(player.SetSend(BusCount, 0.0f), ".*");

        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST(SoundPlayer, SetSendTest)
{
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    bool CustomSubMixCondition[2] = {false, true};
    for(bool isCustomSubMixEnabled : CustomSubMixCondition)
    {
        if(isCustomSubMixEnabled)
        {
            nn::atk::SoundSystem::SoundSystemParam param;
            param.enableCustomSubMix = true;
            param.busCountMax = nn::atk::OutputReceiver::BusCountMax;
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            initializeParam.SetSoundSystemParam(param);
            g_AtkSetup.Initialize(initializeParam, g_Allocator);
        }
        else
        {
            g_AtkSetup.Initialize(g_Allocator);
        }

        nn::atk::SoundHandle soundHandle;
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundPlayer& player = soundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

        const int BusCount = isCustomSubMixEnabled ? nn::atk::OutputReceiver::BusCountMax : nn::atk::AuxBus_Count + 1;

        for(int busIndex = 0; busIndex < BusCount; ++busIndex)
        {
            player.SetSend(busIndex, 1.0f);
            EXPECT_EQ(1.0f, player.GetSend(busIndex));
            player.SetSend(busIndex, 0.0f);
            EXPECT_EQ(0.0f, player.GetSend(busIndex));
        }

        g_AtkSetup.Finalize(g_Allocator);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
