﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>

#include <nn/mem.h>

namespace {

    const int MemoryHeapSize = 32 * 1024 * 1024;

    class SoundArchivePlayerSetup : public nnt::atk::util::AtkCommonSetup
    {
    public:
        // テスト対象である SoundArchivePlayer の初期化をクラス外で行う
        void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::atk::util::AtkCommonSetup::InitializeParam param;
            Initialize(param, allocator);
        }

        void Initialize(nnt::atk::util::AtkCommonSetup::InitializeParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            InitializeSoundSystem(param.GetSoundSystemParam(), allocator);
            InitializeSoundHeap(param.GetSoundHeapSize(), allocator);
            InitializeSoundArchive(allocator);
            InitializeSoundDataManager(allocator);
        }

        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            FinalizeSoundDataManager(allocator);
            FinalizeSoundArchive(allocator);
            FinalizeSoundHeap(allocator);
            FinalizeSoundSystem(allocator);
        }
    };

    struct AddonSoundArchiveSet
    {
        nn::atk::AddonSoundArchive addonSoundArchive;
        void* pMemoryForAddonSoundArchive;
        nn::atk::SoundDataManager addonSoundDataManager;
        void* pMemoryForAddonSoundDataManager;
    };

    nnt::atk::util::FsCommonSetup g_FsSetup;
    SoundArchivePlayerSetup g_AtkSetup;

    std::size_t g_MemSizeForSoundArchivePlayer;
    std::size_t g_MemSizeForStreamBuffer;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    static char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;
}

namespace
{
    void AllocateSoundArchivePlayerBuffer()
    {
        // SoundArchivePlayer の初期化のための設定
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        g_MemSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize(&soundArchive);
        g_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, g_MemSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
        g_MemSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize(&soundArchive);
        g_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, g_MemSizeForStreamBuffer, nn::audio::BufferAlignSize);
    }

    void FreeSoundArchivePlayerBuffer()
    {
        g_Allocator.Free(g_pMemoryForStreamBuffer);
        g_Allocator.Free(g_pMemoryForSoundArchivePlayer);
    }

    void InitializeAddonSoundArchiveSet(AddonSoundArchiveSet* pOutValues, int count)
    {
        char commonSoundArchivePath[128];
        NN_ABORT_UNLESS(nnt::atk::util::GetCommonSoundArchivePath(
            commonSoundArchivePath,
            sizeof(commonSoundArchivePath),
            nnt::atk::util::FsCommonSetup::RomMountName),
            "Cannot get sound archive path.");

        // メモリ上にアーカイブをロード
        for (auto i = 0; i < count; ++i)
        {
            nn::fs::FileHandle fileHandle;
            nn::fs::OpenFile(&fileHandle, commonSoundArchivePath, nn::fs::OpenMode_Read);
            int64_t fileSize = 0;
            nn::Result result = nn::fs::GetFileSize(&fileSize, fileHandle);
            NN_ABORT_UNLESS(result.IsSuccess());

            pOutValues[i].pMemoryForAddonSoundArchive = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, static_cast<size_t>(fileSize));
            size_t readSize = 0;
            result = nn::fs::ReadFile(&readSize, fileHandle, 0, pOutValues[i].pMemoryForAddonSoundArchive, static_cast<size_t>(fileSize));
            NN_ABORT_UNLESS(result.IsSuccess());
            nn::fs::CloseFile(fileHandle);

        }

        // 追加サウンドアーカイブと追加サウンドアーカイブ用の SoundDataManager の初期化
        for (auto i = 0; i < count; ++i)
        {
            pOutValues[i].addonSoundArchive.Initialize(pOutValues[i].pMemoryForAddonSoundArchive);
            pOutValues[i].addonSoundArchive.SetExternalFileRoot("content:/");

            size_t memSizeForAddonSoundDataManager = pOutValues[i].addonSoundDataManager.GetRequiredMemSize(&pOutValues[i].addonSoundArchive);
            pOutValues[i].pMemoryForAddonSoundDataManager =
                nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForAddonSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize);
            bool isSuccess = pOutValues[i].addonSoundDataManager.Initialize(
                &pOutValues[i].addonSoundArchive,
                pOutValues[i].pMemoryForAddonSoundDataManager,
                memSizeForAddonSoundDataManager);
            NN_ABORT_UNLESS(isSuccess, "cannot initialize SoundDataManager");

        }
    }

    void FinalizeAddonSoundArchiveSet(AddonSoundArchiveSet* pOutValues, int count)
    {
        // 追加サウンドアーカイブと追加サウンドアーカイブ用の SoundDataManager の終了処理
        for (auto i = 0; i < count; ++i)
        {
            pOutValues[i].addonSoundDataManager.Finalize();
            pOutValues[i].addonSoundArchive.Finalize();

            g_Allocator.Free(pOutValues[i].pMemoryForAddonSoundDataManager);
            g_Allocator.Free(pOutValues[i].pMemoryForAddonSoundArchive);
        }
    }

    void LoadData()
    {
        // 音源データの読み込み
        g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
        g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
        g_AtkSetup.LoadData(WSD_SNARE, "WSD_SNARE");
    }
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(SoundArchivePlayer, InitializeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    AllocateSoundArchivePlayerBuffer();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        nullptr,
        &soundDataManager,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer ), ".*");

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        &soundArchive,
        nullptr,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer ), ".*");

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        &soundArchive,
        &soundDataManager,
        nullptr, g_MemSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer ), ".*");

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        &soundArchive,
        &soundDataManager,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
        nullptr, g_MemSizeForStreamBuffer ), ".*");

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        &soundArchive,
        &soundDataManager,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer - 1,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer ), ".*");

    EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.Initialize(
        &soundArchive,
        &soundDataManager,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer - 1 ), ".*");

    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchivePlayer, InitializeUsingStreamInstanceBufferDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    nn::atk::SoundArchivePlayer::InitializeParam param;
    param.pSoundArchive = &soundArchive;
    param.pSoundDataManager = &soundDataManager;
    param.enablePreparingStreamInstanceBufferFromSetupBuffer = false;

    //  setupBuffer, strmBuffer, strmInstBuffer のバッファを 32 だけ大きめに確保します ( 32 は適当な数字です )
    //  これにより、「サイズが多いテスト」と「アラインがズレているテスト」でアサートしなかったときにメモリ破壊を起こさないようにしています。
    const size_t setupSize = soundArchivePlayer.GetRequiredMemSize( param );
    uint8_t* setupBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  setupSize + 32, nn::atk::SoundArchivePlayer::BufferAlignSize ) );
    const size_t strmSize = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
    const size_t strmSizeMax = nn::atk::SoundArchivePlayer::StreamBufferTimesMax * strmSize;
    uint8_t* strmBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  strmSizeMax + 32, nn::audio::BufferAlignSize ) );
    const size_t strmInstSize = soundArchivePlayer.GetRequiredStreamInstanceSize( &soundArchive );
    uint8_t* strmInstBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  strmInstSize + 32, nn::audio::BufferAlignSize ) );

    //  nullptr テスト
    param.pSetupBuffer = nullptr;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = nullptr;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = nullptr;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    //  サイズが多いテスト
    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize + 1;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSizeMax + 1;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize + 1;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    //  サイズが少ないテスト
    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize - 1;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize - 1;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize - 1;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    //  アラインがズレているテスト
    param.pSetupBuffer = setupBuffer + 1;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer + 1;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer + 1;
    param.streamInstanceBufferSize = strmInstSize;
    EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.Initialize( param ), ".*" );

    g_Allocator.Free( strmInstBuffer );
    g_Allocator.Free( strmBuffer );
    g_Allocator.Free( setupBuffer );
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

TEST( SoundArchivePlayer, AddonSoundArchiveDeathTest )
{
    nnt::atk::util::OnPreAtkTest();
    const int AddonSoundArchiveCount = 3;

    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );
    // SoundArchivePlayer に AddonSoundArchive を追加できない初期化
    {
        // SoundArchivePlayer の初期化のための設定
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

        AddonSoundArchiveSet set[AddonSoundArchiveCount];
        InitializeAddonSoundArchiveSet( set, AddonSoundArchiveCount );

        // SoundArchivePlayer が未初期化
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive("Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager), ".*");

        g_MemSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize( &soundArchive, 0, 0 );
        g_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
        g_MemSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
        g_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForStreamBuffer, nn::audio::BufferAlignSize );
        nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
        initializeParam.pSoundArchive = &soundArchive;
        initializeParam.pSoundDataManager = &soundDataManager;
        initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
        initializeParam.setupBufferSize = g_MemSizeForSoundArchivePlayer;
        initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
        initializeParam.streamBufferSize = g_MemSizeForStreamBuffer;
        initializeParam.addonSoundArchiveCount = 0;
        EXPECT_TRUE( soundArchivePlayer.Initialize( initializeParam ) );

        // SoundArchivePlayer の初期化時、 addonSoundArchiveCount が 0 であるため、
        // SoundArchivePlayer に AddonSoundArchive を追加できない
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive("Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager), ".*");

        FinalizeAddonSoundArchiveSet( set, AddonSoundArchiveCount );
        soundArchivePlayer.Finalize();
    }

    // SoundArchivePlayer に AddonSoundArchive を追加できる初期化
    {
        // SoundArchivePlayer の初期化のための設定
        nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
        nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
        nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
        g_MemSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize( &soundArchive, 0, AddonSoundArchiveCount );
        g_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
        g_MemSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
        g_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForStreamBuffer, nn::audio::BufferAlignSize );

        nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
        initializeParam.pSoundArchive = &soundArchive;
        initializeParam.pSoundDataManager = &soundDataManager;
        initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
        initializeParam.setupBufferSize = g_MemSizeForSoundArchivePlayer;
        initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
        initializeParam.streamBufferSize = g_MemSizeForStreamBuffer;
        initializeParam.addonSoundArchiveCount = AddonSoundArchiveCount;
        EXPECT_TRUE( soundArchivePlayer.Initialize( initializeParam ) );

        AddonSoundArchiveSet set[AddonSoundArchiveCount];
        InitializeAddonSoundArchiveSet( set, AddonSoundArchiveCount );

        // テスト本体

        // nullptr 指定
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive(nullptr, &set[0].addonSoundArchive, &set[0].addonSoundDataManager), ".*");
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive("Addon0", nullptr, &set[0].addonSoundDataManager), ".*");
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive("Addon0", &set[0].addonSoundArchive, nullptr), ".*");

        // SoundArchivePlayer::AddonSoundArchiveNameLengthMax 以上の名前長を持つ名前の指定
        char invalidName[nn::atk::SoundArchivePlayer::AddonSoundArchiveNameLengthMax + 1];
        memset(invalidName, 'c', sizeof(invalidName));
        invalidName[nn::atk::SoundArchivePlayer::AddonSoundArchiveNameLengthMax] = '\0';
        EXPECT_DEATH_IF_SUPPORTED(soundArchivePlayer.AddAddonSoundArchive(invalidName, &set[0].addonSoundArchive, &set[0].addonSoundDataManager), ".*");

        // 指定個数以上の追加（アサートで落ちる）
        {
            soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
            soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
            soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );
            EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager ), ".*" );

            soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
            soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
            soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
        }

        // AddonSoundArchive が未初期化
        nn::atk::AddonSoundArchive invalidAddonSoundArchive;
        EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.AddAddonSoundArchive( "Addon0", &invalidAddonSoundArchive, &set[0].addonSoundDataManager ), ".*" );

        // SoundDataManager が未初期化
        nn::atk::SoundDataManager invalidSoundDataManager;
        EXPECT_DEATH_IF_SUPPORTED( soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &invalidSoundDataManager ), ".*" );


        FinalizeAddonSoundArchiveSet( set, AddonSoundArchiveCount );
        soundArchivePlayer.Finalize();
    }

    g_Allocator.Free( g_pMemoryForStreamBuffer );
    g_Allocator.Free( g_pMemoryForSoundArchivePlayer );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();

    EXPECT_TRUE( true );
}
#endif

TEST(SoundArchivePlayer, InitializeTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    AllocateSoundArchivePlayerBuffer();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    EXPECT_FALSE(soundArchivePlayer.IsAvailable());

    EXPECT_TRUE(soundArchivePlayer.GetRequiredStreamBufferTimes(&soundArchive) > 0 &&
                soundArchivePlayer.GetRequiredStreamBufferTimes(&soundArchive) <= nn::atk::SoundArchivePlayer::StreamBufferTimesMax);

    EXPECT_TRUE(soundArchivePlayer.Initialize(
                    &soundArchive,
                    &soundDataManager,
                    g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
                    g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer));

    EXPECT_TRUE(soundArchivePlayer.IsAvailable());

    EXPECT_TRUE(&soundArchivePlayer.GetSoundArchive() == &soundArchive);

    soundArchivePlayer.Finalize();
    EXPECT_FALSE(soundArchivePlayer.IsAvailable());

    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchivePlayer, InitializeTestWithUserParamAndCache)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    AllocateSoundArchivePlayerBuffer();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    EXPECT_FALSE(soundArchivePlayer.IsAvailable());

    EXPECT_TRUE(soundArchivePlayer.GetRequiredStreamBufferTimes(&soundArchive) > 0 &&
                soundArchivePlayer.GetRequiredStreamBufferTimes(&soundArchive) <= nn::atk::SoundArchivePlayer::StreamBufferTimesMax);

    // UserParamSize には SoundArchivePlayer::UserParamBoundary の倍数でない適当な値を設定
    const size_t UserParamSize = 10;
    NN_STATIC_ASSERT(UserParamSize % nn::atk::SoundArchivePlayer::UserParamBoundary != 0 &&
                     nn::atk::SoundArchivePlayer::UserParamBoundary % UserParamSize != 0);

    // InitializeTest では行わなかった初期化項目の設定
    size_t memSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize(&soundArchive, UserParamSize);
    void* pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    size_t memSizeForCacheBuffer = soundArchivePlayer.GetRequiredStreamCacheSize(&soundArchive, 16 * 1024);
    void* pMemoryForCacheBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, memSizeForCacheBuffer, nn::atk::FsSoundArchive::BufferAlignSize);

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &soundArchive;
    initializeParam.pSoundDataManager = &soundDataManager;
    initializeParam.pSetupBuffer = pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = g_MemSizeForStreamBuffer;
    initializeParam.pStreamCacheBuffer = pMemoryForCacheBuffer;
    initializeParam.streamCacheSize = memSizeForCacheBuffer;
    initializeParam.userParamSizePerSound = UserParamSize;

    EXPECT_TRUE(soundArchivePlayer.Initialize(initializeParam));
    EXPECT_TRUE(soundArchivePlayer.GetSoundUserParamSize() == nn::util::align_up(UserParamSize, nn::atk::SoundArchivePlayer::UserParamBoundary));

    EXPECT_TRUE(soundArchivePlayer.IsAvailable());

    g_Allocator.Free(pMemoryForSoundArchivePlayer);
    g_Allocator.Free(pMemoryForCacheBuffer);

    soundArchivePlayer.Finalize();
    EXPECT_FALSE(soundArchivePlayer.IsAvailable());

    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchivePlayer, StartSoundTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    AllocateSoundArchivePlayerBuffer();
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    EXPECT_TRUE(soundArchivePlayer.Initialize(
                    &soundArchive,
                    &soundDataManager,
                    g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
                    g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer));

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundHandle soundHandleHold;
    nn::atk::SoundHandle soundHandlePrepare;

    // テスト本体
    EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.HoldSound( &soundHandleHold, SE_YOSHI).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.PrepareSound( &soundHandlePrepare, WSD_SNARE).IsSuccess());

    soundArchivePlayer.Finalize();

    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchivePlayer, StartSoundUsingStreamInstanceBufferTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    nn::atk::SoundArchivePlayer::InitializeParam param;
    param.pSoundArchive = &soundArchive;
    param.pSoundDataManager = &soundDataManager;
    param.enablePreparingStreamInstanceBufferFromSetupBuffer = false;

    const size_t setupSize = soundArchivePlayer.GetRequiredMemSize( param );
    void* setupBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  setupSize, nn::atk::SoundArchivePlayer::BufferAlignSize ) );
    const size_t strmSize = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
    void* strmBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  strmSize, nn::audio::BufferAlignSize ) );
    const size_t strmInstSize = soundArchivePlayer.GetRequiredStreamInstanceSize( &soundArchive );
    void* strmInstBuffer = reinterpret_cast<uint8_t*>( nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  strmInstSize, nn::audio::BufferAlignSize ) );

    param.pSetupBuffer = setupBuffer;
    param.setupBufferSize = setupSize;
    param.pStreamBuffer = strmBuffer;
    param.streamBufferSize = strmSize;
    param.pStreamInstanceBuffer = strmInstBuffer;
    param.streamInstanceBufferSize = strmInstSize;

    EXPECT_TRUE( soundArchivePlayer.Initialize( param ) );
    LoadData();

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundHandle soundHandleHold;
    nn::atk::SoundHandle soundHandlePrepare;

    // テスト本体
    EXPECT_TRUE(soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.HoldSound( &soundHandleHold, SE_YOSHI).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.PrepareSound( &soundHandlePrepare, WSD_SNARE).IsSuccess());

    soundArchivePlayer.Finalize();

    g_Allocator.Free( strmInstBuffer );
    g_Allocator.Free( strmBuffer );
    g_Allocator.Free( setupBuffer );
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(SoundArchivePlayer, GetInstanceNumberTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    AllocateSoundArchivePlayerBuffer();
    LoadData();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    EXPECT_TRUE(soundArchivePlayer.Initialize(
                    &soundArchive,
                    &soundDataManager,
                    g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
                    g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer));

    nn::atk::SoundHandle soundHandle;

    // テスト本体
    EXPECT_TRUE(soundArchivePlayer.GetSoundPlayerCount() > 0);

    int seqSoundNum = soundArchivePlayer.GetFreeSequenceSoundCount();
    EXPECT_TRUE(seqSoundNum > 0);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, SEQ_MARIOKART).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.GetFreeSequenceSoundCount() == seqSoundNum - 1);
    soundHandle.Stop( 0 );
    EXPECT_TRUE(soundArchivePlayer.GetFreeSequenceSoundCount() == seqSoundNum);

    int waveSoundNum = soundArchivePlayer.GetFreeWaveSoundCount();
    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, WSD_SNARE).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.GetFreeWaveSoundCount() == waveSoundNum - 1);
    soundHandle.Stop( 0 );
    EXPECT_TRUE(soundArchivePlayer.GetFreeWaveSoundCount() == waveSoundNum);

    int streamSoundNum = soundArchivePlayer.GetFreeStreamSoundCount();
    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, STRM_MARIOKART).IsSuccess());
    EXPECT_TRUE(soundArchivePlayer.GetFreeStreamSoundCount() == streamSoundNum - 1);
    soundHandle.Stop( 0 );
    EXPECT_TRUE(soundArchivePlayer.GetFreeStreamSoundCount() == streamSoundNum);

    soundArchivePlayer.Finalize();
    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SoundArchivePlayer, AddonSoundArchiveAddRemoveTest )
{
    nnt::atk::util::OnPreAtkTest();
    const int AddonSoundArchiveCount = 3;

    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    // SoundArchivePlayer の初期化のための設定
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
    g_MemSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize( &soundArchive, 0, AddonSoundArchiveCount );
    g_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    g_MemSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
    g_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForStreamBuffer, nn::audio::BufferAlignSize );

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &soundArchive;
    initializeParam.pSoundDataManager = &soundDataManager;
    initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = g_MemSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = g_MemSizeForStreamBuffer;
    initializeParam.addonSoundArchiveCount = AddonSoundArchiveCount;
    EXPECT_TRUE( soundArchivePlayer.Initialize( initializeParam ) );

    AddonSoundArchiveSet set[AddonSoundArchiveCount];
    InitializeAddonSoundArchiveSet(set, AddonSoundArchiveCount );

    // テスト本体
    const char* const AddonSoundArchiveNames[3] = {"Addon0", "Addon1", "Addon2"};

    // 通常の順序で追加・削除
    {
        soundArchivePlayer.AddAddonSoundArchive( AddonSoundArchiveNames[0], &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( AddonSoundArchiveNames[1], &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( AddonSoundArchiveNames[2], &set[2].addonSoundArchive, &set[2].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
    }

    // 逆順で追加
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
    }

    // 適当な順序で追加
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
    }

    // 逆順で削除
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );

    }

    // 適当な順序で削除
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
    }

    // 同名の AddonSoundArchive の追加（現状はできてしまう）
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
    }

    // 同じ物を２回削除（警告が出るだけで落ちない想定）
    {
        soundArchivePlayer.AddAddonSoundArchive( "Addon0", &set[0].addonSoundArchive, &set[0].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon1", &set[1].addonSoundArchive, &set[1].addonSoundDataManager );
        soundArchivePlayer.AddAddonSoundArchive( "Addon2", &set[2].addonSoundArchive, &set[2].addonSoundDataManager );

        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[0].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[1].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );
        soundArchivePlayer.RemoveAddonSoundArchive( &set[2].addonSoundArchive );

    }

    FinalizeAddonSoundArchiveSet( set, AddonSoundArchiveCount );

    soundArchivePlayer.Finalize();
    g_Allocator.Free( g_pMemoryForStreamBuffer );
    g_Allocator.Free( g_pMemoryForSoundArchivePlayer );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();

    EXPECT_TRUE( true );
} // NOLINT(impl/function_size)

TEST( SoundArchivePlayer, AddonSoundArchiveStateTest )
{
    nnt::atk::util::OnPreAtkTest();
    const int AddonSoundArchiveCount = 3;

    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    // SoundArchivePlayer の初期化のための設定
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();
    g_MemSizeForSoundArchivePlayer = soundArchivePlayer.GetRequiredMemSize( &soundArchive, 0, AddonSoundArchiveCount );
    g_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    g_MemSizeForStreamBuffer = soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
    g_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory( g_Allocator, g_MemSizeForStreamBuffer, nn::audio::BufferAlignSize );

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &soundArchive;
    initializeParam.pSoundDataManager = &soundDataManager;
    initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = g_MemSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = g_MemSizeForStreamBuffer;
    initializeParam.addonSoundArchiveCount = AddonSoundArchiveCount;
    EXPECT_TRUE( soundArchivePlayer.Initialize( initializeParam ) );

    AddonSoundArchiveSet set[AddonSoundArchiveCount];
    InitializeAddonSoundArchiveSet(set, AddonSoundArchiveCount );

    // テスト本体
    const char* const AddonSoundArchiveNames[3] = {"Addon0", "Addon1", "Addon2"};

    {
        EXPECT_EQ(0, soundArchivePlayer.GetAddonSoundArchiveCount());

        for(int i = 0; i < AddonSoundArchiveCount; ++i)
        {
            soundArchivePlayer.AddAddonSoundArchive( AddonSoundArchiveNames[i], &set[i].addonSoundArchive, &set[i].addonSoundDataManager );
            EXPECT_EQ(i + 1, soundArchivePlayer.GetAddonSoundArchiveCount());
            EXPECT_EQ(&set[i].addonSoundArchive, soundArchivePlayer.GetAddonSoundArchive(i));
            EXPECT_EQ(&set[i].addonSoundArchive, soundArchivePlayer.GetAddonSoundArchive(AddonSoundArchiveNames[i]));
            EXPECT_EQ(0, strncmp(AddonSoundArchiveNames[i], soundArchivePlayer.GetAddonSoundArchiveName(i), nn::atk::SoundArchivePlayer::AddonSoundArchiveNameLengthMax));
            EXPECT_EQ(&set[i].addonSoundDataManager, soundArchivePlayer.GetAddonSoundDataManager(AddonSoundArchiveNames[i]));
        }

        for(int i = 0; i < AddonSoundArchiveCount; ++i)
        {
            soundArchivePlayer.RemoveAddonSoundArchive( &set[i].addonSoundArchive );
            EXPECT_EQ(AddonSoundArchiveCount - (i + 1), soundArchivePlayer.GetAddonSoundArchiveCount());
        }
    }

    FinalizeAddonSoundArchiveSet( set, AddonSoundArchiveCount );

    soundArchivePlayer.Finalize();
    g_Allocator.Free( g_pMemoryForStreamBuffer );
    g_Allocator.Free( g_pMemoryForSoundArchivePlayer );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();

    EXPECT_TRUE( true );
}

TEST( SoundArchivePlayer, GetStreamSoundFilePathTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );

    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );
    AllocateSoundArchivePlayerBuffer();

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundArchive& soundArchive = g_AtkSetup.GetSoundArchive();
    nn::atk::SoundDataManager& soundDataManager = g_AtkSetup.GetSoundDataManager();

    char filePath[nn::fs::EntryNameLengthMax + 1];

    EXPECT_TRUE( soundArchivePlayer.Initialize(
        &soundArchive,
        &soundDataManager,
        g_pMemoryForSoundArchivePlayer, g_MemSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, g_MemSizeForStreamBuffer ) );

    LoadData();

    // テスト本体

    // 存在しないIDや、ストリームサウンドではない時は失敗
    EXPECT_FALSE( soundArchive.ReadStreamSoundFilePath( filePath, nn::fs::EntryNameLengthMax + 1, nn::atk::SoundArchive::InvalidId ) );
    EXPECT_FALSE( soundArchive.ReadStreamSoundFilePath( filePath, nn::fs::EntryNameLengthMax + 1, SE_YOSHI ) );

    // 短すぎるパスを渡すと失敗
    EXPECT_FALSE( soundArchive.ReadStreamSoundFilePath( filePath, 1, STRM_MARIOKART ) );

    // 以下の場合は成功
    EXPECT_TRUE( soundArchive.ReadStreamSoundFilePath( filePath, nn::fs::EntryNameLengthMax + 1, STRM_MARIOKART ) );
    EXPECT_TRUE(std::strncmp(filePath, "Rom:/stream/kart_title.32.dspadpcm.bfstm", nn::fs::EntryNameLengthMax) == 0);

    soundArchivePlayer.Finalize();

    FreeSoundArchivePlayerBuffer();
    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST( SoundArchivePlayer, ReadWaveSoundDataInfoTest )
{
    nnt::atk::util::OnPreAtkTest();
    nnt::atk::util::AtkCommonSetup atkSetup;
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    atkSetup.Initialize( g_Allocator );

    // WaveSoundDataInfo の読み込みテスト
    atkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
    atkSetup.LoadData(SE_STEREO_YOSHIWIHAHO, "SE_STEREO_YOSHIWIHAHO");
    atkSetup.LoadData(SE_SIN440_LOOP, "SE_SIN440_LOOP");
    {
        nn::atk::WaveSoundDataInfo info;
        EXPECT_TRUE(atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, SE_YOSHI).IsSuccess());
        EXPECT_EQ(info.channelCount, 1);
        EXPECT_EQ(info.sampleRate, 32000);
        EXPECT_FALSE(info.loopFlag);
    }
    {
        nn::atk::WaveSoundDataInfo info;
        EXPECT_TRUE(atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, SE_STEREO_YOSHIWIHAHO).IsSuccess());
        EXPECT_EQ(info.channelCount, 2);
        EXPECT_EQ(info.sampleRate, 44100);
        EXPECT_FALSE(info.loopFlag);
    }
    {
        nn::atk::WaveSoundDataInfo info;
        EXPECT_TRUE(atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, SE_SIN440_LOOP).IsSuccess());
        EXPECT_EQ(info.channelCount, 1);
        EXPECT_EQ(info.sampleRate, 48000);
        EXPECT_TRUE(info.loopFlag);
    }
    {
        // ロードしていないサウンドの WaveSoundDataInfo の取得
        nn::atk::WaveSoundDataInfo info;
        auto result = atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, WSD_ORGAN);
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerSoundNotLoaded::Includes(result));
    }
    {
        // 無効な ID の WaveSoundDataInfo の取得
        nn::atk::WaveSoundDataInfo info;
        auto result = atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, nn::atk::SoundArchive::InvalidId);
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerInvalidId::Includes(result));
    }
    {
        // 無効な名前の WaveSoundDataInfo の取得
        nn::atk::WaveSoundDataInfo info;
        auto result = atkSetup.GetSoundArchivePlayer().ReadWaveSoundDataInfo(&info, "InvalidName");
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerInvalidName::Includes(result));
    }

    atkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SoundArchivePlayer, ReadStreamSoundDataInfoTest )
{
    nnt::atk::util::OnPreAtkTest();
    nnt::atk::util::AtkCommonSetup atkSetup;
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    atkSetup.Initialize( g_Allocator );

    // StreamSoundDataInfo の読み込みテスト
    {
        nn::atk::StreamSoundDataInfo info;
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, STRM_MARIOKART ).IsSuccess() );
        EXPECT_EQ( info.channelCount, 2 );
        EXPECT_EQ( info.sampleRate, 32000 );
        EXPECT_TRUE( info.loopFlag );
    }
    {
        nn::atk::StreamSoundDataInfo info;
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, STRM_PIANO16 ).IsSuccess() );
        EXPECT_EQ( info.channelCount, 1 );
        EXPECT_EQ( info.sampleRate, 16000 );
        EXPECT_TRUE( info.loopFlag );
    }
    {
        nn::atk::StreamSoundDataInfo info;
        const auto result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, "STRM_MARIOKART_48k" );
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerInvalidName::Includes(result));

        //  文字列ラベルを読み込んで文字列引きできるようにします
        atkSetup.LoadLabelStringData( g_Allocator );
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, "STRM_MARIOKART_48k" ).IsSuccess() );
        EXPECT_EQ( info.channelCount, 2 );
        EXPECT_EQ( info.sampleRate, 48000 );
        EXPECT_TRUE( info.loopFlag );
    }
#ifndef NN_SDK_BUILD_RELEASE
    {
        //  Opus の読み込みには失敗する
        nn::atk::StreamSoundDataInfo info;
        EXPECT_DEATH_IF_SUPPORTED( atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, STRM_MARIOKART_OPUS ), ".*" );
    }
#endif
    {
        // 無効な ID の StreamSoundDataInfo の取得
        nn::atk::StreamSoundDataInfo info;
        nn::atk::SoundArchive::ItemId testId[] = { nn::atk::SoundArchive::InvalidId, SE_YOSHI };
        for(const auto id : testId )
        {
            const auto result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo( &info, id );
            EXPECT_TRUE(result.IsFailure());
        }
    }
    {
        // 無効な名前の StreamSoundDataInfo の取得
        nn::atk::StreamSoundDataInfo info;
        auto result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo(&info, "InvalidName");
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerInvalidName::Includes(result));
    }
    {
        //  STRM を再生中に StreamSoundDataInfo を取得
        nn::atk::StreamSoundDataInfo info;
        nn::atk::SoundHandle handle;

        atkSetup.GetSoundArchivePlayer().StartSound( &handle, STRM_MARIOKART );
        auto result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundDataInfo(&info, STRM_MARIOKART);
        EXPECT_TRUE(result.IsFailure());
        EXPECT_TRUE(nn::atk::ResultSoundArchivePlayerStreamSoundIsPlaying::Includes(result));
    }

    atkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( SoundArchivePlayer, ReadStreamSoundRegionDataInfoTest )
{
    nnt::atk::util::OnPreAtkTest();
    nnt::atk::util::AtkCommonSetup atkSetup;
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    atkSetup.Initialize( g_Allocator );

    const size_t bufferSize = nn::atk::SoundArchivePlayer::GetRequiredWorkBufferSizeToReadStreamSoundHeader();
    void* buffer = g_Allocator.Allocate( bufferSize + 1 );  //  アライン確認のために 1 大きく確保します

    //  STRM_NAMEDREGION_TEST の正解データです
    nn::atk::StreamSoundRegionDataInfo NamedRegionCorrectData[] =
    {
        { 40770u, 120797u, 1, "RegionA" },
        { 120797u, 202600u, 2, "RegionB" },
    };

    // StreamSoundRegionDataInfo の読み込みテスト
    {
        nn::atk::StreamSoundRegionDataInfo info;
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, "RegionA", buffer, bufferSize ).IsSuccess() );
        EXPECT_EQ( info.startSamplePosition, NamedRegionCorrectData[0].startSamplePosition );
        EXPECT_EQ( info.endSamplePosition, NamedRegionCorrectData[0].endSamplePosition );
        EXPECT_EQ( info.regionNo, NamedRegionCorrectData[0].regionNo );
        EXPECT_STREQ( info.regionName, NamedRegionCorrectData[0].regionName );

        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, "STRM_NAMEDREGION_TEST", "RegionB", buffer, bufferSize ).IsFailure() );
        atkSetup.LoadLabelStringData( g_Allocator );

        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, "STRM_NAMEDREGION_TEST", "RegionB", buffer, bufferSize ).IsSuccess() );
        EXPECT_EQ( info.startSamplePosition, NamedRegionCorrectData[1].startSamplePosition );
        EXPECT_EQ( info.endSamplePosition, NamedRegionCorrectData[1].endSamplePosition );
        EXPECT_EQ( info.regionNo, NamedRegionCorrectData[1].regionNo );
        EXPECT_STREQ( info.regionName, NamedRegionCorrectData[1].regionName );

        //  アラインの確認のために buffer を 1 つずらして渡します
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_REGION_JUMP, "", static_cast<char*>( buffer ) + 1, bufferSize ).IsSuccess() );
        EXPECT_EQ( info.startSamplePosition, 0u );
        EXPECT_EQ( info.endSamplePosition, 40770u );
        EXPECT_EQ( info.regionNo, 0 );
        EXPECT_STREQ( info.regionName, "" );
    }

    // 複数 StreamSoundRegionDataInfo の読み込みテスト
    {
        const char* regionName[] =
        {
            NamedRegionCorrectData[0].regionName,
            NamedRegionCorrectData[1].regionName
        };
        const int regionCount = sizeof(regionName) / sizeof(regionName[0]);
        nn::atk::StreamSoundRegionDataInfo info[regionCount];

        for(int i = 0; i < regionCount; i++)
        {
            regionName[i] = NamedRegionCorrectData[i].regionName;
        }
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( info, STRM_NAMEDREGION_TEST, regionName, regionCount, buffer, bufferSize ).IsSuccess() );

        for(int i = 0; i < regionCount; i++)
        {
            EXPECT_EQ( info[i].startSamplePosition, NamedRegionCorrectData[i].startSamplePosition );
            EXPECT_EQ( info[i].endSamplePosition, NamedRegionCorrectData[i].endSamplePosition );
            EXPECT_EQ( info[i].regionNo, NamedRegionCorrectData[i].regionNo );
            EXPECT_STREQ( info[i].regionName, NamedRegionCorrectData[i].regionName );
        }
    }

    //  STRM_NAMEDREGION_WITHLOOP の正解データです
    nn::atk::StreamSoundRegionDataInfo NamedRegionWithLoopCorrectData[] =
    {
        { 40770u, 120797u, 1, "RegionA" },
        { 120797u, 202600u, 2, "RegionB" },
    };

    // ループ情報とリージョン情報があるストリームサウンドでも正しく情報が取れるかのテスト
    {
        nn::atk::StreamSoundRegionDataInfo info;
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_WITHLOOP, "RegionA", buffer, bufferSize ).IsSuccess() );
        EXPECT_EQ( info.startSamplePosition, NamedRegionWithLoopCorrectData[0].startSamplePosition );
        EXPECT_EQ( info.endSamplePosition, NamedRegionWithLoopCorrectData[0].endSamplePosition );
        EXPECT_EQ( info.regionNo, NamedRegionWithLoopCorrectData[0].regionNo );
        EXPECT_STREQ( info.regionName, NamedRegionWithLoopCorrectData[0].regionName );

        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_WITHLOOP, "RegionB", buffer, bufferSize ).IsSuccess() );
        EXPECT_EQ( info.startSamplePosition, NamedRegionWithLoopCorrectData[1].startSamplePosition );
        EXPECT_EQ( info.endSamplePosition, NamedRegionWithLoopCorrectData[1].endSamplePosition );
        EXPECT_EQ( info.regionNo, NamedRegionWithLoopCorrectData[1].regionNo );
        EXPECT_STREQ( info.regionName, NamedRegionWithLoopCorrectData[1].regionName );
    }

    //  リージョン情報の読み取り失敗テスト
    {
        nn::atk::StreamSoundRegionDataInfo info;
        {
            nn::Result result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_MARIOKART, "RegionA", buffer, bufferSize );
            EXPECT_TRUE( result.IsFailure() );
            EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerStreamSoundHasNoRegion::Includes(result) );
        }
        {
            nn::Result result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, "NotExistRegion", buffer, bufferSize );
            EXPECT_TRUE( result.IsFailure() );
            EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerStreamSoundRegionNotFound::Includes(result) );
        }
        {
            nn::Result result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, "STRM_NOT_EXIST", "RegionA", buffer, bufferSize );
            EXPECT_TRUE( result.IsFailure() );
            EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerInvalidName::Includes(result) );
        }
        {
            nn::Result result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, nn::atk::InvalidSoundId, "RegionA", buffer, bufferSize );
            EXPECT_TRUE( result.IsFailure() );
            EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerReadStreamSoundFilePathFailed::Includes(result) );
        }

        nn::atk::SoundHandle handle;
        EXPECT_TRUE( atkSetup.GetSoundArchivePlayer().StartSound( &handle, STRM_NAMEDREGION_TEST ).IsSuccess() );
        {
            nn::Result result = atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, "RegionA", buffer, bufferSize );
            EXPECT_TRUE( result.IsFailure() );
            EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerStreamSoundIsPlaying::Includes(result) );
        }
        handle.Stop(0);
    }

#ifndef NN_SDK_BUILD_RELEASE
    //  アサートにかかる失敗テスト
    {
        nn::atk::StreamSoundRegionDataInfo info;
        EXPECT_DEATH_IF_SUPPORTED( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, "RegionA", buffer, bufferSize - 1 ).IsSuccess(), ".*" );
        EXPECT_DEATH_IF_SUPPORTED( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, nullptr, 1, buffer, bufferSize ).IsSuccess(), ".*" );
        EXPECT_DEATH_IF_SUPPORTED( atkSetup.GetSoundArchivePlayer().ReadStreamSoundRegionDataInfo( &info, STRM_NAMEDREGION_TEST, "RegionA", nullptr, bufferSize ).IsSuccess(), ".*" );
    }
#endif

    g_Allocator.Free( buffer );
    atkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
} // NOLINT(impl/function_size)

TEST( SoundArchivePlayer, CheckStreamSoundFileExistingTest )
{
    nnt::atk::util::OnPreAtkTest();
    nnt::atk::util::AtkCommonSetup atkSetup;
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    atkSetup.Initialize( g_Allocator );

    {
        nn::Result result = atkSetup.GetSoundArchivePlayer().CheckStreamSoundFileExisting(STRM_MARIOKART);
        EXPECT_TRUE( result.IsSuccess() );
    }
    {
        nn::Result result = atkSetup.GetSoundArchivePlayer().CheckStreamSoundFileExisting(nn::atk::InvalidSoundId);
        EXPECT_TRUE( result.IsFailure() );
        EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerReadStreamSoundFilePathFailed::Includes(result) );
    }
    {
        nn::Result result = atkSetup.GetSoundArchivePlayer().CheckStreamSoundFileExisting("STRM_MARIOKART_48k");
        EXPECT_TRUE( result.IsFailure() );
        EXPECT_TRUE( nn::atk::ResultSoundArchivePlayerInvalidName::Includes(result) );

        //  文字列ラベルを読み込んで文字列引きできるようにします
        atkSetup.LoadLabelStringData( g_Allocator );
        result = atkSetup.GetSoundArchivePlayer().CheckStreamSoundFileExisting("STRM_MARIOKART_48k");
        EXPECT_TRUE(result.IsSuccess());
    }

    atkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
