﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"
#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/mem.h>

namespace
{
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds( 16 );
    const float FloatThreshold = 1.0f / 65536.0f;
    const float PanRange = 0.9f;

    const int MemoryHeapSize = 32 * 1024 * 1024;
    char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator g_Allocator;
    nnt::atk::util::FsCommonSetup g_FsSetup;

#define     EXPECT_FLOAT_EQ_LOCAL( lhs, rhs )   EXPECT_TRUE( std::abs( (lhs) - (rhs) ) < FloatThreshold )

    class Sound3DSetup : public nnt::atk::util::AtkCommonSetup
    {
    public:
        virtual void Initialize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            //  SoundArchivePlayer を除く AtkCommonSetup の初期化
            nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
            InitializeSoundSystem( initializeParam.GetSoundSystemParam(), allocator );
            InitializeSoundHeap( initializeParam.GetSoundHeapSize(), allocator );
            InitializeSoundArchive( initializeParam.GetSoundArchivePath(), allocator );
            InitializeSoundDataManager( allocator );

            //  Sound3DManager の初期化
            nn::atk::SoundArchive& soundArchive = GetSoundArchive();
            const size_t size = m_Sound3DManager.GetRequiredMemSize( &soundArchive );

            m_pMemoryForSound3DManager = nnt::atk::util::AllocateUninitializedMemory( allocator, size, nn::atk::Sound3DManager::BufferAlignSize );
            EXPECT_TRUE( m_Sound3DManager.Initialize( &soundArchive, m_pMemoryForSound3DManager , size ) );

            m_Sound3DManager.SetSonicVelocity( 340.0f / 60.0f );
            m_Sound3DManager.SetPanRange( PanRange );
            m_Sound3DManager.SetEngine( &m_Sound3DEngine );

            //  SoundArchivePlayer の初期化
            InitializeSoundArchivePlayer( allocator );
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            FinalizeSoundArchivePlayer( allocator );

            m_Sound3DManager.Finalize();
            allocator.Free( m_pMemoryForSound3DManager );

            FinalizeSoundDataManager( allocator );
            FinalizeSoundArchive( allocator );
            FinalizeSoundHeap( allocator );
            FinalizeSoundSystem( allocator );
        }
        nn::atk::Sound3DManager& GetSound3DManager() NN_NOEXCEPT
        {
            return m_Sound3DManager;
        }
        const nn::atk::Sound3DEngine& GetSound3DEngine() const NN_NOEXCEPT
        {
            return m_Sound3DEngine;
        }

    private:
        nn::atk::Sound3DManager m_Sound3DManager;
        void* m_pMemoryForSound3DManager;

        nn::atk::Sound3DEngine m_Sound3DEngine;
    };
    Sound3DSetup g_AtkSetup;

    void LoadData() NN_NOEXCEPT
    {
        g_AtkSetup.LoadData( SE_IDLE_NOVOLUME, "SE_IDLE_NOVOLUME" );
        g_AtkSetup.LoadData( SE_IDLE_DOPPLER, "SE_IDLE_DOPPLER" );
        g_AtkSetup.LoadData( SE_WIHAHO, "SE_WIHAHO" );
    }
    nn::util::Vector3fType GetVector3f(float x, float y, float z) NN_NOEXCEPT
    {
        nn::util::Vector3fType v;
        nn::util::VectorSet( &v, x, y, z );
        return v;
    }
    //  SoundArchive が持つサウンドの数を取得します
    int GetSoundArchiveHavingSoundCount(nn::atk::SoundArchive& soundArchive) NN_NOEXCEPT
    {
        int count = 0;
        nn::atk::SoundArchive::SoundArchivePlayerInfo soundArchivePlayerInfo;
        if ( soundArchive.ReadSoundArchivePlayerInfo( &soundArchivePlayerInfo ) )
        {
            count += soundArchivePlayerInfo.sequenceSoundCount;
            count += soundArchivePlayerInfo.streamSoundCount;
            count += soundArchivePlayerInfo.waveSoundCount;
        }
        return count;
    }
    //  Listener 行列を設定します
    void SetListenerMatrix(nn::atk::Sound3DListener& listener, const nn::util::Vector3fType& position, const nn::util::Vector3fType& direction) NN_NOEXCEPT
    {
        // リスナーの Up 方向のベクトル
        nn::util::Vector3fType upVec;
        nn::util::VectorSet( &upVec, 0.0f, 1.0f, 0.0f );

        // リスナー行列生成, 設定
        nn::util::Vector3fType target;
        nn::util::Matrix4x3fType matrix;
        nn::util::VectorAdd( &target, position, direction );
        nn::util::MatrixLookAtRightHanded( &matrix, position, target, upVec );
        listener.SetMatrix( matrix );
    }

    //  x <= y のときに a <= b となるような条件をテストします
    bool TestMagnitudeRelationship(float x, float y, float a, float b) NN_NOEXCEPT
    {
        //  テストとライブラリでの計算方法の違いや計算誤差を吸収するために判定を緩めます
        //      計算方法の違いによって生まれる最大の差は 0.26 くらいの値です
        //      (atk::Sound3DCalculator::CalculatePanSurround() の center_z の値です)
        //      testAtk_Sound3D.cpp では最大の差が出るようなテストは行っていないため
        //      0.26 を参考にその半分の 0.125 を最大誤差許容範囲とします
        const float looseness = 0.125f / 2.0f;

        if( x <= y )
        {
            if( a - looseness <= b + looseness )
            {
                return true;
            }
        }
        else
        {
            if( a + looseness >= b - looseness )
            {
                return true;
            }
        }

        //  誤差を吸収しても失敗した場合は値を出力しておきます
        NN_SDK_LOG( "-- TestMagnitudeRelationship --\n" );
        NN_SDK_LOG( "  x = %.16f\n", x );
        NN_SDK_LOG( "  y = %.16f\n", y );
        NN_SDK_LOG( "  a = %.16f\n", a );
        NN_SDK_LOG( "  b = %.16f\n", b );
        NN_SDK_LOG( "  x <= y = %d\n", x <= y );
        NN_SDK_LOG( "  a <= b = %d\n", a - looseness <= b + looseness );
        NN_SDK_LOG( "  a >= b = %d\n", a + looseness >= b - looseness );

        return false;
    }

    //  Sound3DActor のテストを行うためのクラスです
    class Sound3DActorTestData
    {
    public:
        void Initialize(nn::atk::SoundArchivePlayer& soundArchivePlayer, nn::atk::Sound3DManager& sound3DManager, const nn::atk::Sound3DListener& sound3DListener, nn::atk::SoundArchive::ItemId soundId) NN_NOEXCEPT
        {
            m_SoundActor.Initialize( &soundArchivePlayer, &sound3DManager );
            m_SoundId = soundId;

            m_pSound3DManager = &sound3DManager;
            m_pSound3DListener = &sound3DListener;
        }
        void Finalize() NN_NOEXCEPT
        {
            m_SoundActor.Finalize();
        }

        void SetPosition(const nn::util::Vector3fType& position) NN_NOEXCEPT
        {
            m_SoundActor.SetPosition( position );
        }
        void SetVelocity(const nn::util::Vector3fType& velocity) NN_NOEXCEPT
        {
            m_SoundActor.SetVelocity( velocity );
        }
        const nn::util::Vector3f GetVelocity() const NN_NOEXCEPT
        {
            return m_SoundActor.GetVelocity();
        }

        void StartSound() NN_NOEXCEPT
        {
            EXPECT_TRUE( m_SoundActor.StartSound( &m_SoundHandle, m_SoundId ).IsSuccess() );
        }
        void Update() NN_NOEXCEPT
        {
            const nn::atk::SoundParam* pParam = m_SoundHandle.GetAmbientParam();
            if( pParam != nullptr )
            {
                //  SE_WIHAHO は途中でサウンドの再生が止まるため、
                //  pParam == nullptr となるタイミングがあります
                const nn::atk:: OutputAmbientParam& tvParam = pParam->GetTvParam();
                m_prvPan = tvParam.GetPan();
                m_prvSPan = tvParam.GetSurroundPan();
                m_prvVolume = tvParam.GetVolume();
                m_prvPitch = pParam->GetPitch();
                m_prvActorVelocity = m_SoundActor.GetVelocity();
                m_prvActorPosition = m_SoundActor.GetPosition();
            }

            //  移動します
            nn::util::Vector3fType pos = m_SoundActor.GetPosition();
            nn::util::VectorAdd( &pos, pos, m_SoundActor.GetVelocity() );
            m_SoundActor.SetPosition( pos );

            if( !m_SoundHandle.IsAttachedSound() )
            {
                //  止まっていたらもう一度鳴らします
                StartSound();
            }
        }

        //  Pan のテストをします
        void TestPan(bool enableSound3DEffect) NN_NOEXCEPT
        {
            const nn::atk::SoundParam* pParam = m_SoundHandle.GetAmbientParam();
            if( pParam == nullptr )
            {
                //  SE_WIHAHO は途中でサウンドの再生が止まるため、
                //  pParam == nullptr となるタイミングがあります
                EXPECT_EQ( m_SoundId, SE_WIHAHO );
                return ;
            }

            const float pan = pParam->GetTvParam().GetPan();
            if( !enableSound3DEffect )
            {
                //  Sound3D の機能が無効ならば Pan == 0
                EXPECT_FLOAT_EQ_LOCAL( pan, 0.0f );
                return ;
            }

            //  Pan の変化は PanRange 以下
            EXPECT_TRUE( -PanRange - FloatThreshold <= pan && pan <= PanRange + FloatThreshold );

            nn::util::Vector3f actorRelativeVector;
            nn::util::VectorTransform( &actorRelativeVector, m_SoundActor.GetPosition(), m_pSound3DListener->GetMatrix() );
            nn::util::Float3 actorRelativePos;  //  リスナーから見たアクターの位置
            nn::util::VectorStore( &actorRelativePos, actorRelativeVector );

            //  リスナーに対して、アクターが右にいれば pan は 0 以上, 左にいれば 0 以下になることを確認します
            EXPECT_EQ( 0.0f <= actorRelativePos.x, 0.0f <= pan );
            EXPECT_EQ( 0.0f >= actorRelativePos.x, 0.0f >= pan );

            if( IsInInteriorSize( m_SoundActor.GetPosition() ) || IsInInteriorSize( m_prvActorPosition ) )
            {
                //  インテリアサイズ内のテスト
                //  TODO: インテリアサイズ内のテストを用意する
            }
            else
            {
                //  インテリアサイズ外のテスト
                nn::util::Vector3f actorPrvRelativeVector;
                nn::util::VectorTransform( &actorPrvRelativeVector, m_prvActorPosition, m_pSound3DListener->GetMatrix() );
                nn::util::Float3 actorRelativePrvPos;  //  リスナーから見たアクターの前の位置
                nn::util::VectorStore( &actorRelativePrvPos, actorPrvRelativeVector );

                if( actorRelativePos.z <= 0.0f )
                {
                    //  リスナーに対してアクターが前にいる
                    const float rad = g_AtkSetup.GetSound3DEngine().GetCalculatePanParam().surroundSpeakerFrontAngle;
                    const float sin = std::sinf( rad ), cos = std::cosf( rad );

                    if( actorRelativePos.x * cos >= actorRelativePos.z * -sin )
                    {
                        //  FR スピーカーよりも右側にいる
                        EXPECT_FLOAT_EQ_LOCAL( pan, PanRange );
                    }
                    else if( actorRelativePos.x * cos <= actorRelativePos.z * sin )
                    {
                        //  FL スピーカーよりも左側にいる
                        EXPECT_FLOAT_EQ_LOCAL( pan, -PanRange );
                    }
                    else
                    {
                        //  FR, FL スピーカーの間

                        //  リスナーから見て右方向に移ったため pan が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, m_prvPan, pan ) );
                        //  リスナーから見て左方向に移ったため pan が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, pan, m_prvPan ) );
                    }
                }
                else
                {
                    //  リスナーに対してアクターが後ろにいる
                    const float rad = g_AtkSetup.GetSound3DEngine().GetCalculatePanParam().surroundSpeakerRearAngle;
                    const float sin = std::sinf( rad ), cos = std::cosf( rad );

                    if( actorRelativePos.x * cos <= actorRelativePos.z * -sin )
                    {
                        //  RR スピーカーよりも右側にいる
                        EXPECT_FLOAT_EQ_LOCAL( pan, PanRange );
                    }
                    else if( actorRelativePos.x * cos >= actorRelativePos.z * sin )
                    {
                        //  RL スピーカーよりも左側にいる
                        EXPECT_FLOAT_EQ_LOCAL( pan, -PanRange );
                    }
                    else
                    {
                        //  RR, RL スピーカーの間

                        //  リスナーから見て右方向に移ったため pan が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, pan, m_prvPan ) );
                        //  リスナーから見て左方向に移ったため pan が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, m_prvPan, pan ) );
                    }
                }
            }
        }

        //  SurroundPan のテストをします
        void TestSurroundPan(bool enableSound3DEffect) NN_NOEXCEPT
        {
            const nn::atk::SoundParam* pParam = m_SoundHandle.GetAmbientParam();
            if( pParam == nullptr )
            {
                //  SE_WIHAHO は途中でサウンドの再生が止まるため、
                //  pParam == nullptr となるタイミングがあります
                EXPECT_EQ( m_SoundId, SE_WIHAHO );
                return ;
            }

            const float span = pParam->GetTvParam().GetSurroundPan() - g_AtkSetup.GetSound3DEngine().GetCalculatePanParam().surroundPanOffset;
            if( !enableSound3DEffect )
            {
                //  Sound3D の機能が無効ならば SPan == 1
                EXPECT_FLOAT_EQ_LOCAL( span, 1.0f );
                return ;
            }

            //  SPan の変化は PanRange + 1 以下
            EXPECT_TRUE( -PanRange + 1.0f - FloatThreshold <= span && span <= PanRange + 1.0f + FloatThreshold );

            nn::util::Vector3f actorRelativeVector;
            nn::util::VectorTransform( &actorRelativeVector, m_SoundActor.GetPosition(), m_pSound3DListener->GetMatrix() );
            nn::util::Float3 actorRelativePos;  //  リスナーから見たアクターの位置
            nn::util::VectorStore( &actorRelativePos, actorRelativeVector );

            if( IsInInteriorSize( m_SoundActor.GetPosition() ) || IsInInteriorSize( m_prvActorPosition ) )
            {
                //  インテリアサイズ内のテスト
                //  TODO: インテリアサイズ内のテストを用意する
            }
            else
            {
                //  インテリアサイズ外のテスト
                nn::util::Vector3f actorPrvRelativeVector;
                nn::util::VectorTransform( &actorPrvRelativeVector, m_prvActorPosition, m_pSound3DListener->GetMatrix() );
                nn::util::Float3 actorRelativePrvPos;  //  リスナーから見たアクターの前の位置
                nn::util::VectorStore( &actorRelativePrvPos, actorPrvRelativeVector );

                //  リスナーに対して、アクターが前に入れば span は 1 以下, 後ろにいれば span は 1 以上であることを確認します
                EXPECT_EQ( 0.0f <= actorRelativePos.z, 1.0f <= span );
                EXPECT_EQ( 0.0f >= actorRelativePos.z, 1.0f >= span );

                if( actorRelativePos.z <= 0.0f )
                {
                    //  リスナーに対してアクターが前にいる
                    const float rad = g_AtkSetup.GetSound3DEngine().GetCalculatePanParam().surroundSpeakerFrontAngle;
                    const float sin = std::sinf( rad ), cos = std::cosf( rad );

                    if( actorRelativePos.x * cos >= actorRelativePos.z * -sin )
                    {
                        //  FR スピーカーよりも右側にいる

                        //  リスナーから見て右方向に移ったため span が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, m_prvSPan, span ) );
                        //  リスナーから見て左方向に移ったため span が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, span, m_prvSPan ) );
                    }
                    else if( actorRelativePos.x * cos <= actorRelativePos.z * sin )
                    {
                        //  FL スピーカーよりも左側にいる

                        //  リスナーから見て右方向に移ったため span が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, span, m_prvSPan ) );
                        //  リスナーから見て左方向に移ったため span が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, m_prvSPan, span ) );
                    }
                    else
                    {
                        //  FR, FL スピーカーの間
                        EXPECT_FLOAT_EQ_LOCAL( span, -PanRange + 1.0f );
                    }
                }
                else
                {
                    //  リスナーに対してアクターが後ろにいる
                    const float rad = g_AtkSetup.GetSound3DEngine().GetCalculatePanParam().surroundSpeakerRearAngle;
                    const float sin = std::sinf( rad ), cos = std::cosf( rad );

                    if( actorRelativePos.x * cos <= actorRelativePos.z * -sin )
                    {
                        //  RR スピーカーよりも右側にいる

                        //  リスナーから見て右方向に移ったため span が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, m_prvSPan, span ) );
                        //  リスナーから見て左方向に移ったため span が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, span, m_prvSPan ) );
                    }
                    else if( actorRelativePos.x * cos >= actorRelativePos.z * sin )
                    {
                        //  FL スピーカーよりも左側にいる

                        //  リスナーから見て右方向に移ったため span が減る
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.x * actorRelativePrvPos.z, actorRelativePos.z * actorRelativePrvPos.x, span, m_prvSPan ) );
                        //  リスナーから見て左方向に移ったため span が増える
                        EXPECT_TRUE( TestMagnitudeRelationship( actorRelativePos.z * actorRelativePrvPos.x, actorRelativePos.x * actorRelativePrvPos.z, m_prvSPan, span ) );
                    }
                    else
                    {
                        //  RR, RL スピーカーの間
                        EXPECT_FLOAT_EQ_LOCAL( span, PanRange + 1.0f );
                    }
                }
            }
        }

        //  volume のテストをします
        void TestVolume(bool enableSound3DEffect) NN_NOEXCEPT
        {
            const nn::atk::SoundParam* pParam = m_SoundHandle.GetAmbientParam();
            if( pParam == nullptr )
            {
                //  SE_WIHAHO は途中でサウンドの再生が止まるため、
                //  pParam == nullptr となるタイミングがあります
                EXPECT_EQ( m_SoundId, SE_WIHAHO );
                return ;
            }

            const float volume = pParam->GetTvParam().GetVolume();
            if( !enableSound3DEffect )
            {
                //  Sound3D の機能が無効ならば Vol == 1
                EXPECT_FLOAT_EQ_LOCAL( volume, 1.0f );
                return ;
            }

            if( IsInMaxVolumeDistance( m_SoundActor.GetPosition() ) )
            {
                //  最大音量距離以内にいるならば、ボリュームは 1 になります
                EXPECT_FLOAT_EQ_LOCAL( volume, 1.0f );
            }
            else
            {
                const float distLisToAct = nn::util::VectorDistance( m_SoundActor.GetPosition(), m_pSound3DListener->GetPosition() );
                const float distLisToPrvAct = nn::util::VectorDistance( m_prvActorPosition, m_pSound3DListener->GetPosition() );

                if( distLisToAct < distLisToPrvAct )
                {
                    //  前の位置よりも近づいているため、ボリュームは増えます
                    EXPECT_GT( volume, m_prvVolume );
                }
                else if( distLisToAct > distLisToPrvAct )
                {
                    //  前の位置よりも遠ざかっているため、ボリュームは減ります
                    EXPECT_LT( volume, m_prvVolume );
                }
                else
                {
                    //  前の位置と同じ距離なので、ボリュームは変わりません
                    EXPECT_FLOAT_EQ_LOCAL( volume, m_prvVolume );
                }
            }
        }

        //  pitch のテストをします
        void TestPitch(bool enableSound3DEffect) NN_NOEXCEPT
        {
            const nn::atk::SoundParam* pParam = m_SoundHandle.GetAmbientParam();
            if( pParam == nullptr )
            {
                //  SE_WIHAHO は途中でサウンドの再生が止まるため、
                //  pParam == nullptr となるタイミングがあります
                EXPECT_EQ( m_SoundId, SE_WIHAHO );
                return ;
            }

            const float pitch = pParam->GetPitch();
            if( !enableSound3DEffect )
            {
                //  Sound3D の機能が無効ならば Pitch == 1
                EXPECT_FLOAT_EQ_LOCAL( pitch, 1.0f );
                return ;
            }

            nn::util::Vector3f vecLisToNow;  //  リスナーからアクターへのベクトル
            nn::util::VectorSubtract( &vecLisToNow, m_SoundActor.GetPosition(), m_pSound3DListener->GetPosition() );
            nn::util::VectorNormalize( &vecLisToNow, vecLisToNow );
            nn::util::Vector3f vecLisToPrv;  //  リスナーから前の位置のアクターへのベクトル
            nn::util::VectorSubtract( &vecLisToPrv, m_prvActorPosition, m_pSound3DListener->GetPosition() );
            nn::util::VectorNormalize( &vecLisToPrv, vecLisToPrv );

            const float dotPrv = nn::util::VectorDot( vecLisToPrv, m_prvActorVelocity );
            const float dotNow = nn::util::VectorDot( vecLisToNow, m_SoundActor.GetVelocity() );

            //  遠ざかっているときと、近づいているときで pitch の大きさが 1 以下か 1 以上かをテストします
            //  ( dotNow > 0 のときは遠ざかっており、dotNow < 0 のときは近づいています )
            EXPECT_EQ( dotNow >= 0.0f, pitch <= 1.0f );
            EXPECT_EQ( dotNow <= 0.0f, pitch >= 1.0f );

            //  前よりも遠ざかっているか、近づいているかで pitch の大きさの相対変化をテストします
            //  ( dotNow > dotPrv のときは、今の方が前よりもより速く遠ざかっています )
            EXPECT_EQ( dotNow >= dotPrv, pitch <= m_prvPitch );
            EXPECT_EQ( dotNow <= dotPrv, pitch >= m_prvPitch );
        }

    private:
        bool IsInInteriorSize(const nn::util::Vector3f& position) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL( m_pSound3DListener );
            const float interiorSize = m_pSound3DListener->GetInteriorSize();
            return nn::util::VectorDistanceSquared( position, m_pSound3DListener->GetPosition() ) <= interiorSize*interiorSize;
        }
        bool IsInMaxVolumeDistance(const nn::util::Vector3f& position) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_NOT_NULL( m_pSound3DListener );
            const float maxVolumeDist = m_pSound3DListener->GetMaxVolumeDistance();
            return nn::util::VectorDistanceSquared( position, m_pSound3DListener->GetPosition() ) <= maxVolumeDist*maxVolumeDist;
        }

    private:
        nn::atk::Sound3DActor m_SoundActor;
        nn::atk::SoundHandle  m_SoundHandle;
        nn::atk::SoundArchive::ItemId m_SoundId;

        const nn::atk::Sound3DManager* m_pSound3DManager;
        const nn::atk::Sound3DListener* m_pSound3DListener;

        nn::util::Vector3f m_prvActorPosition;
        nn::util::Vector3f m_prvActorVelocity;
        float m_prvPan;
        float m_prvSPan;
        float m_prvVolume;
        float m_prvPitch;
    };
}

#if !defined(NN_SDK_BUILD_RELEASE)
TEST( Sound3D, DeathTest )
{
    nnt::atk::util::OnPreAtkTest();
    const float eps = 1.0f / 65536.0f;  //  小さな値

    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    //  Sound3DLister のパラメータの境界テスト
    {
        nn::atk::Sound3DListener sound3DListener;
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetInteriorSize( 0.0f ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetMaxBiquadFilterValue( 1.0f + eps ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetMaxBiquadFilterValue( 0.0f - eps ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetMaxVolumeDistance( 0.0f - eps ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetUnitBiquadFilterValue( 1.0f + eps ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetUnitBiquadFilterValue( 0.0f - eps ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DListener.SetUnitDistance( 0.0f ), "" );
    }

    //  Sound3DActor のパラメータの境界テスト
    {
        nn::atk::Sound3DActor sound3DActor;
        EXPECT_DEATH_IF_SUPPORTED( sound3DActor.SetOutputPan( nn::atk::OutputDevice::OutputDevice_Count, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DActor.SetOutputVolume( nn::atk::OutputDevice::OutputDevice_Count, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DActor.SetPlayableSoundCount( -1, 0 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DActor.SetPlayableSoundCount( 4, 0 ), "" );
    }

    //  Sound3DManaer のパラメータの境界テスト
    {
        nn::atk::Sound3DManager sound3DManager;
        EXPECT_DEATH_IF_SUPPORTED( sound3DManager.SetBiquadFilterType( nn::atk::BiquadFilterType::BiquadFilterType_Min - 1 ), "" );
        EXPECT_DEATH_IF_SUPPORTED( sound3DManager.SetBiquadFilterType( nn::atk::BiquadFilterType::BiquadFilterType_Max + 1 ), "" );
    }

    //  Sound3DLister, Actor, Manager のアラインテスト
    {
        char* buffer;
        size_t size;

        size = sizeof(nn::atk::Sound3DListener) + 1;    //  アラインテストのために少し大きめに確保
        buffer = reinterpret_cast<char*>( g_Allocator.Allocate( size, nn::atk::Sound3DManager::NnUtilMathTypeAlignSize ) );
        EXPECT_DEATH_IF_SUPPORTED( new(buffer + 1) nn::atk::Sound3DListener(), "" );
        g_Allocator.Free( buffer );

        size = sizeof(nn::atk::Sound3DActor) + 1;    //  アラインテストのために少し大きめに確保
        buffer = reinterpret_cast<char*>( g_Allocator.Allocate( size, nn::atk::Sound3DManager::NnUtilMathTypeAlignSize ) );
        EXPECT_DEATH_IF_SUPPORTED( new(buffer + 1) nn::atk::Sound3DActor(), "" );
        g_Allocator.Free( buffer );

        nn::atk::Sound3DManager sound3DManager;
        nn::atk::SoundArchive* pSoundArchive = &g_AtkSetup.GetSoundArchive();
        size = sound3DManager.GetRequiredMemSize( pSoundArchive ) + 1;    //  アラインテストのために少し大きめに確保
        buffer = reinterpret_cast<char*>( g_Allocator.Allocate( size, nn::atk::Sound3DManager::NnUtilMathTypeAlignSize ) );
        EXPECT_DEATH_IF_SUPPORTED( sound3DManager.Initialize( pSoundArchive, buffer + 1, size ), "" );
        g_Allocator.Free( buffer );
    }

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

TEST( Sound3D, InitializeSound3DManagerTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    char commonSoundArchiveFilePath[128];
    NN_ABORT_UNLESS( nnt::atk::util::GetCommonSoundArchivePath( commonSoundArchiveFilePath, sizeof(commonSoundArchiveFilePath), nnt::atk::util::FsCommonSetup::RomMountName ), "Cannot get sound archive path.\n" );

    //  SoundArchive を複数読み込みます
    const int soundArchiveCount = 4;
    nn::atk::FsSoundArchive soundArchive[soundArchiveCount];
    void* pSoundArchiveBuffer[soundArchiveCount];
    for( int i = 0; i < soundArchiveCount; i++ )
    {
        //  TODO: 今は 1 つのサウンドアーカイブを複数回読み込んでいる
        //        これを異なるサウンドアーカイブを読み込むようにする
        EXPECT_TRUE( soundArchive[i].Open( commonSoundArchiveFilePath ) );

        const size_t infoBlockSize = soundArchive[i].GetHeaderSize();
        pSoundArchiveBuffer[i] = nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
        EXPECT_TRUE( soundArchive[i].LoadHeader( pSoundArchiveBuffer[i], infoBlockSize ) );
    }

    //  Sound3DManager の初期化
    nn::atk::Sound3DManager sound3DManager;
    size_t sound3DManagerMemSize = 0;
    for( int i = 0; i < soundArchiveCount; i++ )
    {
        const size_t size = sound3DManager.GetRequiredMemSize( &soundArchive[i] );

        //  要求メモリを確認します。
        //      Sound3DManager の要求メモリは頻繁に増えないため、計算結果をテストで確認します。
        //      要求メモリの計算式を変更する際は、テストの計算式も変更します。
        EXPECT_EQ( size, static_cast<size_t>( GetSoundArchiveHavingSoundCount( soundArchive[i] ) ) * nn::util::align_up( sizeof( nn::atk::Sound3DParam ), nn::atk::Sound3DManager::BufferAlignSize ) );

        if( i < soundArchiveCount - 1 )
        {
            //  1 つ分少ないサイズで用意します
            sound3DManagerMemSize += size;
        }
    }

    void* pSound3DManagerBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator,  sound3DManagerMemSize, nn::atk::Sound3DManager::BufferAlignSize );
    EXPECT_TRUE( sound3DManager.Initialize( &soundArchive[0], pSound3DManagerBuffer, sound3DManagerMemSize ) );

    //  Sound3DManager に複数の SoundArchive を登録する
    for( int i = 1; i < soundArchiveCount - 1; i++ )
    {
        EXPECT_TRUE( sound3DManager.InitializeWithMoreSoundArchive( &soundArchive[i] ) );
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    //  1 つ分少ないメモリを用意しているため、失敗するはず
    EXPECT_DEATH_IF_SUPPORTED( sound3DManager.InitializeWithMoreSoundArchive( &soundArchive[soundArchiveCount - 1] ), "" );
#endif

    sound3DManager.Finalize();
    g_Allocator.Free( pSound3DManagerBuffer );
    for( int i = 0; i < soundArchiveCount; i++ )
    {
        soundArchive[i].Close();
        g_Allocator.Free( pSoundArchiveBuffer[i] );
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( Sound3D, MultiListenerTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );
    LoadData();
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::Sound3DManager& sound3DManager = g_AtkSetup.GetSound3DManager();

    const float ActorRange = 16.0f;  // Actor の移動範囲です
    nn::atk::Sound3DActor actor;
    actor.Initialize( &soundArchivePlayer, &sound3DManager );
    actor.SetPosition( GetVector3f( 0, 0, ActorRange ) );
    actor.SetVelocity( GetVector3f( 0, 0, 0 ) );

    const int ListenerCount = 2;
    nn::atk::Sound3DListener listener[ListenerCount];

    SetListenerMatrix( listener[0], GetVector3f(  0.5f * ActorRange, 0, 0 ), GetVector3f( 0, 0, -1 ) );
    SetListenerMatrix( listener[1], GetVector3f( -0.5f * ActorRange, 0, 0 ), GetVector3f( 0, 0, -1 ) );
    listener[0].SetMaxVolumeDistance( 5.0f );
    listener[1].SetMaxVolumeDistance( 3.2f );
    for( int i = 0; i < ListenerCount; i++ )
    {
        listener[i].SetVelocity( GetVector3f( 0, 0, 0 ) );
        listener[i].SetUnitDistance( 5.0f );
        listener[i].SetInteriorSize( 5.0f );
        sound3DManager.AddListener( &listener[i] );
    }

    const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds( 2400 );
    nn::atk::SoundHandle soundHandle;
    actor.StartSound( &soundHandle, SE_IDLE_DOPPLER );

    //  2 回目以降のループで相対比較によるテストを行うための変数です
    //  初期値は 1 回目のループでテストをパスするための値を設定しています
    float prvVolume = 0.0f;
    float prvMinDistance = 2.0f * ActorRange;

    for( nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 ); elapsed < TimeOut; elapsed += WaitTime )
    {
        const float term = static_cast<float>( elapsed.GetMilliSeconds() ) / TimeOut.GetMilliSeconds();
        const float radian = 2.0f * nn::util::FloatPi * term;
        actor.SetPosition( GetVector3f( std::sin( radian ) * ActorRange, 0, -std::cos( radian * 2.0f ) * ActorRange ) );

        soundArchivePlayer.Update();
#if defined( NNT_ATK_ENABLE_VOICE_COMMAND )
        nn::atk::SoundSystem::VoiceCommandProcess( 6 );
#endif
        nn::os::SleepThread( WaitTime );

        //  マルチリスナーの時は Pan, SPan が変化しないことを確認します
        //  また、ドップラー効果もかからないはずなので、Pitch が変わらないことも確認します
        const auto* pParam = soundHandle.GetAmbientParam();
        EXPECT_NE( pParam, nullptr );
        const auto& tvParam = pParam->GetTvParam();

        EXPECT_FLOAT_EQ_LOCAL( tvParam.GetPan(), 0.0f );
        EXPECT_FLOAT_EQ_LOCAL( tvParam.GetSurroundPan(), 0.0f );
        EXPECT_FLOAT_EQ_LOCAL( pParam->GetPitch(), 1.0f );

        //  最大音量範囲を考慮して、アクターとリスナーの位置の最小距離を求めます
        float minDistance = 2.0f * ActorRange;
        for( int i = 0; i < ListenerCount; i++ )
        {
            const float dist = nn::util::VectorDistance( actor.GetPosition(), listener[i].GetPosition() );

            if( dist <= listener[i].GetMaxVolumeDistance() )
            {
                //  最大音量範囲内にいるため、最小距離を 0 にします
                minDistance = 0.0f;
                break;
            }
            else
            {
                //  最大音量範囲外にいるため、アクターとリスナーの間の距離から
                //  最大音量範囲の長さを引いた値を最小距離として計算します

                if( minDistance > dist - listener[i].GetMaxVolumeDistance() )
                {
                    minDistance = dist - listener[i].GetMaxVolumeDistance();
                }
            }
        }

        const float volume = tvParam.GetVolume();
        if( minDistance > 0.0f )
        {
            //  アクターがすべてのリスナーの最大音量範囲に入っていません

            //  以前よりも遠ざかっていればボリュームが減り、
            //  近づいていればボリュームが増えていることを確認します
            EXPECT_EQ( minDistance <= prvMinDistance, volume >= prvVolume );
            EXPECT_EQ( minDistance >= prvMinDistance, volume <= prvVolume );
        }
        else
        {
            //  アクターがいずれかのリスナーの最大音量範囲以内にいる場合は minDistance が 0.0f になります
            //  そのとき、volume は 1 になるはずです
            EXPECT_FLOAT_EQ_LOCAL( volume, 1.0f );
        }

        prvMinDistance = minDistance;
        prvVolume = volume;
    }

    actor.Finalize();
    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( Sound3D, MultiActorTest )
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );
    LoadData();
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::Sound3DManager& sound3DManager = g_AtkSetup.GetSound3DManager();

    nn::atk::Sound3DListener sound3DListener;
    sound3DListener.SetMaxVolumeDistance( 5.0f );
    sound3DListener.SetUnitDistance( 5.0f );
    sound3DListener.SetInteriorSize( 5.0f );
    sound3DManager.AddListener( &sound3DListener );

    const int ActorCount = 4;
    Sound3DActorTestData actor[ActorCount];
    const nn::atk::SoundArchive::ItemId actorSoundId[] =
    {
        //  それぞれの actor で鳴らすサウンドです
        SE_IDLE_DOPPLER,  //  Pan, SPan, Volume, Pitch が Sound3D の機能によって変化します
        SE_WIHAHO,        //  Pan, SPan, Volume        が Sound3D の機能によって変化します
        SE_IDLE_NOVOLUME, //  Pan, SPan                が Sound3D の機能によって変化します
        SE_IDLE_DOPPLER   //  Pan, SPan, Volume, Pitch が Sound3D の機能によって変化します
    };
    for( int i = 0; i < ActorCount; i++ )
    {
        actor[i].Initialize( soundArchivePlayer, sound3DManager, sound3DListener, actorSoundId[i] );
        actor[i].StartSound();
    }

    //  リスナーの位置, 向きを何回か変更してテストするための設定です
    nn::util::Vector3f ListenerPos[] =
    {
        GetVector3f( -3.0f, 0, -4.0f ),
        GetVector3f( -3.0f, 0,  4.0f ),
        GetVector3f(  2.0f, 0, -1.0f )
    };
    nn::util::Vector3f ListenerDir[] =
    {
        GetVector3f(  0, 0, -1 ),
        GetVector3f(  4, 0, -5 ),
        GetVector3f( -2, 0,  1 )
    };

    for( int k = 0; k < sizeof(ListenerPos) / sizeof(ListenerPos[0]); k++ )
    {
        //  リスナーの位置, 向きを変更します
        //      アクターのテストはリスナーの位置, 向きがテスト中に変わらないことを前提に作られているため
        //      ループの初めに設定を行います
        nn::util::VectorNormalize( &ListenerDir[k], ListenerDir[k] );
        SetListenerMatrix( sound3DListener, ListenerPos[k], ListenerDir[k] );
        sound3DListener.SetVelocity( GetVector3f( 0, 0, 0 ) );

        //  アクターをセットします
        const float Distance = 10.0f;
        const float Speed = 0.25f;
        for( int i = 0; i < ActorCount; i++ )
        {
            actor[i].SetPosition( GetVector3f( ( i + 1 ) * Distance, 0, 0 ) );
            actor[i].SetVelocity( GetVector3f( -Speed, 0, 0 ) );
        }

        //  パラメータを反映させます
        soundArchivePlayer.Update();
#if defined( NNT_ATK_ENABLE_VOICE_COMMAND )
        nn::atk::SoundSystem::VoiceCommandProcess( 6 );
#endif
        nn::os::SleepThread( WaitTime );

        //  アクターを移動させてテストします
        const nn::TimeSpan TimeOut = nn::TimeSpan::FromMilliSeconds(
            static_cast<int64_t>( WaitTime.GetMilliSeconds() * ( ActorCount + 1 ) * Distance / Speed )
        );
        for( nn::TimeSpan elapsed = nn::TimeSpan::FromMilliSeconds( 0 ); elapsed < TimeOut; elapsed += WaitTime )
        {
            //  アクターを更新します
            for( int i = 0; i < ActorCount; i++ )
            {
                actor[i].Update();
            }

            //  actor[3] の速さを 0 にして更新を行います
            //  これにより、ドップラー効果がかからないことをテストします
            const nn::util::Vector3f velocity = actor[3].GetVelocity();
            actor[3].SetVelocity( GetVector3f( 0, 0, 0 ) );

            //  アクターの設定からパラメータを計算します
            soundArchivePlayer.Update();
#if defined( NNT_ATK_ENABLE_VOICE_COMMAND )
            nn::atk::SoundSystem::VoiceCommandProcess( 6 );
#endif
            nn::os::SleepThread( WaitTime );

            //  速さを 0 にしたアクターの速度を元の値に戻します
            actor[3].SetVelocity( velocity );

            //  actor[0] は Pan, SPan, Volume が変化し、ドップラー効果がかかります
            //  ドップラー効果は Pitch の変化によって確認できるため、
            //  Pan, SPan, Volume, Pitch のテストを行います
            actor[0].TestPan( true );
            actor[0].TestSurroundPan( true );
            actor[0].TestVolume( true );
            actor[0].TestPitch( true );

            //  actor[1] は Pan, SPan, Volume が変化し、Pitch は変化しません
            actor[1].TestPan( true );
            actor[1].TestSurroundPan( true );
            actor[1].TestVolume( true );
            actor[1].TestPitch( false );

            //  actor[2] は Pan, SPan が変化し、Volume, Pitch は変化しません
            actor[2].TestPan( true );
            actor[2].TestSurroundPan( true );
            actor[2].TestVolume( false );
            actor[2].TestPitch( false );

            //  actor[3] は Pan, SPan, Volume が変化し、ドップラー効果がかかります
            //  しかし、actor[3] の速さを 0 にして更新しているため、Pitch の値は変わらないはずです
            //  Pitch に変化がないことを確認するために、Sound3D 機能をオフして Pitch のテストを行います
            actor[3].TestPan( true );
            actor[3].TestSurroundPan( true );
            actor[3].TestVolume( true );
            actor[3].TestPitch( false );
        }
    }

    for( int i = 0; i < ActorCount; i++ )
    {
        actor[i].Finalize();
    }
    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
