﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <sstream>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {
    const int MemoryHeapSize = 32 * 1024 * 1024;

    nnt::atk::util::FsCommonSetup g_FsSetup;
    nnt::atk::util::AtkCommonSetup g_AtkSetup;

    static char g_HeapMemory[MemoryHeapSize];
    nn::mem::StandardAllocator  g_Allocator;

    const int NormalSampleRateSet[] = {32000, 48000};
    const int MinimalSampleRateSet[] = {48000};
    const nn::atk::EffectBase::ChannelMode NormalEffectChannelSet[] =
    {
        nn::atk::EffectBase::ChannelMode_1Ch,
        nn::atk::EffectBase::ChannelMode_2Ch,
        nn::atk::EffectBase::ChannelMode_4Ch,
        nn::atk::EffectBase::ChannelMode_6Ch
    };
    const nn::atk::EffectBase::ChannelMode MinimalEffectChannelSet[] =
    {
        nn::atk::EffectBase::ChannelMode_2Ch
    };
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    const std::string TestNameSuffix = "(VoiceCommand)";
#else
    const std::string TestNameSuffix = "";
#endif

    bool g_IsMinimalSampleRateSet = false;
    bool g_IsMinimalEffectChannelSet = false;

    int g_SoundThreadCoreNumber = -1;
    int g_TaskThreadCoreNumber = -1;
    bool g_IsTaskThreadEnabled = true;
    bool g_IsEffectEnabled = true;

    void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer, const nn::TimeSpan& waitTime) NN_NOEXCEPT
    {
        archivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(4);
#endif
        nn::os::SleepThread(waitTime);
    }
    void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
    {
        const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
        UpdateAndWait(archivePlayer, WaitTime);
    }

#if !defined(NN_SDK_BUILD_RELEASE)
    void WaitSoundThreadProcess(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
    {
        uint32_t commandTag = 0;
        archivePlayer.Update();
        while(!nnt::atk::util::SendDummyCommand(&commandTag))
        {
            UpdateAndWait(archivePlayer);
        }

        while(!nnt::atk::util::IsCommandFinished(commandTag))
        {
            UpdateAndWait(archivePlayer);
        }
    }

    void PlayAndUpdateForDeathTest(
        nn::atk::SoundArchivePlayer& archivePlayer,
        nn::atk::SoundHandle& handle,
        nn::atk::SoundArchive::ItemId soundId) NN_NOEXCEPT
    {
        archivePlayer.StartSound(&handle, soundId).IsSuccess();
        WaitSoundThreadProcess(archivePlayer);
    }

    void AppendEffectAndUpdateForDeathTest(
        nn::atk::SoundArchivePlayer& archivePlayer,
        nn::atk::AuxBus bus,
        nn::atk::EffectBase& effect,
        void* effectBuffer,
        size_t effectBufferSize) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::AppendEffect(bus, &effect, effectBuffer, effectBufferSize);
        WaitSoundThreadProcess(archivePlayer);
    }

    void AppendEffectAuxAndUpdateForDeathTest(
        nn::atk::SoundArchivePlayer& archivePlayer,
        nn::atk::AuxBus bus,
        nn::atk::EffectAux& effectAux,
        void* effectAuxBuffer,
        size_t effectAuxBufferSize) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::AppendEffect(bus, &effectAux, effectAuxBuffer, effectAuxBufferSize);
        WaitSoundThreadProcess(archivePlayer);
    }

    void AppendEffectAuxToFinalMixAndUpdateForDeathTest(
        nn::atk::SoundArchivePlayer& archivePlayer,
        nn::atk::EffectAux& effectAux,
        void* effectAuxBuffer,
        size_t effectAuxBufferSize) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::AppendEffectToFinalMix(&effectAux, effectAuxBuffer, effectAuxBufferSize);
        WaitSoundThreadProcess(archivePlayer);
    }

    void ClearEffectAndUpdateForDeathTest(
        nn::atk::SoundArchivePlayer& archivePlayer,
        nn::atk::AuxBus bus) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::ClearEffect(bus);
        WaitSoundThreadProcess(archivePlayer);
    }

    void ClearEffectFromFinalMixAndUpdateForDeathTest(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT
    {
        nn::atk::SoundSystem::ClearEffectFromFinalMix();
        WaitSoundThreadProcess(archivePlayer);
    }
#endif

    class AtkSetupWithUserBuffer : public nnt::atk::util::AtkCommonSetup
    {
    public:
        virtual void InitializeSoundSystem(const nn::atk::SoundSystem::SoundSystemParam param, nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundSystem::SoundSystemParam prm = param;
            prm.enableMemoryPoolManagement = false;

            std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( prm );
            m_pMemoryForSoundSystem = nnt::atk::util::AllocateUninitializedMemory( allocator, memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );

            std::size_t memSizeForSoundSystemMemoryPool = nn::util::align_up( nn::atk::SoundSystem::GetRequiredMemSizeForMemoryPool( prm ), nn::audio::MemoryPoolType::SizeGranularity );
            m_pMemoryForSoundSystemMemoryPool = nnt::atk::util::AllocateUninitializedMemory( allocator, memSizeForSoundSystemMemoryPool, nn::audio::MemoryPoolType::AddressAlignment );

            bool result = nn::atk::SoundSystem::Initialize(
                prm,
                reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystem),
                memSizeForSoundSystem,
                reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystemMemoryPool),
                memSizeForSoundSystemMemoryPool );
            NN_ABORT_UNLESS( result, "cannot initialize SoundSystem" );
            NN_ABORT_UNLESS( nn::atk::SoundSystem::IsInitialized(), "Initialize state is invalid" );

            nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForSoundSystem, m_pMemoryForSoundSystemMemoryPool, memSizeForSoundSystemMemoryPool );

            m_IsInitializedSoundSystem = true;
        }
        virtual void InitializeSoundArchivePlayer(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            nn::atk::SoundArchivePlayer::InitializeParam param;
            param.pSoundArchive = &m_FsSoundArchive;
            param.pSoundDataManager = &m_SoundDataManager;
            param.enablePreparingStreamInstanceBufferFromSetupBuffer = false;

            size_t memSizeForSoundArchivePlayer = m_SoundArchivePlayer.GetRequiredMemSize( param );
            m_pMemoryForSoundArchivePlayer = nnt::atk::util::AllocateUninitializedMemory(allocator, memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
            param.pSetupBuffer = m_pMemoryForSoundArchivePlayer;
            param.setupBufferSize = memSizeForSoundArchivePlayer;

            size_t memSizeForStreamBuffer = m_SoundArchivePlayer.GetRequiredStreamBufferSize(&m_FsSoundArchive);
            size_t alignedMemSizeForStreamBuffer = nn::util::align_up( memSizeForStreamBuffer, nn::audio::MemoryPoolType::SizeGranularity );
            m_pMemoryForStreamBuffer = nnt::atk::util::AllocateUninitializedMemory(allocator, alignedMemSizeForStreamBuffer, nn::audio::MemoryPoolType::AddressAlignment);
            nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForStreamBuffer, m_pMemoryForStreamBuffer, alignedMemSizeForStreamBuffer );
            param.pStreamBuffer = m_pMemoryForStreamBuffer;
            param.streamBufferSize = memSizeForStreamBuffer;

            size_t memSizeForStreamInstance = m_SoundArchivePlayer.GetRequiredStreamInstanceSize(&m_FsSoundArchive);
            size_t alignedMemSizeForStreamInstance = nn::util::align_up( memSizeForStreamInstance, nn::audio::MemoryPoolType::SizeGranularity );
            m_pMemoryForSoundArchivePlayerStreamInstance = nnt::atk::util::AllocateUninitializedMemory(allocator, alignedMemSizeForStreamInstance, nn::audio::MemoryPoolType::AddressAlignment);
            nn::atk::SoundSystem::AttachMemoryPool( &m_MemoryPoolForSoundArchivePlayerStreamInstance, m_pMemoryForSoundArchivePlayerStreamInstance, alignedMemSizeForStreamInstance );
            param.pStreamInstanceBuffer = m_pMemoryForSoundArchivePlayerStreamInstance;
            param.streamInstanceBufferSize = memSizeForStreamInstance;

            NN_ABORT_UNLESS(m_SoundArchivePlayer.Initialize( param ),"cannot initialize SoundArchivePlayer");
        }
        virtual void Finalize(nn::mem::StandardAllocator& allocator) NN_NOEXCEPT NN_OVERRIDE
        {
            FinalizeSoundArchivePlayer(allocator);
            FinalizeSoundDataManager(allocator);
            FinalizeSoundArchive(allocator);
            FinalizeSoundHeap(allocator);

            nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPoolForSoundArchivePlayerStreamInstance );
            nn::atk::SoundSystem::DetachMemoryPool( &m_MemoryPoolForSoundSystem );
            FinalizeSoundSystem(allocator);

            allocator.Free( m_pMemoryForSoundArchivePlayerStreamInstance );
            allocator.Free( m_pMemoryForSoundSystemMemoryPool );
        }

    private:
        void* m_pMemoryForSoundSystemMemoryPool;
        nn::audio::MemoryPoolType m_MemoryPoolForSoundSystem;
        void* m_pMemoryForSoundArchivePlayerStreamInstance;
        nn::audio::MemoryPoolType m_MemoryPoolForSoundArchivePlayerStreamInstance;
    };

    class EffectAuxTest : public nn::atk::EffectAux
    {
    public:
        virtual bool Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG( "EffectAuxTest: Initialize().\n" );
            return true;
        }
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_LOG( "EffectAuxTest: Finalize().\n" );
        }
    protected:
        virtual void UpdateSamples( int32_t* pSamples, const UpdateSamplesArg& arg ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(pSamples);
            NN_UNUSED(arg);
        }
    };

    void TestSequenceTick(int64_t current, int64_t prev)
    {
        // SEQ の Tick はループ後も増加
        EXPECT_GE(current, prev);
    }

    void TestStreamTick(int64_t current, int64_t prev, nn::atk::StreamSoundHandle& streamSoundHandle, bool isLoop)
    {
        // STRM の再生位置チェック
        if (streamSoundHandle.IsPrepared())
        {
            nn::atk::StreamSoundDataInfo streamDataInfo;
            streamSoundHandle.ReadStreamSoundDataInfo(&streamDataInfo);

            if( !isLoop )
            {
                EXPECT_GE(current, prev);
            }
            else
            {
                EXPECT_LT(current, prev);
            }
        }
        else
        {
            // リスタート直後のサウンドハンドルの切断が行われ、 -1 の値が返ってくるときはチェックしない
            if (current >= 0)
            {
                // Prepare 完了前は ReadStreamSoundDataInfo 関数を呼び出すことによる警告を避けたうえで、
                // 再生位置が不正になっていないかをチェックする
                EXPECT_GE(current, prev);
            }
        }
    }

    void TestWsdTick(int64_t current, int64_t prev)
    {
        // 再生が完了して 0 になっている場合があるのでそれを省く
        if (current != 0)
        {
            EXPECT_GE(current, prev);
        }
    }
}

void LoadData()
{
    g_AtkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
    g_AtkSetup.LoadData(SE_YOSHI, "SE_YOSHI");
}

void WaitForEffectClear(nn::atk::EffectBase& effect)
{
    const auto WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    const auto TimeOut = nn::TimeSpan::FromMilliSeconds(5000);

    for(auto elapsed = nn::TimeSpan::FromMilliSeconds(0); elapsed < TimeOut; elapsed += WaitTime)
    {
        nn::os::SleepThread(WaitTime);
        if(effect.IsRemovable()) break;
    }
    EXPECT_TRUE(effect.IsRemovable()) << "Effect clear timed out.";
}

void ParseUserArgument(int argc, char** argv)
{
    NN_ABORT_UNLESS_NOT_NULL(argv);
    for(int i = 0; i < argc; i++)
    {
        NN_ABORT_UNLESS_NOT_NULL(argv[i]);
        if(strncmp(argv[i], "sampleRateSet=minimal", sizeof("sampleRateSet=minimal")) == 0)
        {
            g_IsMinimalSampleRateSet = true;
        }
        if(strncmp(argv[i], "effectChannelSet=minimal", sizeof("effectChannelSet=minimal")) == 0)
        {
            g_IsMinimalEffectChannelSet = true;
        }

        auto arg = std::string(argv[i]);
        auto pos = arg.find("=");
        if (pos != std::string::npos)
        {
            auto parameter = arg.substr(0, pos);
            if (parameter == "soundThreadCoreNumber")
            {
                g_SoundThreadCoreNumber = std::stoi(arg.substr(pos + 1));
            }
            if (parameter == "taskThreadCoreNumber")
            {
                g_TaskThreadCoreNumber = std::stoi(arg.substr(pos + 1));
            }
            if (parameter == "isTaskThreadEnabled")
            {
                std::istringstream iss(arg.substr(pos + 1));
                iss >> std::boolalpha >> g_IsTaskThreadEnabled;
            }
            if (parameter == "isEffectEnabled")
            {
                std::istringstream iss(arg.substr(pos + 1));
                iss >> std::boolalpha >> g_IsEffectEnabled;
            }
        }
    }
}

nn::atk::EffectBase::SampleRate ConvertIntToEffectSampleRate(int sampleRate)
{
    switch(sampleRate)
    {
    case 32000:
        return nn::atk::EffectReverb::SampleRate_32000;
    case 48000:
        return nn::atk::EffectReverb::SampleRate_48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

TEST(PlaySound, PlayTwoSoundsShortly)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    g_AtkSetup.Initialize(g_Allocator);
    LoadData();

    const unsigned int TestData[] = {SEQ_MARIOKART, SE_YOSHI, STRM_MARIOKART};
    const char* TestDataName[] ={"SEQ_MARIOKART", "SE_YOSHI", "STRM_MARIOKART"};

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundHandle holdSoundHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayTwoSoundsShortly(VoiceCommand)";
#else
    char TestName[] = "PlayTwoSoundsShortly";
#endif

    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(20);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    // テスト本体
    for(int i = 0; i < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); i++)
    {
        for(int j = 0; j < static_cast<int>(sizeof(TestData) / sizeof(TestData[0])); j++)
        {
            soundHandle.Stop(0);
            holdSoundHandle.Stop(0);

            nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

            EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
            NN_LOG("[%s] Start %s.\n", TestName, TestDataName[i]);
            bool isHoldSoundStarted = false;
            bool isVoiceCountChanged = false;

            for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
            {
                if(!isVoiceCountChanged)
                {
                    isVoiceCountChanged = (nn::atk::SoundSystem::GetVoiceCount() > 0);
                }

                // 10 秒後から HoldSound を開始する
                if(elapsed >= nn::TimeSpan::FromSeconds(10))
                {
                    if(!isHoldSoundStarted)
                    {
                        // HoldSound が始まるまでに StartSound によりボイス数が増加していることを確認する
                        EXPECT_TRUE(isVoiceCountChanged);

                        NN_LOG("[%s] Hold %s.\n", TestName, TestDataName[j]);
                        isHoldSoundStarted = true;
                    }
                    EXPECT_TRUE(soundArchivePlayer.HoldSound(&holdSoundHandle, TestData[j]).IsSuccess());
                }
                soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
                // soundHandle の音源は最後まで回したら再度再生
                if(!soundHandle.IsAttachedSound())
                {
                    soundHandle.Stop(0);
                    EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, TestData[i]).IsSuccess());
                }
                nn::os::SleepThread(WaitTime);
            }

            NN_LOG("[%s] Stop %s.\n", TestName, TestDataName[i]);
            NN_LOG("[%s] Stop %s.\n", TestName, TestDataName[j]);
        }
    }

    soundHandle.Stop(0);
    holdSoundHandle.Stop(0);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlayThreeSoundsShortlyWithEffect)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = (g_IsMinimalSampleRateSet || g_IsMinimalEffectChannelSet) ? false : true; // 最小限構成の場合はマージ条件のテストのため、排他チェックを無効にする
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);

    const int* pTestSampleRate;
    int sampleRateArraySize = 0;

    const nn::atk::EffectBase::ChannelMode* pTestEffectChannel;
    int effectChannelArraySize = 0;

    // 最小限で回す場合と、普段の場合に応じてテストするサンプルレートを変更する
    if(g_IsMinimalSampleRateSet)
    {
        pTestSampleRate = MinimalSampleRateSet;
        sampleRateArraySize = static_cast<int>(sizeof(MinimalSampleRateSet) / sizeof(MinimalSampleRateSet[0]));
    }
    else
    {
        pTestSampleRate = NormalSampleRateSet;
        sampleRateArraySize = static_cast<int>(sizeof(NormalSampleRateSet) / sizeof(NormalSampleRateSet[0]));
    }
    // サンプルレートと同様、最小限と普段の場合とでテストするエフェクトのチャンネル数を変更する
    if(g_IsMinimalEffectChannelSet)
    {
        pTestEffectChannel = MinimalEffectChannelSet;
        effectChannelArraySize = static_cast<int>(sizeof(MinimalEffectChannelSet) / sizeof(MinimalEffectChannelSet[0]));
    }
    else
    {
        pTestEffectChannel = NormalEffectChannelSet;
        effectChannelArraySize = static_cast<int>(sizeof(NormalEffectChannelSet) / sizeof(NormalEffectChannelSet[0]));
    }

    for (int effectChannelArrayIndex = 0; effectChannelArrayIndex < effectChannelArraySize; effectChannelArrayIndex++)
    {
        for (int sampleRateArrayIndex = 0 ; sampleRateArrayIndex < sampleRateArraySize; sampleRateArrayIndex++)
        {
            g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
            g_FsSetup.Initialize();

            nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
            soundSystemParam.rendererSampleRate = pTestSampleRate[sampleRateArrayIndex];
            if (g_SoundThreadCoreNumber >= 0)
            {
                soundSystemParam.soundThreadCoreNumber = g_SoundThreadCoreNumber;
            }
            if (g_TaskThreadCoreNumber >= 0)
            {
                soundSystemParam.taskThreadCoreNumber = g_TaskThreadCoreNumber;
            }
            soundSystemParam.enableTaskThread = g_IsTaskThreadEnabled;
            nnt::atk::util::AtkCommonSetup::InitializeParam param;
            param.SetSoundSystemParam(soundSystemParam);
            g_AtkSetup.Initialize(param, g_Allocator);
            LoadData();

#if defined( NN_ATK_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION )
            //  指定したコア番号でスレッドが動いていることを確認します
            if (g_SoundThreadCoreNumber >= 0)
            {
                EXPECT_EQ( nn::atk::SoundSystem::GetSoundThreadCoreNumber(), g_SoundThreadCoreNumber );
            }
            if (g_TaskThreadCoreNumber >= 0)
            {
                EXPECT_EQ( nn::atk::SoundSystem::GetTaskThreadCoreNumber(), g_TaskThreadCoreNumber );
            }
#endif

            nn::atk::SoundHandle seqMarioKartHandle;
            nn::atk::SoundHandle seYoshiHandle;
            nn::atk::SoundHandle strmMarioKartHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
            char TestName[] = "PlayThreeSoundsShortly(VoiceCommand)";
#else
            char TestName[] = "PlayThreeSoundsShortly";
#endif
            nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

            // reverb の設定
            nn::atk::EffectReverb reverb;
            bool isReverbChannelSupported;

            isReverbChannelSupported = reverb.SetChannelMode(pTestEffectChannel[effectChannelArrayIndex]);
            reverb.SetSampleRate(ConvertIntToEffectSampleRate(pTestSampleRate[sampleRateArrayIndex]));

            size_t reverbBufferSize = reverb.GetRequiredMemSize();
            size_t alignedReverbBufferSize = nn::util::align_up(reverbBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
            void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alignedReverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);

            nn::audio::MemoryPoolType reverbMemoryPool;
            if(isReverbChannelSupported)
            {
                nn::atk::SoundSystem::AttachMemoryPool(&reverbMemoryPool, reverbBuffer, alignedReverbBufferSize);
                reverb.SetEnabled(true);
                EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &reverb, reverbBuffer, reverbBufferSize));
            }

            // I3DL2 Reverb の設定
            nn::atk::EffectI3dl2Reverb i3dl2Reverb;
            bool isI3dl2ReverbChannelSupported;

            isI3dl2ReverbChannelSupported = i3dl2Reverb.SetChannelMode(pTestEffectChannel[effectChannelArrayIndex]);
            i3dl2Reverb.SetSampleRate(ConvertIntToEffectSampleRate(pTestSampleRate[sampleRateArrayIndex]));

            size_t i3dl2ReverbBufferSize = i3dl2Reverb.GetRequiredMemSize();
            size_t alignedI3dl2ReverbBufferSize = nn::util::align_up(i3dl2ReverbBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
            void* i3dl2ReverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alignedI3dl2ReverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS_NOT_NULL(i3dl2ReverbBuffer);

            nn::audio::MemoryPoolType i3dl2ReverbMemoryPool;
            if (isI3dl2ReverbChannelSupported)
            {
                nn::atk::SoundSystem::AttachMemoryPool(&i3dl2ReverbMemoryPool, i3dl2ReverbBuffer, alignedI3dl2ReverbBufferSize);
                i3dl2Reverb.SetEnabled(true);
                EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &i3dl2Reverb, i3dl2ReverbBuffer, i3dl2ReverbBufferSize));
            }

            // delay の設定
            nn::atk::EffectDelay delay;
            bool isDelayChannelSupported;
            isDelayChannelSupported = delay.SetChannelMode(pTestEffectChannel[effectChannelArrayIndex]);
            delay.SetSampleRate(ConvertIntToEffectSampleRate(pTestSampleRate[sampleRateArrayIndex]));

            size_t delayBufferSize = delay.GetRequiredMemSize();
            size_t alignedDelayBufferSize = nn::util::align_up(delayBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
            void* delayBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alignedDelayBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
            NN_ABORT_UNLESS_NOT_NULL(delayBuffer);

            nn::audio::MemoryPoolType delayMemoryPool;
            if(isDelayChannelSupported)
            {
                nn::atk::SoundSystem::AttachMemoryPool(&delayMemoryPool, delayBuffer, alignedDelayBufferSize);
                delay.SetEnabled(true);
                EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &delay, delayBuffer, delayBufferSize));
            }

            // 音源の再生
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
            seqMarioKartHandle.SetMainSend(-0.2f);
            seqMarioKartHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
            NN_LOG("[%s] Start SEQ_MARIOKART.\n", TestName);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
            seYoshiHandle.SetMainSend(-0.2f);
            seYoshiHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
            NN_LOG("[%s] Start SE_YOSHI.\n", TestName);
            if (g_IsTaskThreadEnabled)
            {
                EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
                strmMarioKartHandle.SetMainSend(-0.2f);
                strmMarioKartHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
                NN_LOG("[%s] Start STRM_MARIOKART.\n", TestName);
            }

            const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(10);
            const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
            int seYoshiCount = 0;

            for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
            {
                soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
                nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
                // SE_YOSHI は最後まで回したら再度再生
                if(!seYoshiHandle.IsAttachedSound())
                {
                    seYoshiHandle.Stop(0);
                    EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
                    seYoshiHandle.SetMainSend(-0.2f);
                    seYoshiHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
                    seYoshiCount++;

                    NN_LOG("[%s] Start SE_YOSHI %d Times.\n", TestName, seYoshiCount);
                }

                nn::os::SleepThread(WaitTime);
            }

            seqMarioKartHandle.Stop(0);
            seYoshiHandle.Stop(0);
            if (g_IsTaskThreadEnabled)
            {
                strmMarioKartHandle.Stop(0);
            }
            soundArchivePlayer.Update();

            if(isReverbChannelSupported)
            {
                reverb.SetEnabled(false);
                WaitForEffectClear(reverb);
            }

            if (isI3dl2ReverbChannelSupported)
            {
                i3dl2Reverb.SetEnabled(false);
                WaitForEffectClear(i3dl2Reverb);
            }

            if(isDelayChannelSupported)
            {
                delay.SetEnabled(false);
                WaitForEffectClear(delay);
            }

            nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
            if(isReverbChannelSupported)
            {
                nn::atk::SoundSystem::DetachMemoryPool(&reverbMemoryPool);
            }
            if (isI3dl2ReverbChannelSupported)
            {
                nn::atk::SoundSystem::DetachMemoryPool(&i3dl2ReverbMemoryPool);
            }
            if(isDelayChannelSupported)
            {
                nn::atk::SoundSystem::DetachMemoryPool(&delayMemoryPool);
            }

            g_AtkSetup.Finalize(g_Allocator);
            g_FsSetup.Finalize();
            g_Allocator.Finalize();
        }
    }
} // NOLINT(impl/function_size)

TEST(PlaySound, PlayThreeSoundsShortlyWithPause)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableEffect = g_IsEffectEnabled;
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.SetSoundSystemParam(soundSystemParam);

    g_AtkSetup.Initialize(param, g_Allocator);
    LoadData();

    nn::atk::SoundHandle seqMarioKartHandle;
    nn::atk::SoundHandle seYoshiHandle;
    nn::atk::SoundHandle strmMarioKartHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayThreeSounds(VoiceCommand)";
#else
    char TestName[] = "PlayThreeSounds";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start SEQ_MARIOKART.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
    NN_LOG("[%s] Start SE_YOSHI.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start STRM_MARIOKART.\n", TestName);

    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(15);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    bool isPaused = false;
    const int PauseTimes = 2;
    int pauseCounter = 0;
    int seYoshiCount = 0;

    int64_t prevSeqMarioKartPosition = 0;
    int64_t prevSeYoshiPosition = 0;
    int64_t prevStrmMarioKartPosition = 0;
    int prevStrmMarioKartLoopCount = 0;

    for (nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
    {
        nn::atk::WaveSoundHandle waveSoundHandle(&seYoshiHandle);
        nn::atk::SequenceSoundHandle sequenceSoundHandle(&seqMarioKartHandle);
        nn::atk::StreamSoundHandle streamSoundHandle(&strmMarioKartHandle);
        nn::atk::StreamSoundDataInfo streamDataInfo;

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif

        // ポーズ状態への変更
        if (pauseCounter++ % PauseTimes == 0)
        {
            streamSoundHandle.Pause(!isPaused, 0);
            sequenceSoundHandle.Pause(!isPaused, 0);
            waveSoundHandle.Pause(!isPaused, 0);
            isPaused = !isPaused;
        }

        // SE_YOSHI は最後まで回したら再度再生
        if (!seYoshiHandle.IsAttachedSound())
        {
            seYoshiHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
            seYoshiHandle.SetMainSend(-0.2f);
            seYoshiHandle.SetEffectSend(nn::atk::AuxBus_A, 0.8f);
            seYoshiCount++;
            prevSeYoshiPosition = 0;
        }

        // 再生位置に関するチェック
        //（チェックが完了する前に再生位置が更新されるのを防ぐため一時変数に保存）
        int64_t currentSeYoshiPosition = waveSoundHandle.GetPlaySamplePosition();
        int64_t currentSeqMarioKartPosition = sequenceSoundHandle.GetTick();
        int64_t currentStrmMarioKartPosition = streamSoundHandle.GetPlaySamplePosition();
        int currentStrmMarioKartLoopCount = streamSoundHandle.GetPlayLoopCount();
        // SEQ の再生位置チェック
        EXPECT_GE(currentSeqMarioKartPosition, prevSeqMarioKartPosition);
        // STRM の再生位置チェック
        if (streamSoundHandle.IsPrepared())
        {
            streamSoundHandle.ReadStreamSoundDataInfo(&streamDataInfo);
            const int InvalidLoopCount = -1;
            // ループが想定される箇所以外でストリームサウンドの再生位置チェック
            if (currentStrmMarioKartLoopCount != prevStrmMarioKartLoopCount
                && currentStrmMarioKartLoopCount != InvalidLoopCount && prevStrmMarioKartLoopCount != InvalidLoopCount)
            {
                EXPECT_GE(currentStrmMarioKartPosition, prevStrmMarioKartPosition);
            }
        }
        // WSD の再生位置チェック (再生が完了して 0 になっている場合があるのでそれを省く)
        if (currentSeYoshiPosition != 0)
        {
            EXPECT_GE(currentSeYoshiPosition, prevSeYoshiPosition);
        }

        prevSeqMarioKartPosition = currentSeqMarioKartPosition;
        prevSeYoshiPosition = currentSeYoshiPosition;
        prevStrmMarioKartPosition = currentStrmMarioKartPosition;
        prevStrmMarioKartLoopCount = currentStrmMarioKartLoopCount;

        nn::os::SleepThread(WaitTime);
    }

    seqMarioKartHandle.Stop(0);
    seYoshiHandle.Stop(0);
    strmMarioKartHandle.Stop(0);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlayThreeSounds)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableEffect = g_IsEffectEnabled;
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.SetSoundSystemParam(soundSystemParam);

    g_AtkSetup.Initialize(param, g_Allocator);
    LoadData();

    nn::atk::SoundHandle seqMarioKartHandle;
    nn::atk::SoundHandle seYoshiHandle;
    nn::atk::SoundHandle strmMarioKartHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayThreeSounds(VoiceCommand)";
#else
    char TestName[] = "PlayThreeSounds";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start SEQ_MARIOKART.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
    NN_LOG("[%s] Start SE_YOSHI.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start STRM_MARIOKART.\n", TestName);

    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(6 * 60);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    int seYoshiCount = 0;

    int64_t prevSeqMarioKartPosition = 0;
    int64_t prevSeYoshiPosition = 0;
    int64_t prevStrmMarioKartPosition = 0;
    int prevStrmMarioKartLoopCount = 0;

    for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
    {
        nn::atk::WaveSoundHandle waveSoundHandle(&seYoshiHandle);
        nn::atk::SequenceSoundHandle sequenceSoundHandle(&seqMarioKartHandle);
        nn::atk::StreamSoundHandle streamSoundHandle(&strmMarioKartHandle);

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        // SEQ_MARIOKART が 2 周分再生したことが確認できたら再度再生 (SEQ_MARIOKART はタイムベース 96)
        if(sequenceSoundHandle.GetTick() >= (36 + 34) * 96 * 4)
        {
            seqMarioKartHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
            prevSeqMarioKartPosition = 0;
            NN_LOG("[%s] Restart SEQ_MARIOKART.\n", TestName);
        }
        // SE_YOSHI は最後まで回したら再度再生
        if(!seYoshiHandle.IsAttachedSound())
        {
            seYoshiHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
            seYoshiCount++;
            prevSeYoshiPosition = 0;

            // 50 回再生されるごとにログ表示
            if(seYoshiCount % 50 == 0)
            {
                NN_LOG("[%s] Start SE_YOSHI %d Times.\n", TestName, seYoshiCount);
            }
        }
        // STRM_MARIOKART はループ点に 2 回到達したことが確認できたら再度再生
        if(streamSoundHandle.GetPlayLoopCount() == 2)
        {
            strmMarioKartHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
            prevStrmMarioKartPosition = 0;
            NN_LOG("[%s] Restart STRM_MARIOKART.\n", TestName);
        }

        // 再生位置に関するチェック
        //（チェックが完了する前に再生位置が更新されるのを防ぐため一時変数に保存）
        int64_t currentSeYoshiPosition = waveSoundHandle.GetPlaySamplePosition();
        int64_t currentSeqMarioKartPosition = sequenceSoundHandle.GetTick();
        int64_t currentStrmMarioKartPosition = streamSoundHandle.GetPlaySamplePosition();
        int currentStrmMarioKartLoopCount = streamSoundHandle.GetPlayLoopCount();

        const int InvalidLoopCount = -1;
        bool isLoop = (currentStrmMarioKartLoopCount != prevStrmMarioKartLoopCount)
                   && (currentStrmMarioKartLoopCount != InvalidLoopCount) && (prevStrmMarioKartLoopCount != InvalidLoopCount);
        if(isLoop)
        {
            // ループの再生位置であることを確定させるために、再度再生位置を取得する。
            currentStrmMarioKartPosition = streamSoundHandle.GetPlaySamplePosition();
        }

        TestSequenceTick(currentSeqMarioKartPosition, prevSeqMarioKartPosition);
        TestStreamTick(currentStrmMarioKartPosition, prevStrmMarioKartPosition, streamSoundHandle, isLoop);
        TestWsdTick(currentSeYoshiPosition, prevSeYoshiPosition);

        prevSeqMarioKartPosition = currentSeqMarioKartPosition;
        prevSeYoshiPosition = currentSeYoshiPosition;
        prevStrmMarioKartLoopCount = currentStrmMarioKartLoopCount;
        prevStrmMarioKartPosition = currentStrmMarioKartPosition;

        nn::os::SleepThread(WaitTime);
    }

    seqMarioKartHandle.Stop(0);
    seYoshiHandle.Stop(0);
    strmMarioKartHandle.Stop(0);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlayStreamWithFewVoiceCount)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.voiceCountMax = 1;
    initializeParam.SetSoundSystemParam(soundSystemParam);

    g_AtkSetup.Initialize(initializeParam, g_Allocator);
    LoadData();

    // テスト開始
    nn::atk::SoundHandle strmMarioKartHandle;
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayStreamWithFewVoiceCount(VoiceCommand)";
#else
    char TestName[] = "PlayStreamWithFewVoiceCount";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start STRM_MARIOKART.\n", TestName);

    const nn::TimeSpan LoopTime = nn::TimeSpan::FromMilliSeconds(100);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
    {
        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        nn::os::SleepThread(WaitTime);
    }

    // 無事停止しなければOK

    strmMarioKartHandle.Stop(0);

    // 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlayWithPlayerHeap)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundHandle waveSoundHandle;
    nn::atk::SoundHandle sequenceSoundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    const std::string TestName = "PlayWithPlayerHeap";
    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(20);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);

    int waveSoundPlayCount = 0;
    int sequenceSoundPlayCount = 0;
    for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
    {
        if (!waveSoundHandle.IsAttachedSound())
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&waveSoundHandle, SE_YOSHI_PLAYERHEAP).IsSuccess());
            waveSoundPlayCount++;
            NN_LOG("[%s] Start %s %d.\n", (TestName + TestNameSuffix).c_str(), "SE_YOSHI_PLAYERHEAP", waveSoundPlayCount);
        }
        if (!sequenceSoundHandle.IsAttachedSound())
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&sequenceSoundHandle, SEQ_MARIOKART_PLAYERHEAP).IsSuccess());
            sequenceSoundPlayCount++;
            NN_LOG("[%s] Start %s %d.\n", (TestName + TestNameSuffix).c_str(), "SEQ_MARIOKART_PLAYERHEAP", sequenceSoundPlayCount);
        }
        UpdateAndWait(soundArchivePlayer);
    }

    waveSoundHandle.Stop(0);
    sequenceSoundHandle.Stop(0);

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( PlaySound, CanPlayWsdOnPlayerHeap )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    const std::string TestName = "CanPlayWsdOnPlayerHeap";
    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds( 1 );
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds( 16 );

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    int voiceCount = 0;
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI_PLAYERHEAP ).IsSuccess() );
    NN_LOG( "[%s] Start %s.\n", (TestName + TestNameSuffix).c_str(), "SE_YOSHI_PLAYERHEAP" );
    for ( nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime )
    {
        int currentVoiceCount = nn::atk::SoundSystem::GetVoiceCount();
        if ( voiceCount < currentVoiceCount )
        {
            voiceCount = currentVoiceCount;
        }
        UpdateAndWait( soundArchivePlayer );
    }
    soundHandle.Stop( 0 );
    EXPECT_GT( voiceCount, 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( PlaySound, CanPlaySeqOnPlayerHeap )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    const std::string TestName = "CanPlaySeqOnPlayerHeap";
    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds( 1 );
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds( 16 );

    nn::atk::SoundHandle soundHandle;
    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    int voiceCount = 0;
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART_PLAYERHEAP ).IsSuccess() );
    NN_LOG( "[%s] Start %s.\n", (TestName + TestNameSuffix).c_str(), "SEQ_MARIOKART_PLAYERHEAP" );
    for ( nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime )
    {
        int currentVoiceCount = nn::atk::SoundSystem::GetVoiceCount();
        if ( voiceCount < currentVoiceCount )
        {
            voiceCount = currentVoiceCount;
        }
        UpdateAndWait( soundArchivePlayer );
    }
    soundHandle.Stop( 0 );
    EXPECT_GT( voiceCount, 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST( PlaySound, PlayWithNoSubMixMode )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableSubMix = false;
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.SetSoundSystemParam( soundSystemParam );

    g_AtkSetup.Initialize( param, g_Allocator );
    LoadData();

    nn::atk::SoundHandle soundHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayWithNoSubMixMode(VoiceCommand)";
#else
    char TestName[] = "PlayWithNoSubMixMode";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start SEQ_MARIOKART.\n", TestName );
    soundHandle.Stop(0);
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess() );
    NN_LOG( "[%s] Start SE_YOSHI.\n", TestName );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start STRM_MARIOKART.\n", TestName );
    soundHandle.Stop( 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST( PlaySound, PlayWithStereoMode )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableStereoMode = true;
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.SetSoundSystemParam( soundSystemParam );

    g_AtkSetup.Initialize( param, g_Allocator );
    LoadData();

    nn::atk::SoundHandle soundHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayWithStereoMode(VoiceCommand)";
#else
    char TestName[] = "PlayWithStereoMode";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start SEQ_MARIOKART.\n", TestName );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess() );
    NN_LOG( "[%s] Start SE_YOSHI.\n", TestName );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start STRM_MARIOKART.\n", TestName );
    soundHandle.Stop( 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST( PlaySound, PlayWithStereoAndNoSubmixMode )
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize( g_HeapMemory, MemoryHeapSize );
    g_FsSetup.Initialize();

    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableStereoMode = true;
    soundSystemParam.enableSubMix = false;
    nnt::atk::util::AtkCommonSetup::InitializeParam param;
    param.SetSoundSystemParam( soundSystemParam );

    g_AtkSetup.Initialize( param, g_Allocator );
    LoadData();

    nn::atk::SoundHandle soundHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayWithStereoAndNoSubmixMode(VoiceCommand)";
#else
    char TestName[] = "PlayWithStereoAndNoSubmixMode";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SEQ_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start SEQ_MARIOKART.\n", TestName );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, SE_YOSHI ).IsSuccess() );
    NN_LOG( "[%s] Start SE_YOSHI.\n", TestName );
    soundHandle.Stop( 0 );
    EXPECT_TRUE( soundArchivePlayer.StartSound( &soundHandle, STRM_MARIOKART ).IsSuccess() );
    NN_LOG( "[%s] Start STRM_MARIOKART.\n", TestName );
    soundHandle.Stop( 0 );

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlayThreeSoundsWithUserBuffer)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    AtkSetupWithUserBuffer atkSetup;
    atkSetup.Initialize( g_Allocator );
    atkSetup.LoadData(SEQ_MARIOKART, "SEQ_MARIOKART");
    atkSetup.LoadData(SE_YOSHI, "SE_YOSHI");

    nn::atk::SoundHandle seqMarioKartHandle;
    nn::atk::SoundHandle seYoshiHandle;
    nn::atk::SoundHandle strmMarioKartHandle;

#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
    char TestName[] = "PlayThreeSoundsWithUserBuffer(VoiceCommand)";
#else
    char TestName[] = "PlayThreeSoundsWithUserBuffer";
#endif

    nn::atk::SoundArchivePlayer& soundArchivePlayer = atkSetup.GetSoundArchivePlayer();

    EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start SEQ_MARIOKART.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
    NN_LOG("[%s] Start SE_YOSHI.\n", TestName);
    EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
    NN_LOG("[%s] Start STRM_MARIOKART.\n", TestName);

    const nn::TimeSpan LoopTime = nn::TimeSpan::FromSeconds(10);
    const nn::TimeSpan WaitTime = nn::TimeSpan::FromMilliSeconds(16);
    int seYoshiCount = 0;

    for(nn::TimeSpan elapsed = 0; elapsed <= LoopTime; elapsed += WaitTime)
    {
        nn::atk::SequenceSoundHandle sequenceSoundHandle(&seqMarioKartHandle);
        nn::atk::StreamSoundHandle streamSoundHandle(&strmMarioKartHandle);

        soundArchivePlayer.Update();
#if defined(NNT_ATK_ENABLE_VOICE_COMMAND)
        nn::atk::SoundSystem::VoiceCommandProcess(6);
#endif
        // SEQ_MARIOKART が 2 周分再生したことが確認できたら再度再生 (SEQ_MARIOKART はタイムベース 96)
        if(sequenceSoundHandle.GetTick() >= (36 + 34) * 96 * 4)
        {
            seqMarioKartHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seqMarioKartHandle, SEQ_MARIOKART).IsSuccess());
            NN_LOG("[%s] Restart SEQ_MARIOKART.\n", TestName);
        }
        // SE_YOSHI は最後まで回したら再度再生
        if(!seYoshiHandle.IsAttachedSound())
        {
            seYoshiHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&seYoshiHandle, SE_YOSHI).IsSuccess());
            seYoshiCount++;

            // 50 回再生されるごとにログ表示
            if(seYoshiCount % 50 == 0)
            {
                NN_LOG("[%s] Start SE_YOSHI %d Times.\n", TestName, seYoshiCount);
            }
        }
        // STRM_MARIOKART はループ点に 2 回到達したことが確認できたら再度再生
        if(streamSoundHandle.GetPlayLoopCount() == 2)
        {
            strmMarioKartHandle.Stop(0);
            EXPECT_TRUE(soundArchivePlayer.StartSound(&strmMarioKartHandle, STRM_MARIOKART).IsSuccess());
            NN_LOG("[%s] Restart STRM_MARIOKART.\n", TestName);
        }

        nn::os::SleepThread(WaitTime);
    }

    seqMarioKartHandle.Stop(0);
    seYoshiHandle.Stop(0);
    strmMarioKartHandle.Stop(0);

    atkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();

    g_Allocator.Finalize();
}

TEST(PlaySound, PlaySinglePlaySound)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    g_AtkSetup.LoadData(SE_SIN440_LOOP_NEWEST, "SE_SIN440_LOOP_NEWEST");
    g_AtkSetup.LoadData(SE_SIN440_LOOP_NEWEST_DURATION, "SE_SIN440_LOOP_NEWEST_DURATION");
    g_AtkSetup.LoadData(SE_SIN440_LOOP_OLDEST, "SE_SIN440_LOOP_OLDEST");
    g_AtkSetup.LoadData(SE_SIN440_LOOP_OLDEST_DURATION, "SE_SIN440_LOOP_OLDEST_DURATION");

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle, nextHandle;

    const nn::atk::SoundArchive::ItemId TestSoundId[] = {
        SE_SIN440_LOOP_NEWEST,
        SE_SIN440_LOOP_NEWEST_DURATION,
        SE_SIN440_LOOP_OLDEST,
        SE_SIN440_LOOP_OLDEST_DURATION,
    };
    const nn::atk::SinglePlayType TestPlayType[] =
    {
        nn::atk::SinglePlayType_PrioritizeNewest,
        nn::atk::SinglePlayType_PrioritizeNewestWithDuration,
        nn::atk::SinglePlayType_PrioritizeOldest,
        nn::atk::SinglePlayType_PrioritizeOldestWithDuration,
    };

    //  SE_SIN440_LOOP_NEWEST_DURATION, SE_SIN440_LOOP_OLDEST_DURATION の効果時間は
    //  1000 ms に設定されているため、その半分の 500 を設定します
    const nn::TimeSpan halfDuration = nn::TimeSpan::FromMilliSeconds(500);

    for(int i = 0; i < sizeof(TestSoundId) / sizeof(TestSoundId[0]); i++)
    {
        const auto soundId = TestSoundId[i];
        const auto playType = TestPlayType[i];
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, soundId).IsSuccess());

        // 効果時間指定の半分だけ待ちます
        UpdateAndWait(soundArchivePlayer, halfDuration);

        if(playType == nn::atk::SinglePlayType_PrioritizeNewest || playType == nn::atk::SinglePlayType_PrioritizeNewestWithDuration)
        {
            // 後着優先の場合、nextHandle の再生に成功し、soundHandle は停止します
            EXPECT_TRUE(soundArchivePlayer.StartSound(&nextHandle, soundId).IsSuccess());
            EXPECT_FALSE(soundHandle.IsAttachedSound());
        }
        else
        {
            // 先着優先の場合、nextHandle の再生に失敗し、soundHandle は再生し続けています
            EXPECT_FALSE(soundArchivePlayer.StartSound(&nextHandle, soundId).IsSuccess());
            EXPECT_TRUE(soundHandle.IsAttachedSound());
        }

        soundHandle.Stop(0);
        nextHandle.Stop(0);
        EXPECT_TRUE(soundArchivePlayer.StartSound(&soundHandle, soundId).IsSuccess());

        // 効果時間指定の 1.5 倍待ちます (TimeSpan には operator + のみ定義されていたため半分を 3 つ足します)
        UpdateAndWait(soundArchivePlayer, halfDuration + halfDuration + halfDuration);

        // 時間指定のない先着優先のみ nextHandle の再生に失敗します
        if(playType == nn::atk::SinglePlayType_PrioritizeOldest)
        {
            EXPECT_FALSE(soundArchivePlayer.StartSound(&nextHandle, soundId).IsSuccess());
        }
        else
        {
            EXPECT_TRUE(soundArchivePlayer.StartSound(&nextHandle, soundId).IsSuccess());
        }

        // 時間指定のない後着優先のみ soundHandle の再生が停止します
        if(playType == nn::atk::SinglePlayType_PrioritizeNewest)
        {
            EXPECT_FALSE(soundHandle.IsAttachedSound());
        }
        else
        {
            EXPECT_TRUE(soundHandle.IsAttachedSound());
        }

        soundHandle.Stop(0);
        nextHandle.Stop(0);
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#ifndef NN_SDK_BUILD_RELEASE
TEST(PlaySound, PlayDeathWithoutTaskThread)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableTaskThread = g_IsTaskThreadEnabled;
    initializeParam.SetSoundSystemParam(soundSystemParam);

    g_AtkSetup.Initialize(initializeParam, g_Allocator);

    // テスト開始
    nn::atk::SoundHandle soundHandle;
    const std::string TestName = "PlayDeathWithoutTaskThread";

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    // デステスト内の SoundArchivePlayer::Update は子プロセスで呼ばれるため、このプロセスで事前に Update を呼んでおく。
    WaitSoundThreadProcess(soundArchivePlayer);

    // Loader の初期化でクラッシュする
    NN_LOG("[%s] Start %s.\n", (TestName + TestNameSuffix).c_str(), "STRM_MARIOKART");
    EXPECT_DEATH_IF_SUPPORTED(PlayAndUpdateForDeathTest(soundArchivePlayer, soundHandle, STRM_MARIOKART), ".*");

    NN_LOG("[%s] Start %s.\n", (TestName + TestNameSuffix).c_str(), "SE_YOSHI_PLAYERHEAP");
    EXPECT_DEATH_IF_SUPPORTED(PlayAndUpdateForDeathTest(soundArchivePlayer, soundHandle, SE_YOSHI_PLAYERHEAP), ".*");

    NN_LOG("[%s] Start %s.\n", (TestName + TestNameSuffix).c_str(), "SEQ_MARIOKART_PLAYERHEAP");
    EXPECT_DEATH_IF_SUPPORTED(PlayAndUpdateForDeathTest(soundArchivePlayer, soundHandle, SEQ_MARIOKART_PLAYERHEAP), ".*");

    // 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

TEST(PlaySound, PlayDeathWithoutEffect)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // 初期化パラメータでエフェクト無効化フラグを設定
    nnt::atk::util::AtkCommonSetup::InitializeParam initializeParam;
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam;
    soundSystemParam.enableEffect = g_IsEffectEnabled;
    initializeParam.SetSoundSystemParam(soundSystemParam);

    g_AtkSetup.Initialize(initializeParam, g_Allocator);

    // テスト開始
    nn::atk::SoundHandle soundHandle;
    const std::string TestName = "PlayDeathWithoutEffect";

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    nn::atk::EffectReverb reverb;
    size_t reverbBufferSize = reverb.GetRequiredMemSize();
    size_t alingedReverbBufferSize = nn::util::align_up(reverbBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
    void* reverbBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alingedReverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
    nn::audio::MemoryPoolType memoryPoolForReverb;
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForReverb, reverbBuffer, alingedReverbBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(reverbBuffer);

    nn::atk::EffectDelay delay;
    size_t delayBufferSize = delay.GetRequiredMemSize();
    size_t alingedDelayBufferSize = nn::util::align_up(delayBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
    void* delayBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, alingedDelayBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
    nn::audio::MemoryPoolType memoryPoolForDelay;
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForDelay, delayBuffer, alingedDelayBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(delayBuffer);

    EffectAuxTest auxTest;
    size_t auxTestBufferSize = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&auxTest);
    size_t alingedAuxTestBufferSize = nn::util::align_up(auxTestBufferSize, nn::audio::MemoryPoolType::SizeGranularity);
    void* auxTestBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, auxTestBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
    nn::audio::MemoryPoolType memoryPoolForAuxTest;
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForAuxTest, auxTestBuffer, alingedAuxTestBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(auxTestBuffer);

    EffectAuxTest auxTestForFinalMix;
    size_t auxTestBufferSizeForFinalMix = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&auxTestForFinalMix);
    size_t alingedAuxTestBufferSizeForFinalMix = nn::util::align_up(auxTestBufferSizeForFinalMix, nn::audio::MemoryPoolType::SizeGranularity);
    void* auxTestBufferForFinalMix = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, auxTestBufferSizeForFinalMix, nn::audio::MemoryPoolType::AddressAlignment);
    nn::audio::MemoryPoolType memoryPoolForAuxTestFinalMix;
    nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForAuxTestFinalMix, auxTestBufferForFinalMix, alingedAuxTestBufferSizeForFinalMix);
    NN_ABORT_UNLESS_NOT_NULL(auxTestBufferForFinalMix);

    WaitSoundThreadProcess(soundArchivePlayer);

    NN_LOG("[%s] Append reverb.\n", (TestName + TestNameSuffix).c_str());
    reverb.SetEnabled(true);
    EXPECT_DEATH_IF_SUPPORTED(AppendEffectAndUpdateForDeathTest(soundArchivePlayer, nn::atk::AuxBus_A, reverb, reverbBuffer, reverbBufferSize), ".*");

    NN_LOG("[%s] Append delay.\n", (TestName + TestNameSuffix).c_str());
    delay.SetEnabled(true);
    EXPECT_DEATH_IF_SUPPORTED(AppendEffectAndUpdateForDeathTest(soundArchivePlayer, nn::atk::AuxBus_A, delay, delayBuffer, delayBufferSize), ".*");

    NN_LOG("[%s] Append aux.\n", (TestName + TestNameSuffix).c_str());
    auxTest.SetEnabled(true);
    EXPECT_DEATH_IF_SUPPORTED(AppendEffectAuxAndUpdateForDeathTest(soundArchivePlayer, nn::atk::AuxBus_A, auxTest, auxTestBuffer, auxTestBufferSize), ".*");

    NN_LOG("[%s] Append aux to finalMix.\n", (TestName + TestNameSuffix).c_str());
    auxTest.SetEnabled(true);
    EXPECT_DEATH_IF_SUPPORTED(AppendEffectAuxToFinalMixAndUpdateForDeathTest(soundArchivePlayer, auxTestForFinalMix, auxTestBufferForFinalMix, auxTestBufferSizeForFinalMix), ".*");

    NN_LOG("[%s] Clear effect.\n", (TestName + TestNameSuffix).c_str());
    EXPECT_DEATH_IF_SUPPORTED(ClearEffectAndUpdateForDeathTest(soundArchivePlayer, nn::atk::AuxBus_A), ".*");

    NN_LOG("[%s] Clear effect from finalMix.\n", (TestName + TestNameSuffix).c_str());
    EXPECT_DEATH_IF_SUPPORTED(ClearEffectFromFinalMixAndUpdateForDeathTest(soundArchivePlayer), ".*");

    nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForAuxTestFinalMix);
    nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForAuxTest);
    nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForDelay);
    nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForReverb);

    // 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

#if defined( NN_ATK_CONFIG_ENABLE_THREAD_CORE_NUMBER_OBSERVATION )
TEST(PlaySound, PlayDeathGetThreadCoreNumber)
{
    nnt::atk::util::PreAtkTestArg preAtkTestArg;
    preAtkTestArg.isResourceExclusionCheckEnabled = true;
    nnt::atk::util::OnPreAtkTest(preAtkTestArg);
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::GetSoundThreadCoreNumber(), ".*" );
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::GetTaskThreadCoreNumber(), ".*" );

    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);
    EXPECT_NE( nn::atk::SoundSystem::GetSoundThreadCoreNumber(), nn::atk::detail::fnd::Thread::InvalidCoreNumber );
    EXPECT_NE( nn::atk::SoundSystem::GetTaskThreadCoreNumber(), nn::atk::detail::fnd::Thread::InvalidCoreNumber );

    WaitSoundThreadProcess(g_AtkSetup.GetSoundArchivePlayer());

    // 終了処理
    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();

    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::GetSoundThreadCoreNumber(), ".*" );
    EXPECT_DEATH_IF_SUPPORTED( nn::atk::SoundSystem::GetTaskThreadCoreNumber(), ".*" );
}
#endif

#endif

#ifndef NN_SDK_BUILD_RELEASE
struct TestRegionCallbackParam
{
    uint32_t* pRegionList;
    uint32_t  regionListSize;
    uint32_t* pRegionListIndex;
};

nn::atk::StreamRegionCallbackResult StreamRegionCallbackSimple(nn::atk::StreamRegionCallbackParam* param, void* arg )
{
    //NN_ASSERT_NOT_NULL(param);
    //NN_ASSERT_NOT_NULL(arg);

    TestRegionCallbackParam* pParam = static_cast<TestRegionCallbackParam*>(arg);
    //NN_ASSERT_NOT_NULL( pParam );
    //NN_ASSERT_RANGE(*pParam->pRegionListIndex, 0u, pParam->regionListSize);

    param->regionNo = pParam->pRegionList[*pParam->pRegionListIndex];
    (*pParam->pRegionListIndex)++;
    return nn::atk::StreamRegionCallbackResult_Continue;
}



void PlaySoundAndWaitForDeath(
    nn::atk::SoundArchivePlayer& archivePlayer,
    nn::atk::SoundHandle& handle,
    nn::atk::SoundArchive::ItemId soundId,
    nn::atk::SoundStartable::StartInfo* pStartInfo) NN_NOEXCEPT
{
    archivePlayer.StartSound(&handle, soundId, pStartInfo).IsSuccess();

    // 100 回以内には死ぬはず…という値です。
    for (int i = 0; i < 100; ++i)
    {
        nnt::atk::util::UpdateAndWait(archivePlayer);
    }
}

TEST(PlaySound, DeathTestWithStreamJumpAndLoop)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize( g_Allocator );

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();
    nn::atk::SoundHandle soundHandle;

    uint32_t regionList[] = {
        0,
        1,
        1
    };

    uint32_t regionListIndex = 0;

    TestRegionCallbackParam regionCallbackParam;
    regionCallbackParam.pRegionList = regionList;
    regionCallbackParam.regionListSize = sizeof(regionList) / sizeof(regionList[0]);
    regionCallbackParam.pRegionListIndex = &regionListIndex;

    nn::atk::SoundStartable::StartInfo info;
    info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_StreamSoundInfo;
    info.streamSoundInfo.regionCallback = StreamRegionCallbackSimple;
    info.streamSoundInfo.regionCallbackArg = &regionCallbackParam;
    EXPECT_DEATH_IF_SUPPORTED(PlaySoundAndWaitForDeath(soundArchivePlayer, soundHandle, STRM_NAMEDREGION_WITHLOOP, &info), ".*");

    g_AtkSetup.Finalize( g_Allocator );
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif

extern "C" void nnMain()
{
    int argc = nnt::GetHostArgc();
    char** argv = nnt::GetHostArgv();

    // ユーザー引数による設定
    ParseUserArgument(argc, argv);

    // GoogleTEST 初期化
    ::testing::InitGoogleTest(&argc, argv);
    // GoogleTEST 実行
    int testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);

    return;
}
