﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Log.h>

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>

namespace {

const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup   g_FsSetup;
nnt::atk::util::AtkCommonSetup   g_AtkSetup;

static char                 g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

}

// デフォルトのデコーダ数での初期化・終了処理がうまくいくことをチェック
TEST(OpusDecoder, OpusInitializeFinalizeDefaultTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize();
    void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
    nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize);

    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.Finalize(g_Allocator);

    nn::atk::FinalizeOpusDecoder();
    g_Allocator.Free(decoderWorkBuffer);

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

// デコーダ数指定版での初期化・終了処理がうまくいくことをチェック
TEST(OpusDecoder, OpusInitializeFinalizeMultiDecoderTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    for (int decoderCount = 1; decoderCount < 5; ++decoderCount )
    {
        size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(decoderCount);
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
        nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize, decoderCount);

        g_AtkSetup.Initialize(g_Allocator);
        g_AtkSetup.Finalize(g_Allocator);

        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

// 複数回初期化・終了処理を呼んでもうまくいくことをチェック
TEST(OpusDecoder, OpusInitializeFinalizeTwiceTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize();
    void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
    nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize);
    nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize);

    g_AtkSetup.Initialize(g_Allocator);
    g_AtkSetup.Finalize(g_Allocator);

    nn::atk::FinalizeOpusDecoder();
    nn::atk::FinalizeOpusDecoder();
    g_Allocator.Free(decoderWorkBuffer);

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}

// 初期化関連で事前条件を満たさず落ちるパターンのテスト
#if !defined(NN_SDK_BUILD_RELEASE)
TEST(OpusDecoder, OpusDecoderInitializeDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();

    // 初期化時に渡すバッファサイズを間違えるパターン１
    {
        size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(1);
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize, 3), "");
        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
    }

    // 初期化時に渡すバッファサイズを間違えるパターン２
    {
        size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(4);
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize, 2), "");

        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
    }

    // デコーダー数０を指定するパターン１
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::GetRequiredOpusDecoderBufferSize(0), "");
    }

    // デコーダー数０を指定するパターン２
    {
        size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(2);
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::InitializeOpusDecoder(decoderWorkBuffer, workBufferSize, 0), "");

        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
    }

    // nullptr なバッファを渡してしまうパターン
    {
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::InitializeOpusDecoder(nullptr, 100u, 2), "");
        nn::atk::FinalizeOpusDecoder();
    }

    // サイズを 0 で指定してしまうパターン
    {
        size_t workBufferSize = nn::atk::GetRequiredOpusDecoderBufferSize(2);
        void* decoderWorkBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, workBufferSize);
        EXPECT_DEATH_IF_SUPPORTED(nn::atk::InitializeOpusDecoder(decoderWorkBuffer, 0u, 2), "");

        nn::atk::FinalizeOpusDecoder();
        g_Allocator.Free(decoderWorkBuffer);
    }

    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
#endif
