﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.fsid"
#include "testAtk_TestEffectUtil.h"

#include <nnt.h>
#include <nnt/atkUtil/testAtk_Util.h>
#include <nnt/atkUtil/testAtk_CommonSetup.h>
#include <nn/atk.h>
#include <nn/nn_Log.h>

#include <nn/mem.h>

namespace {
const int MemoryHeapSize = 32 * 1024 * 1024;

nnt::atk::util::FsCommonSetup g_FsSetup;
nnt::atk::util::AtkCommonSetup g_AtkSetup;

static char g_HeapMemory[MemoryHeapSize];
nn::mem::StandardAllocator  g_Allocator;

bool IsDelayChannelModeSupported(nn::atk::EffectBase::ChannelMode channelMode)
{
    return channelMode == nn::atk::EffectBase::ChannelMode_1Ch || channelMode == nn::atk::EffectBase::ChannelMode_2Ch
        || channelMode == nn::atk::EffectBase::ChannelMode_4Ch || channelMode == nn::atk::EffectBase::ChannelMode_6Ch;
}
}

// TODO: チャンネル設定以外のテストを追加する
#ifndef NN_SDK_BUILD_RELEASE
TEST(Effect, DelayChannelSettingDeathTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectDelay delay;

        if (!IsDelayChannelModeSupported(channelMode))
        {
            continue;
        }

        delay.SetChannelMode(channelMode);
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
        nn::atk::ChannelIndex* pInvalidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
        NN_ABORT_UNLESS_NOT_NULL(pInvalidChannelSetting);
        // チャンネルの重複
        if (ChannelCount > 1)
        {
            for (int i = 0; i < ChannelCount; ++i)
            {
                if (i == 0 || i == 1)
                {
                    pInvalidChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
                else
                {
                    pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
                }
            }
            EXPECT_DEATH_IF_SUPPORTED(delay.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");
        }

        // チャンネル設定の値に -1 が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i - 1);
        }
        EXPECT_DEATH_IF_SUPPORTED(delay.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        // チャンネル設定の値に nn::atk::ChannelIndex_Count が含まれる
        for (int i = 0; i < ChannelCount; ++i)
        {
            if (i == ChannelCount - 1)
            {
                pInvalidChannelSetting[i] = nn::atk::ChannelIndex_Count;
            }
            else
            {
                pInvalidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
            }
        }
        EXPECT_DEATH_IF_SUPPORTED(delay.SetChannelIndex(pInvalidChannelSetting, channelMode), ".*");

        g_Allocator.Free(reinterpret_cast<void*>(pInvalidChannelSetting));
    }
    g_Allocator.Finalize();
}
#endif

TEST(Effect, DelaySetChannelCountTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectDelay delay;
        if (IsDelayChannelModeSupported(channelMode))
        {
            EXPECT_TRUE(delay.SetChannelMode(channelMode));
            ASSERT_EQ(channelMode, delay.GetChannelMode());
            const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);
            nn::atk::ChannelIndex* pDelayChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pDelayChannelSetting);
            delay.GetChannelIndex(pDelayChannelSetting, ChannelCount);
            for (int i = 0; i < ChannelCount; ++i)
            {
                EXPECT_EQ(static_cast<nn::atk::ChannelIndex>(i), pDelayChannelSetting[i]);
            }
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, DelaySetChannelIndexTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectDelay delay;
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

        if (IsDelayChannelModeSupported(channelMode))
        {
            // サポートしているチャンネル数の場合
            EXPECT_TRUE(delay.SetChannelMode(channelMode));
            EXPECT_EQ(channelMode, delay.GetChannelMode());

            nn::atk::ChannelIndex* pChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
            NN_ABORT_UNLESS_NOT_NULL(pChannelSetting);
            // 初期化
            for (int i = 0; i < ChannelCount; ++i)
            {
                pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
            }

            // 適切なチャンネル設定となるものについて総当たり検査
            while (pChannelSetting[ChannelCount - 1] != nn::atk::ChannelIndex_Count)
            {
                // テストケースとして適切か(チャンネル設定の被りがないか)確認
                bool isValidChannelSetting = true;
                for (int i = 0; i < ChannelCount; ++i)
                {
                    for (int j = i + 1; j < ChannelCount; ++j)
                    {
                        isValidChannelSetting &= (pChannelSetting[i] != pChannelSetting[j]);
                    }
                }

                // テスト本体
                if (isValidChannelSetting)
                {
                    EXPECT_TRUE(delay.SetChannelIndex(pChannelSetting, channelMode));

                    ASSERT_EQ(channelMode, delay.GetChannelMode());
                    nn::atk::ChannelIndex* pDelayChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
                    NN_ABORT_UNLESS_NOT_NULL(pDelayChannelSetting);
                    delay.GetChannelIndex(pDelayChannelSetting, ChannelCount);
                    for (int i = 0; i < ChannelCount; ++i)
                    {
                        EXPECT_EQ(pChannelSetting[i], pDelayChannelSetting[i]);
                    }
                    g_Allocator.Free(reinterpret_cast<void*>(pDelayChannelSetting));
                }

                // テストケースの更新
                for (int i = 0; i < ChannelCount; ++i)
                {
                    pChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(pChannelSetting[i] + 1);
                    // 繰り上がりしない場合
                    if (pChannelSetting[i] != nn::atk::ChannelIndex_Count || i == ChannelCount - 1)
                    {
                        break;
                    }

                    pChannelSetting[i] = nn::atk::ChannelIndex_FrontLeft;
                }
            }

            g_Allocator.Free(reinterpret_cast<void*>(pChannelSetting));
        }
        else
        {
            // サポートしていないチャンネル数の場合
            EXPECT_FALSE(delay.SetChannelMode(channelMode));
        }
    }

    g_Allocator.Finalize();
}

TEST(Effect, DelayAppendTest)
{
    nnt::atk::util::OnPreAtkTest();
    g_Allocator.Initialize(g_HeapMemory, MemoryHeapSize);
    g_FsSetup.Initialize();
    g_AtkSetup.Initialize(g_Allocator);

    nn::atk::SoundArchivePlayer& soundArchivePlayer = g_AtkSetup.GetSoundArchivePlayer();

    for (auto& channelMode : nnt::atk::effectUtil::TestEffectChannelMode)
    {
        nn::atk::EffectDelay delay;
        const int ChannelCount = nnt::atk::effectUtil::ConvertChannelModeToInt(channelMode);

        // 使用するチャンネル設定の初期化
        nn::atk::ChannelIndex* pValidChannelSetting = reinterpret_cast<nn::atk::ChannelIndex*>(nnt::atk::util::AllocateUninitializedMemory(g_Allocator, sizeof(nn::atk::ChannelIndex) * ChannelCount));
        NN_ABORT_UNLESS_NOT_NULL(pValidChannelSetting);
        // 初期化
        for (int i = 0; i < ChannelCount; ++i)
        {
            pValidChannelSetting[i] = static_cast<nn::atk::ChannelIndex>(i);
        }

        // Append 前に変更できることをテストする
        if (IsDelayChannelModeSupported(channelMode))
        {
            EXPECT_TRUE(delay.SetChannelMode(channelMode));
            EXPECT_TRUE(delay.SetChannelIndex(pValidChannelSetting, channelMode));
        }
        else
        {
            // 変更できない場合はそれを確認して次のテスト項目へ
            EXPECT_FALSE(delay.SetChannelMode(channelMode));
            EXPECT_FALSE(delay.SetChannelIndex(pValidChannelSetting, channelMode));
            continue;
        }

        // エフェクトの Append
        EXPECT_TRUE(delay.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));
        size_t delayBufferSize = nn::util::align_up(delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        void* delayBuffer = nnt::atk::util::AllocateUninitializedMemory(g_Allocator, delayBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(delayBuffer);
        nn::audio::MemoryPoolType memoryPoolForDelay;
        nn::atk::SoundSystem::AttachMemoryPool(&memoryPoolForDelay, delayBuffer, delayBufferSize);
        delay.SetEnabled(true);
        EXPECT_TRUE(nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &delay, delayBuffer, delayBufferSize));

        // Append で送ったコマンドが処理されるのを待つ
        nnt::atk::util::WaitForProcessCommand();

        // nn::audio のエフェクト追加 API 呼出後、
        // RequestUpdateAudioRendrer が呼ばれて IsRemovable が false になるまで待つ
        nnt::atk::effectUtil::WaitForEffectAppend(delay);

        // Append 完了後に変更ができないことをテストする
        EXPECT_FALSE(delay.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));
        EXPECT_FALSE(delay.SetChannelMode(channelMode));
        EXPECT_FALSE(delay.SetChannelIndex(pValidChannelSetting, channelMode));

        delay.SetEnabled(false);
        nnt::atk::effectUtil::UpdateAndProcess(soundArchivePlayer);
        nnt::atk::effectUtil::WaitForEffectClear(delay);
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &delay);
        nn::atk::SoundSystem::DetachMemoryPool(&memoryPoolForDelay);

        // Remove 後に変更できることをテストする
        EXPECT_TRUE(delay.SetChannelMode(channelMode));
        EXPECT_TRUE(delay.SetChannelIndex(pValidChannelSetting, channelMode));
        EXPECT_TRUE(delay.SetSampleRate(nn::atk::EffectReverb::SampleRate_48000));

        g_Allocator.Free(reinterpret_cast<void*>(pValidChannelSetting));
    }

    g_AtkSetup.Finalize(g_Allocator);
    g_FsSetup.Finalize();
    g_Allocator.Finalize();
}
