﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/atk/atk_WavOutRecorder.h>
#include <nn/atk/fnd/io/atkfnd_FileStreamImpl.h>

#include "testAtk_TestHeap.h"


//---------------------------------------------------------------------------
//! @brief  テスト用のレコーダーです。
//---------------------------------------------------------------------------
class TestRecorder
{
public:
    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    TestRecorder() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~TestRecorder() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  レコーダーを初期化します。
    //!
    //! @param[in] pThreadStack    録音スレッドのスタックとして使用するバッファです
    //! @param[in] threadStackSize 録音スレッドのスタックとして使用するバッファの大きさです
    //! @param[in] workBuffer      録音処理に使用するワークバッファです
    //!
    //! @pre
    //! - workBuffer != nullptr
    //! - workBuffer で確保したバッファのサイズが GetRequiredWorkBufferSize 以上
    //! - pThreadStack != nullptr
    //! - pThreadStack が nn::os::ThreadStackAlignment にアラインしている
    //! - threadStackSize >= DeviceOutRecorder::RequiredThreadStackSize
    //---------------------------------------------------------------------------
    void Initialize(void* buffer, size_t bufferSize, void *pThreadStack, size_t threadStackSize) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  レコーダーの終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  レコーダーのバッファに必要なサイズを取得します。
    //---------------------------------------------------------------------------
    size_t GetRequiredMemorySize() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  録音を開始します。
    //!
    //! @param[in] frames          録音を行うフレーム数です
    //---------------------------------------------------------------------------
    void StartRecording(uint32_t frames = 0) NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  録音を終了します。
    //---------------------------------------------------------------------------
    void StopRecording() NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  録音中であるかどうかを取得します。
    //---------------------------------------------------------------------------
    bool IsRecording() const NN_NOEXCEPT;

    //---------------------------------------------------------------------------
    //! @brief  録音したサンプルデータの出力ファイル名を設定します。
    //!
    //! @param[in]  録音したサンプルデータの出力ファイル名です。
    //---------------------------------------------------------------------------
    void SetOutputFileName( const char* pOutputFileName ) NN_NOEXCEPT
    {
        if ( m_IsRecording == false )
        {
            m_pOutputFileName = pOutputFileName;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief  録音したサンプルデータの出力ファイル名を取得します。
    //!
    //! @return  録音したサンプルデータの出力ファイル名を返します。
    //---------------------------------------------------------------------------
    const char* GetOutputFileName() const NN_NOEXCEPT
    {
        return m_pOutputFileName;
    }

private:
    bool                      m_IsInitialized; //!< 初期化されたかどうか
    bool                      m_IsRecording;   //!< 録音中かどうか
    const char*               m_pOutputFileName; //!< 出力ファイル名

    nn::atk::WavOutRecorder   m_Recorder;
    nn::atk::detail::fnd::FileStreamImpl m_RecordingFileStream;
};
