﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cassert>

#include <nn/nn_Common.h>

#include <nn/fs.h>

#include "testAtk_FileManager.h"


//---------------------------------------------------------------------------
//  ヒープを設定します。
//---------------------------------------------------------------------------
void FileManager::SetHeap( TestHeap* pHeap )
{
    m_pHeap = pHeap;
}

//------------------------------------------------------------------------------
//  ファイルからデータを読み込みます。
//------------------------------------------------------------------------------
bool FileManager::ReadFile( const char* pPath, char** ppData, size_t* pSize )
{
    *ppData = NULL;
    *pSize = 0;

    // ファイルを開く
    nn::fs::FileHandle handle;
    {
        nn::Result result = nn::fs::OpenFile( &handle, pPath, nn::fs::OpenMode_Read );
        if ( result.IsFailure() )
        {
            return false;
        }
    }

    // ファイルサイズを取得
    size_t fileSize;
    {
        int64_t size = 0;
        nn::Result result = nn::fs::GetFileSize( &size, handle );
        fileSize = static_cast<size_t>(size);
        if ( result.IsFailure() )
        {
            nn::fs::CloseFile( handle );
            return false;
        }
    }

    // ファイルサイズのメモリを確保
    void* pData;
    pData = m_pHeap->Alloc( fileSize );

    // ファイルを先頭から読み込む
    size_t readSize;
    {
        nn::Result result = nn::fs::ReadFile( &readSize, handle, 0, pData, static_cast<size_t>(fileSize) );
        if ( result.IsFailure() )
        {
            m_pHeap->Free( pData );
            nn::fs::CloseFile( handle );
            return false;
        }
    }

    *ppData = static_cast<char*>( pData );
    *pSize = readSize;

    // ファイルを閉じる
    nn::fs::CloseFile( handle );
    return true;
}

//------------------------------------------------------------------------------
//  ファイルから読み込んだデータを破棄します。
//------------------------------------------------------------------------------
void FileManager::ReleaseData( char* pData )
{
    m_pHeap->Free( pData );
}

//------------------------------------------------------------------------------
//  データをファイルに保存します。
//------------------------------------------------------------------------------
bool FileManager::WriteFile( const char* pData, size_t size, const char* pPath )
{
    // ファイルを開く
    nn::fs::FileHandle handle;
    {
        nn::Result result = nn::fs::OpenFile( &handle, pPath, nn::fs::OpenMode_Write );
        if ( result.IsFailure() )
        {
            return false;
        }
    }

    // データを保存する
    {
        nn::fs::WriteOption option;
        nn::Result result = nn::fs::WriteFile( handle, 0, reinterpret_cast<const char*>( pData ), static_cast<size_t>(size), option );
        if ( result.IsFailure() )
        {
            nn::fs::CloseFile( handle );
            return false;
        }
    }

    // ファイルを閉じる
    nn::fs::CloseFile( handle );
    return true;
}
