﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>
#include <nn/npns.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/http.h>
#include <nn/ssl.h>
#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/account.h>
#include <nn/account/account_ResultForAdministrators.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nsd/nsd_ApiForMenu.h>
#include <nn/util/util_StringUtil.h>
#include <nnt/npnsUtil/npnsUtil_NotificationPublisher.h>
#include <nnt/npnsUtil/npnsUtil_TestBase.h>

#include "../../../Programs/Eris/Sources/Processes/npns/npns_Config.h"

namespace nnt {
namespace npns {
namespace util {

using namespace nn;

NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
nn::http::ConnectionBroker s_ConnectionBroker;

struct NotificationPublisherDefinition {
    const char* pEnvironment;
    NotificationPublisher publisher;
}
s_NotificationPublisherDefinitions[] =
    {
        { "td1", NotificationPublisher(
            s_ConnectionBroker, "c17e59e1df28556e62b837d5a36e87bc") },
        { "dd1", NotificationPublisher(
            s_ConnectionBroker, "c17e59e1df28556e62b837d5a36e87bc") },
        { "jd1", NotificationPublisher(
            s_ConnectionBroker, "c17e59e1df28556e62b837d5a36e87bc") },
    };

TestBase::TestBase()
{
}

void TestBase::SetUpTestCaseImpl(bool establishConnection)
{
    nn::Result result;

    result = nn::time::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

#ifndef NN_BUILD_CONFIG_OS_WIN
    result = nn::socket::Initialize(
        s_SocketMemoryPoolBuffer,
        sizeof(s_SocketMemoryPoolBuffer),
        nn::socket::MinSocketAllocatorSize,
        2);
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::ssl::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::http::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    result = nn::nifm::Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);

    if (establishConnection)
    {
        nn::nifm::SubmitNetworkRequestAndWait();
    }
#endif

    result = nn::npns::InitializeForSystem();
    NNT_ASSERT_RESULT_SUCCESS(result);

    s_ConnectionBroker.Initialize();
    NNT_ASSERT_RESULT_SUCCESS(result);
#ifdef NN_BUILD_CONFIG_OS_WIN
    s_ConnectionBroker.SetProxy("proxy.nintendo.co.jp", 8080);
#endif
    s_ConnectionBroker.SetMaxTotalConnections(2);
    s_ConnectionBroker.SetVerbose(true);

    PollState(nn::npns::State_Connected, nn::TimeSpan::FromSeconds(3));
}

void TestBase::SetUpTestCase()
{
    SetUpTestCaseImpl(true);
}

void TestBase::SetUpTestCaseWithoutConnection()
{
    SetUpTestCaseImpl(false);
}

void TestBase::TearDownTestCase()
{
    s_ConnectionBroker.Finalize();

    nn::npns::FinalizeForSystem();

#ifndef NN_BUILD_CONFIG_OS_WIN
    nn::http::Finalize();
    nn::ssl::Finalize();
    nn::socket::Finalize();
#endif

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
}


void TestBase::PublishNotification(const nn::npns::NotificationToken token, const char* pPayload)
{
#ifdef SEND_AUTO
    Result result = GetNotificationPublisher()->Send(token, pPayload);
    NNT_ASSERT_RESULT_SUCCESS(result);
#else
    NN_LOG(
        "*** Please run manually: ***\n"
        "curl --request POST \\\n"
        "  --proxy http://proxy.nintendo.co.jp:8080 \\\n"
        "  --url https://provider-" NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/notifications \\\n"
        "  --header 'content-type: application/json' \\\n"
        "  --header 'X-Ndid-AccessToken: c17e59e1df28556e62b837d5a36e87bc' \\\n"
        "  --data '{\"requests\": [{\"to\": \"%s\", \"data\": \"%s\", \"store_offline\": true }]}'\n", token.data, pPayload);
#endif
}


void TestBase::PublishNotificationByTopic(const char* pTopicName, const char* pPayload)
{
    Result result = GetNotificationPublisher()->SendByTopic(pTopicName, pPayload);
    NNT_ASSERT_RESULT_SUCCESS(result);
}

void TestBase::CreateTopic(char* pNameOut, size_t length, nn::ApplicationId appId, const char* pPrefix)
{
    Result result = GetNotificationPublisher()->CreateTopic(pNameOut, length, appId, pPrefix);
    NNT_ASSERT_RESULT_SUCCESS(result);
}

bool TestBase::PollState(nn::npns::State state, const nn::TimeSpan& timeout)
{
    os::SystemEvent eventState;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::npns::GetStateChangeEvent(eventState)
    );
    os::Tick tickStart = nn::os::GetSystemTick();
    TimeSpan spanRest;
    while (nn::npns::GetState() != state
        && (spanRest = (tickStart + os::Tick(timeout) - nn::os::GetSystemTick()).ToTimeSpan()) > 0)
    {
        eventState.TimedWait(spanRest);
        NN_LOG("(PollState) State = %s at %s\n", GetStateName(nn::npns::GetState()), GetDateTimeStr());
    }
    return nn::npns::GetState() == state;
}

NotificationPublisher * TestBase::GetNotificationPublisher()
{
    nsd::EnvironmentIdentifier envName;
    if(std::strcmp(NN_NPNS_SERVER_ENVIRONMENT, "%") == 0)
    {
        nn::nsd::GetEnvironmentIdentifier(&envName);
    }
    else
    {
        nn::util::Strlcpy<char>(envName.value, NN_NPNS_SERVER_ENVIRONMENT, sizeof(envName.value));
    }

    for (int i = 0; i < sizeof(s_NotificationPublisherDefinitions) / sizeof(s_NotificationPublisherDefinitions[0]); ++i)
    {
        NotificationPublisherDefinition& def = s_NotificationPublisherDefinitions[i];
        if (std::strcmp(def.pEnvironment, envName.value) == 0)
        {
            return &def.publisher;
        }
    }
    NN_ABORT("Can not determine configuration for %s.", envName.value);
    return nullptr;
}

void TestBase::SetUp()
{
    NN_LOG("Start at %s\n", GetDateTimeStr());
}

void TestBase::TearDown()
{
    NN_LOG("Finish at %s\n", GetDateTimeStr());
}

// スレッドアンセーフ
const char* TestBase::GetStateName(nn::npns::State state)
{
    switch(state)
    {
        case nn::npns::State_Initial:           return "State_Initial";
        case nn::npns::State_Idle:              return "State_Idle";
        case nn::npns::State_AwaitingOnline:    return "State_AwaitingOnline";
        case nn::npns::State_Online:            return "State_Online";
        case nn::npns::State_NoValidJid:        return "State_NoValidJid";
        case nn::npns::State_RequestingJid:     return "State_RequestingJid";
        case nn::npns::State_ReadyToConnect:    return "State_ReadyToConnect";
        case nn::npns::State_Connecting:        return "State_Connecting";
        case nn::npns::State_Connected:         return "State_Connected";
        case nn::npns::State_BackoffWaiting:    return "State_BackoffWaiting";
        case nn::npns::State_Suspend:           return "State_Suspend";
        case nn::npns::State_ShutingDown:       return "State_ShutingDown";
        case nn::npns::State_Exit:              return "State_Exit";
        case nn::npns::State_AwaitingEulaAccepted:  return "State_AwaitingEulaAccepted";
        case nn::npns::State_CheckingArrival:       return "State_CheckingArrival";
        case nn::npns::State_ConnectedOnHalfAwake:  return "State_ConnectedOnHalfAwake";
        default:
        {
            static char buffer[32]; // マルチスレッド利用は考えてない
            nn::util::SNPrintf(buffer, sizeof(buffer), "State(%d)", static_cast<int>(state));
            return buffer;
        }
        break;
    }
}

// スレッドアンセーフ
const char* TestBase::GetDateTimeStr()
{
    static char s_Buffer[128];

    nn::time::PosixTime current;
    auto result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&current);
    if(result.IsFailure())
    {
        nn::util::SNPrintf(s_Buffer, sizeof(s_Buffer), "(NetworkSystemClock invalid)");
        return s_Buffer;
    }

    nn::time::CalendarTime c = nn::time::ToCalendarTimeInUtc(current);
    nn::util::SNPrintf(s_Buffer, sizeof(s_Buffer), "[UTC]%04d/%02d/%02d %02d:%02d:%02d ", c.year, c.month, c.day, c.hour, c.minute, c.second);
    return s_Buffer;
}

}
}
}
