﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/npnsUtil/npnsUtil_NotificationPublisher.h>

#include "../../../Programs/Eris/Sources/Processes/npns/npns_Config.h"

using namespace nn;

namespace nnt {
namespace npns {
namespace util {

NotificationPublisher::NotificationPublisher(nn::http::ConnectionBroker& cb, const char* pNdidAccessToken)
    : m_ConnectionBroker(cb)
    , m_pNdidAccessToken(pNdidAccessToken)
{
}

nn::Result NotificationPublisher::Send(const nn::npns::NotificationToken token, const char* pPayload, bool bStoreOffline)
{
    Result result;
    http::Request req(m_ConnectionBroker);

    SetupRequest(&req);
    req.SetUrlPointer("https://provider-" NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/notifications");
    req.SetPostDataByFormatString(
        "{\n"
        "   \"requests\": [\n"
        "       {\n"
        "           \"to\": \"%s\",\n"
        "           \"data\": \"%s\",\n"
        "           \"store_offline\": %s\n"
        "       }\n"
        "   ]\n"
        "}\n",
        token.data, pPayload, bStoreOffline ? "true" : "false"
    );

    http::Response res;
    result = req.GetResponse(&res);
    if (result.IsFailure())
    {
        return result;
    }

    return DiscardResponseBody(&res);
}

nn::Result NotificationPublisher::SendByTopic(const char* pTopicName, const char* pPayload)
{
    Result result;
    http::Request req(m_ConnectionBroker);

    SetupRequest(&req);
    req.SetUrlByFormatString("https://provider-" NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/topics/%s/publish", pTopicName);
    req.SetPostDataByFormatString(
        "{\n"
        "   \"data\": \"%s\"\n"
        "}\n",
        pPayload
    );

    http::Response res;
    result = req.GetResponse(&res);
    if (result.IsFailure())
    {
        NN_LOG("GetResponse failed.(0x%08x)", result.GetInnerValueForDebug());
        return result;
    }

    return DiscardResponseBody(&res);
}

nn::Result NotificationPublisher::SendByTopic(nn::ApplicationId appId, const char* pPrefix, const char* pPayload)
{
    char name[64];
    nn::util::SNPrintf(name, sizeof(name), "nx_%s_%016llx", pPrefix, appId.value);
    return SendByTopic(name, pPayload);
}

nn::Result NotificationPublisher::CreateTopic(char* pNameOut, size_t length, nn::ApplicationId appId, const char* pPrefix)
{
    Result result;
    http::Request req(m_ConnectionBroker);

    SetupRequest(&req);
    req.SetUrlByFormatString("https://provider-" NN_NPNS_SERVER_ENVIRONMENT ".npns.srv.nintendo.net/api/v1/topics");
    req.SetPostDataByFormatString(
        "{\n"
        "   \"device_type\": \"nx\",\n"
        "   \"topic_types\": [\"%s\"],\n"
        "   \"application_id\": \"%016llx\",\n"
        "   \"sub_id\": \"%016llx\"\n"
        "}\n",
        pPrefix, appId.value, appId.value
    );

    http::Response res;
    result = req.GetResponse(&res);
    if (result.IsFailure())
    {
        NN_LOG("GetResponse failed.(0x%08x)", result.GetInnerValueForDebug());
        return result;
    }
    nn::util::SNPrintf(pNameOut, length, "nx_%s_%016llx", pPrefix, appId.value);
    return DiscardResponseBody(&res);
}

Result NotificationPublisher::DiscardResponseBody(http::Response* pRes)
{
    char buf[128];
    size_t readlen;

    NN_LOG("http status = %d, content length = %d\n", pRes->GetStatusCode(), pRes->GetContentLength());

    do
    {
        Result result = pRes->ReadBody(&readlen, buf, sizeof(buf));
        if (result.IsFailure())
        {
            NN_LOG("ReadBody failed.(0x%08x)", result.GetInnerValueForDebug());
            return result;
        }
        NN_LOG("%.*s", readlen, buf);
    } while (readlen == sizeof(buf));
    NN_LOG("\n");
    return ResultSuccess();
}

Result NotificationPublisher::SetupRequest(http::Request* pReq)
{
    pReq->SetMethodType(http::MethodType_Post);
    NN_RESULT_DO(pReq->AddHeaderFormat("Content-Type: application/json"));
    NN_RESULT_DO(pReq->AddHeaderFormat("X-Ndid-AccessToken: %s", m_pNdidAccessToken));
    return ResultSuccess();
}

}
}
}
