﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string> // std::string
#include <cstring> // std::strrchr
#include <cctype> // std::isxdigit

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <direct.h>  // _getcwd
#endif  // defined(NN_BUILD_CONFIG_OS_WIN)

namespace nnt {
namespace codec {
namespace util {
namespace {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
const std::string GetCwd() NN_NOEXCEPT
{
    std::string cwd;
    const char* pPath = nn::os::GetHostArgv()[0];
    const char* pEnd = std::strrchr(pPath, '\\');
    NN_ABORT_UNLESS_NOT_NULL(pEnd);
    const size_t length = pEnd - pPath + 1;
    cwd.append(pPath, length);
    return cwd;
}
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)

const std::string g_MountHostName = "host";
const std::string g_MountRomName = "rom";

const std::string g_InputDirectoryPath = ":/Inputs";
const std::string g_OutputDirectoryPath = ":/Outputs";
const std::string g_ReferenceDirectoryPath = ":/References";

const std::size_t FcivFileBufferSize = 256;
}

////////////////////////////////////////////////////////////////////////////////
// Codec utility functions
////////////////////////////////////////////////////////////////////////////////
std::string GetMountPointHost() NN_NOEXCEPT
{
    return g_MountHostName;
}

std::string GetMountPointRom() NN_NOEXCEPT
{
    return g_MountRomName;
}

std::string GetInputDirectoryPath() NN_NOEXCEPT
{
    return GetMountPointRom() + g_InputDirectoryPath;
}

std::string GetOutputDirectoryPath() NN_NOEXCEPT
{
    return GetMountPointHost() + g_OutputDirectoryPath;
}

std::string GetReferenceDirectoryPath() NN_NOEXCEPT
{
    return GetMountPointRom() + g_ReferenceDirectoryPath;
}

std::string GetCodecTestBinariesPath() NN_NOEXCEPT
{
    const std::string relativePath = "\\Externals\\TestBinaries\\Codec";
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto cwd = GetCwd();
#else  // defined(NN_BUILD_CONFIG_OS_HORIZON)
    char path[nn::fs::EntryNameLengthMax];
    auto cwd = std::string(_getcwd(path, sizeof(path)));
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)
    auto position = cwd.find("\\Tests");
    if (position == std::string::npos)
    {
        position = cwd.find("\\Externals");
    }
    auto rootDir = cwd.substr(0, position);
    return rootDir + relativePath;
}

std::string GetValueStringLabeledWith(
    const std::string& inputString,
    const std::string& variableName,
    const std::string& startCharactor = "=",
    const std::string& endCharactors = ","
)
{
    const auto searchTag = variableName + startCharactor;
    const auto variableNameStartAddress =
        std::strstr(inputString.c_str(), searchTag.c_str());
    if (nullptr == variableNameStartAddress)
    {
        return "0";
    }
    const auto valueStringStartPosition =
        variableNameStartAddress - inputString.c_str() + searchTag.length();
    const auto endCharactorPosition =
        inputString.find_first_of(endCharactors.c_str(), valueStringStartPosition);
    return (std::string::npos == endCharactorPosition)
        ? inputString.substr(valueStringStartPosition)
        : inputString.substr(valueStringStartPosition, endCharactorPosition - valueStringStartPosition);
}

std::string GetValueStringLabeledWith(
    const char* inputString,
    const char* variableName,
    const char* startCharactor = "=",
    const char* endCharactors = ","
)
{
    return GetValueStringLabeledWith(
        std::string(inputString),
        std::string(variableName),
        std::string(startCharactor),
        std::string(endCharactors));
}

int GetIntegerValueLabeledWith(
    const std::string& inputString,
    const std::string& variableName,
    const std::string& startCharactor = "=",
    const std::string& endCharactors = ","
)
{
    return std::stoi(GetValueStringLabeledWith(inputString, variableName, startCharactor, endCharactors));
}

float GetFloatValueLabeledWith(
    const std::string& inputString,
    const std::string& variableName,
    const std::string& startCharactor = "=",
    const std::string& endCharactors = ","
)
{
    return std::stof(GetValueStringLabeledWith(inputString, variableName, startCharactor, endCharactors));
}

std::string GetFileExtension(const std::string& path, int dotCount)
{
    if (path.length() == 0)
    {
        return "";
    }
    std::string::size_type position = path.length() - 1;
    for (auto k = 0; k < dotCount; ++k)
    {
        position = path.find_last_of(".", position);
        if (position == std::string::npos)
        {
            break;
        }
        position -= 1;
    }
    return (position == std::string::npos) ? "" : path.substr(position + 2);
}

////////////////////////////////////////////////////////////////////////////////
// Hash
////////////////////////////////////////////////////////////////////////////////
bool GetHashValueFromFile(uint8_t hash[], size_t hashSize, const std::string& path) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    const auto result = nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
    if (static_cast<int64_t>(hashSize) > size
        || size > static_cast<int64_t>(FcivFileBufferSize) )
    {
        return false;
    }
    char fileBuffer[FcivFileBufferSize];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, fileBuffer, static_cast<size_t>(size)));
    nn::fs::CloseFile(handle);

    auto head = fileBuffer;
    size_t count = 0;

    for (auto k = 0; k < size; ++k)
    {
        if ( 0 != std::isxdigit(static_cast<char>(fileBuffer[k])) )
        {
            ++count;
        }
        else
        {
            count = 0;
            head = &fileBuffer[k + 1];
        }
        if (count >= hashSize * 2)
        {
            break;
        }
    }
    if (count != hashSize * 2)
    {
        return false;
    }
    const std::string hashString = std::string(head).substr(0, hashSize * 2);
    for (size_t k = 0; k < hashSize; ++k)
    {
        hash[k] = static_cast<uint8_t>(std::stoi(hashString.substr(k * 2, 2), nullptr, 16));
    }
    return true;
}

std::string GetHashValueString(const uint8_t hash[], size_t hashSize) NN_NOEXCEPT
{
    std::string valueString;
    for (auto k = 0u; k < hashSize; ++k)
    {
        const size_t size = 3;
        char tmp[size];
        nn::util::SNPrintf(tmp, size, "%02x", hash[k]);
        valueString += tmp;
    }
    return valueString;
}

void Dump(void* ptr, std::size_t size, const std::size_t byteCount, bool isShowAddress)
{
    auto buffer = static_cast<const uint8_t*>(ptr);
    if (isShowAddress)
    {
        NN_LOG("%p: ", buffer);
    }
    for (auto k = 0u; k < size; ++k)
    {
        NN_LOG("%02x ", buffer[k]);
        if ( 0 == ( (k + 1) % byteCount ) )
        {
            NN_LOG("\n");
            if (isShowAddress)
            {
                if (k != size - 1)
                {
                    NN_LOG("%p: ", &buffer[k + 1]);
                }
            }
        }
    }
    NN_LOG("\n");
}

}}} // nnt::codec::util
