﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TutorialBaseClass.Windows.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This class is shared among all tutorials in order to abstract away per-operating system
 *  details related to NVN initialization and operating system integration. Each tutorial
 *  derives from the TutorialBaseClass, and provides an implementation of the t()
 *  global singleton accessor function.
 */


#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#include <nn/vi.h>

#include <mmsystem.h>
#include <exception>
#include <stdexcept>
#include <tchar.h>

#include <nn/fs.h>
#include <nn/nn_Assert.h>

#include <nvntutorial/TutorialBaseClass.h>

    // Force the NVidia GPU on Optimus systems
extern "C"
{
    _declspec(dllexport) DWORD NvOptimusEnablement = 0x00000001;
}

namespace {
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
} // namespace

class Tutorial
{
    public:
        explicit Tutorial(HINSTANCE hinstance);
        ~Tutorial();

        int Run(const CallbackFunctionType* callbackFunctions);

    private:

        static int const totalFrameTime = 16; // 60fps.
        // If there is less than this many milliseconds in a frame, just proceed to the next.
        static int const padFrameTime = 2;

        void InitWindow();
        void InitOpenGL();
        void InitNVN();

        void Draw(uint64_t millisec);

        HINSTANCE m_Hinstance;
        HWND      m_Hwnd;
        HDC       m_Hdc;
        HGLRC     m_Hglrc;

        nn::vi::Display* m_pDisplay;
        nn::vi::Layer* m_pLayer;
        std::exception_ptr m_exception;
};

Tutorial::Tutorial(HINSTANCE hinstance)
    : m_Hinstance(hinstance)
    , m_Hwnd(0)
    , m_Hdc(0)
    , m_Hglrc(0)
    , m_pDisplay(0)
    , m_pLayer(0)
{
    InitWindow();
    InitOpenGL();
    InitNVN();

    t()->Resize(1280, 720);
}

Tutorial::~Tutorial()
{
    t()->Shutdown();

    if (m_Hglrc != NULL)
    {
        wglMakeCurrent(0, 0);
        wglDeleteContext(m_Hglrc);
        m_Hglrc = 0;
    }

    if (m_Hdc != NULL)
    {
        NN_ASSERT(m_Hwnd);
        ReleaseDC(m_Hwnd, m_Hdc);
        m_Hdc = 0;
    }

    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();

    timeEndPeriod(1);
}

int Tutorial::Run(const CallbackFunctionType* callbackFunctions)
{
    MSG msg = { 0 };

    uint32_t frameTime = 0;
    uint32_t frameTimePrev = timeGetTime();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

    int frame = 0;

    while (runLoop)
    {
        bool didFrame = false;
        if (doFrame)
        {
            frameTime = timeGetTime();
            NN_ASSERT(frameTime >= frameTimePrev);
            Draw(frameTime - frameTimePrev);
            frameTimePrev = frameTime;

            doFrame = false;
            didFrame = true;
        }

        doSleep = runLoop;
        if (doSleep)
        {
                // Figure out how much time to sleep off, if any
            sleepTime = totalFrameTime - static_cast<int>(timeGetTime() - frameTime);

            if (sleepTime <= padFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

        if (doSleep)
        {
            switch (MsgWaitForMultipleObjects(
                0, 0, // No kernel object array.
                FALSE,
                sleepTime,
                QS_ALLEVENTS
                ))
            {
            case WAIT_OBJECT_0:
                break;

            case WAIT_TIMEOUT:
                doFrame = true;
                break;

            default:
            case WAIT_FAILED:
                throw std::runtime_error("MsgWaitForMultipleObjects");
            }
        }
        if(didFrame)
        {
            frame++;
            bool callbackRequestLoop = false;
            if(callbackFunctions->endFrameCallbackFunction)
            {
                CallbackFunctionType::EndFrameCallbackParameter param;
                param.currentFrame = frame;
                callbackFunctions->endFrameCallbackFunction(&callbackRequestLoop, &param, callbackFunctions->endFrameCallbackUserParam);
            }

            if(callbackRequestLoop)
            {
                runLoop = false;
            }
        }
    }

    return (static_cast<int>(msg.wParam));
}

void Tutorial::InitWindow()
{
    nn::vi::Initialize();

    if (timeBeginPeriod(1) != TIMERR_NOERROR)
    {
        throw std::runtime_error("timeBeginPeriod");
    }

    nn::vi::OpenDefaultDisplay(&m_pDisplay);

    nn::vi::CreateLayer(&m_pLayer, m_pDisplay);

    nn::vi::GetNativeWindow(reinterpret_cast<nn::vi::NativeWindowHandle*>(&m_Hwnd), m_pLayer);

    if (!SetWindowLongPtr(m_Hwnd, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(this)) &&
        GetLastError())
    {
        throw std::runtime_error("SetWindowLongPtr");
    }
}

void Tutorial::InitOpenGL()
{
    NN_ASSERT(m_Hwnd && !m_Hdc);
    m_Hdc = GetDC(m_Hwnd);

    if (m_Hdc == NULL)
    {
        throw std::runtime_error("GetDC");
    }

    PIXELFORMATDESCRIPTOR pfd = { 0 };
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 32;
    int selectedPixelFormat = ChoosePixelFormat(m_Hdc, &pfd);

    if (!SetPixelFormat(m_Hdc, selectedPixelFormat, &pfd))
    {
        throw std::runtime_error("SetPixelFormat");
    }

    m_Hglrc = wglCreateContext(m_Hdc);

    if (m_Hglrc == NULL)
    {
        throw std::runtime_error("wglCreateContext");
    }

        // A current OpenGL context is required on the thread that will call
        // nvnQueuePresentTexture in Windows.
    if (!wglMakeCurrent(m_Hdc, m_Hglrc))
    {
        throw std::runtime_error("wglMakeCurrent");
    }
}

void Tutorial::InitNVN()
{
        // The NVN loader is exposed as a obfuscated OpenGL extension in Windows
    t()->Init(reinterpret_cast<PFNNVNBOOTSTRAPLOADERPROC>(wglGetProcAddress("rq34nd2ffz")), m_Hwnd);
}

void Tutorial::Draw(uint64_t millisec)
{
    t()->Draw(millisec);

        // In Windows, it's necessary to swap the buffers of the window; nvnQueuePresentTexture
        // only does a blit in to the window back buffer.
    //if (!SwapBuffers(m_Hdc))
    //{
    //    throw std::runtime_error("SwapBuffers");
    //}
}

void TutorialStartup()
{
}

void TutorialRun(const CallbackFunctionType* callbackFunctions)
{
    HINSTANCE hInstance = GetModuleHandle(NULL);
    nn::fs::SetAllocator(Allocate, Deallocate);
    Tutorial tutorial(hInstance);
    tutorial.Run(callbackFunctions);
    return;
}
