﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{RenderThreadPool.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that handles running
 *  multiple threads in parallel that create command
 *  buffers to render objects from a work queue.
 */

#include <nn/nn_Assert.h>
#include <nvntutorial/RenderThreadPool.h>
#include <nvntutorial/GraphicsObject.h>
#include <nvntutorial/TutorialUtil.h>
static const size_t g_CommandMemorySize = 128 * 1024;
static const size_t g_ControlMemorySize = 32 * 1024;

static const size_t g_CommandMemoryChunkSize = 1024;
static const size_t g_ControlMemoryChunkSize = 1024;

static const size_t g_ThreadStackSize = 64 * 1024;
static const size_t g_ThreadStackAlignment = 4096;

void RenderThread(void* pParam);

/*
 * RenderThreadPool Constructor
 * ----------------------------
 * Sets up the mutexes for getting work and writing
 * output. Sets other members to initial values.
 */
RenderThreadPool::RenderThreadPool() :
    m_pWorkQueue(NULL),
    m_pOutput(NULL),
    m_WorkIndex(0),
    m_pThreadStackMemory(NULL),
    m_Running(false)
{
}

/*
 * RenderThreadPool Destructor
 * ---------------------------
 * Cleans up the mutexes and all data created
 * for the threads.
 */
RenderThreadPool::~RenderThreadPool()
{
    nn::os::FinalizeMutex(&m_WorkMutex);
    nn::os::FinalizeMutex(&m_WriteMutex);

    Stop();
    for (size_t i = 0; i < m_ThreadStateMutex.size(); ++i)
    {
        nn::os::FinalizeMutex(&m_ThreadStateMutex[i]);
    }

    Clean();
}

/*
 * RenderThreadPool::Clean
 * -----------------------
 * Cleans up the per thread command buffer data
 * and thread stack memory.
 */
void RenderThreadPool::Clean()
{
    if(m_CommandBuffers.size() > 0)
    {
        for(size_t i = 0; i < m_CommandBuffers.size(); ++i)
        {
            nvnCommandBufferFinalize(&m_CommandBuffers[i].m_CommandBuffer);

            m_CommandBuffers[i].m_MemoryPool->Shutdown();
            delete m_CommandBuffers[i].m_MemoryPool;
            m_CommandBuffers[i].m_MemoryPool = NULL;

            if (m_CommandBuffers[i].m_pControlPool)
            {
                AlignedDeallocate(m_CommandBuffers[i].m_pControlPool);
                m_CommandBuffers[i].m_pControlPool = NULL;
            }

            if (m_CommandBuffers[i].m_MemoryManager.m_commandMemoryManager)
            {
                delete m_CommandBuffers[i].m_MemoryManager.m_commandMemoryManager;
                m_CommandBuffers[i].m_MemoryManager.m_commandMemoryManager = NULL;
            }

            if (m_CommandBuffers[i].m_MemoryManager.m_controlMemoryManager)
            {
                delete m_CommandBuffers[i].m_MemoryManager.m_controlMemoryManager;
                m_CommandBuffers[i].m_MemoryManager.m_controlMemoryManager = NULL;
            }

            if (m_CommandBuffers[i].m_pCompletionTracker)
            {
                delete m_CommandBuffers[i].m_pCompletionTracker;
                m_CommandBuffers[i].m_pCompletionTracker = NULL;
            }
        }

        m_CommandBuffers.clear();
    }

    if(m_pThreadStackMemory != NULL)
    {
        AlignedDeallocate(m_pThreadStackMemory);
        m_pThreadStackMemory = NULL;
    }
}

/*
 * RenderThreadPool::Init
 * ----------------------
 * Sets up the per thread command buffer data and
 * allocates memory for the per thread stacks.
 */
void RenderThreadPool::Init(int numThreads, NVNdevice* pDevice)
{
    Clean();

    m_CommandBuffers.reserve(numThreads);

        /* Grabs the command and control memory alignment. */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(pDevice, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /*
         * Creates a command buffer per worker thread. Each
         * command buffer will be used to generate multiple
         * sets of commands to draw individual objects from
         * the work queue.
         */
    for(int i = 0; i < numThreads; ++i)
    {
        m_CommandBuffers.push_back(CommandBufferData());

        CommandBufferData& commandBufferData = m_CommandBuffers.back();

        if (!nvnCommandBufferInitialize(&commandBufferData.m_CommandBuffer, pDevice))
        {
            NN_ASSERT(0, "nvnCommandBufferInitialize failed");
        }

        commandBufferData.m_MemoryPool = new MemoryPool();

        commandBufferData.m_MemoryPool->Init(NULL,
                                             Align(g_CommandMemorySize, commandBufferCommandAlignment),
                                             NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                             pDevice);

        commandBufferData.m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        commandBufferData.m_pCompletionTracker = new CompletionTracker(pDevice, 16);
        commandBufferData.m_MemoryManager.m_commandMemoryManager = new TrackedCommandMemRingBuffer(&commandBufferData.m_CommandBuffer, commandBufferData.m_MemoryPool->GetMemoryPool(), commandBufferData.m_pCompletionTracker, g_CommandMemorySize, 0, g_CommandMemoryChunkSize, g_CommandMemoryChunkSize, 1);
        commandBufferData.m_MemoryManager.m_controlMemoryManager = new TrackedControlMemRingBuffer(&commandBufferData.m_CommandBuffer, commandBufferData.m_pCompletionTracker, g_ControlMemorySize, reinterpret_cast<char*>(commandBufferData.m_pControlPool), g_ControlMemoryChunkSize, g_ControlMemoryChunkSize, 1);

        commandBufferData.m_MemoryManager.m_commandMemoryManager->setupNewChunk();
        commandBufferData.m_MemoryManager.m_controlMemoryManager->setupNewChunk();

        nvnCommandBufferSetMemoryCallback(&commandBufferData.m_CommandBuffer, outOfMemory);
        CommandBufferMemoryManager* temp = &commandBufferData.m_MemoryManager;
        nvnCommandBufferSetMemoryCallbackData(&commandBufferData.m_CommandBuffer, temp);
    }

        /* Allocate memory for the per thread stacks. */
    m_ThreadHandles.resize(numThreads);
    if(m_pThreadStackMemory == NULL)
    {
        m_pThreadStackMemory = reinterpret_cast<char*>(AlignedAllocate(g_ThreadStackSize * numThreads, g_ThreadStackAlignment));
    }

        /*
         * Initialize mutexes for grabbing new work items and writing
         * command handle outputs
         */
    nn::os::InitializeMutex(&m_WorkMutex, false, 0);
    nn::os::InitializeMutex(&m_WriteMutex, false, 0);

        /* Initialize thread data and mutexes for checking thread state */
    m_ThreadData.resize(numThreads);
    m_ThreadStateMutex.resize(numThreads);
    for (int i = 0; i < numThreads; ++i)
    {
        ThreadData& threadData = m_ThreadData[i];
        threadData.m_pParentPool = this;
        threadData.m_pCommandBufferData = &m_CommandBuffers[i];
        threadData.m_ThreadState = TUTORIAL_THREAD_STATE_NUM_STATES;
        nn::os::InitializeMutex(&m_ThreadStateMutex[i], false, 0);
        threadData.m_ThreadStateCheckMutex = &m_ThreadStateMutex[i];
    }
}

/*
 * RenderThreadPool::Run
 * ---------------------
 * Runs the worker threads with the given work to be done
 * and saves the command handles in the given output buffer.
 */
void RenderThreadPool::Run(std::vector<NVNcommandHandle>* pOutput, std::vector<GraphicsObject*>* pWork, NVNqueue* pQueue)
{
    NN_ASSERT(pWork != NULL || pOutput != NULL, "Work and/or output not provided to thread pool.");
        /* Insert a fence for the command buffer. */
    for(size_t i = 0; i < m_CommandBuffers.size(); ++i)
    {
        m_CommandBuffers[i].m_pCompletionTracker->insertFence(pQueue);
    }

    m_WorkIndex = 0;
    m_pWorkQueue = pWork;
    m_pOutput = pOutput;

    m_pOutput->clear();
    m_pOutput->resize(m_pWorkQueue->size());
    nn::Result result;

        /* Set thread data to indicate the threads should run */
    for (size_t i = 0; i < m_ThreadStateMutex.size(); ++i)
    {
        nn::os::LockMutex(&m_ThreadStateMutex[i]);
        m_ThreadData[i].m_ThreadState = TUTORIAL_THREAD_STATE_RUN;
        nn::os::UnlockMutex(&m_ThreadStateMutex[i]);
    }

        /* Set threads to actually start if this is the first run */
    if (m_Running == false)
    {
            /* Create and run threads with the command buffer data from Init. */
        for (size_t i = 0; i < m_ThreadHandles.size(); ++i)
        {
            result = nn::os::CreateThread(&m_ThreadHandles[i],                          /* Address of ThreadType */
                                          RenderThread,                                 /* Function pointer to call */
                                          &m_ThreadData[i],                             /* Data to pass */
                                          m_pThreadStackMemory + i * g_ThreadStackSize, /* Stack memory for thread to use */
                                          g_ThreadStackSize,                            /* Size of stack */
                                          nn::os::DefaultThreadPriority);               /* Priority */

            NN_ASSERT(result.IsSuccess());
            nn::os::StartThread(&m_ThreadHandles[i]);
        }

        m_Running = true;
    }
}

/*
* RenderThreadPool::Wait
* ----------------------
* Wait for current frame of rendering to finish, must be called
* after Run
*/
void RenderThreadPool::Wait()
{
    bool running = true;
    while (running)
    {
        running = false;
        for (size_t i = 0; i < m_ThreadStateMutex.size(); ++i)
        {
            nn::os::LockMutex(&m_ThreadStateMutex[i]);
            TutorialThreadState state = m_ThreadData[i].m_ThreadState;

                /*
                 * Check if thread is still running, continue waiting
                 * if a thread is still running
                 */
            if (state == TUTORIAL_THREAD_STATE_RUN)
            {
                running = true;
                nn::os::UnlockMutex(&m_ThreadStateMutex[i]);

                /*
                 *	NOTE: Since threads are run to completion on NX, a call to YieldThread
                 *        is necessary in order to allow other threads a chance to run
                 */
                nn::os::YieldThread();
                break;
            }
            nn::os::UnlockMutex(&m_ThreadStateMutex[i]);
        }
    }
}

/*
* RenderThreadPool::Stop
* ----------------------
* Stop render thread pool from running and destroy threads
*/
void RenderThreadPool::Stop()
{
        /* Set thread data to indicate the threads should stop if they are running */
    for (size_t i = 0; i < m_ThreadStateMutex.size(); ++i)
    {
        nn::os::LockMutex(&m_ThreadStateMutex[i]);
        m_ThreadData[i].m_ThreadState = TUTORIAL_THREAD_STATE_STOP;
        nn::os::UnlockMutex(&m_ThreadStateMutex[i]);
    }

        /* Wait for the threads to be done running. */
    for(size_t i = 0; i < m_ThreadHandles.size(); ++i)
    {
        nn::os::WaitThread(&m_ThreadHandles[i]);
    }

        /* Destroy the threads. */
    for(size_t i = 0; i < m_ThreadHandles.size(); ++i)
    {
        nn::os::DestroyThread(&m_ThreadHandles[i]);
    }

    m_Running = false;
}

/*
 * RenderThreadPool::GetNextWorkItem
 * ---------------------------------
 * Atomically grabs the next available work item
 * from the work queue.
 */
std::pair<int, GraphicsObject*> RenderThreadPool::GetNextWorkItem()
{
    nn::os::LockMutex(&m_WorkMutex);

    std::pair<int, GraphicsObject*> res;

        /* If there's no work left, return null data. */
    if(static_cast<size_t>(m_WorkIndex) >= m_pWorkQueue->size())
    {
        res.first = m_WorkIndex;
        res.second = NULL;

        nn::os::UnlockMutex(&m_WorkMutex);

        return res;
    }

        /* Get the work item at the current index. */
    res.first = m_WorkIndex;
    res.second = (*m_pWorkQueue)[m_WorkIndex];

        /* Increment the current work index. */
    ++m_WorkIndex;

    nn::os::UnlockMutex(&m_WorkMutex);

        /* Return the data to be processed. */
    return res;
}

/*
 * RenderThreadPool::WriteThreadOutput
 * -----------------------------------
 * Atomically write the resulting command handle from
 * from a work item at the given index.
 */
void RenderThreadPool::WriteThreadOutput(int index, const NVNcommandHandle& handle)
{
    nn::os::LockMutex(&m_WriteMutex);

    (*m_pOutput)[index] = handle;

    nn::os::UnlockMutex(&m_WriteMutex);
}

/*
 * RenderThread
 * ------------
 * This is the funstion that each work thread runs.
 * The function runs until there are no more work
 * items left in the queue to process.
 */
void RenderThread(void* pParam)
{
    ThreadData* data = reinterpret_cast<ThreadData*>(pParam);

        /* Grab the command buffer data. */
    CommandBufferData* commandBufferData = data->m_pCommandBufferData;

    bool run = true;
    while(run)
    {
            /* Check if thread's state has changed */
        nn::os::LockMutex(data->m_ThreadStateCheckMutex);
        TutorialThreadState state = data->m_ThreadState;
        nn::os::UnlockMutex(data->m_ThreadStateCheckMutex);

            /* Determine thread behavior based on state */
        switch (state)
        {
        case TUTORIAL_THREAD_STATE_RUN:
            {
                    /* Get the first work item. */
                std::pair<int, GraphicsObject*> currentWorkItem = data->m_pParentPool->GetNextWorkItem();

                    /* Run while there is still work to be done. */
                while (currentWorkItem.second != NULL)
                {
                        /* Update the object's uniform buffers. */
                    currentWorkItem.second->UpdateUniforms();

                        /* Start recording commands to draw the object. */
                    nvnCommandBufferBeginRecording(&commandBufferData->m_CommandBuffer);

                        /* Bind the render states for the object. */
                    currentWorkItem.second->BindState(&commandBufferData->m_CommandBuffer);

                        /* Draw the object. */
                    currentWorkItem.second->Draw(&commandBufferData->m_CommandBuffer);

                        /*
                        * Grab the command handle for the newly recorded
                        * commands and write it to the output list at the
                        * same index as the work item.
                        */
                    NVNcommandHandle result = nvnCommandBufferEndRecording(&commandBufferData->m_CommandBuffer);
                    data->m_pParentPool->WriteThreadOutput(currentWorkItem.first, result);

                        /* Grab the next available work item. */
                    currentWorkItem = data->m_pParentPool->GetNextWorkItem();
                }

                    /* Put thread to Wait state */
                nn::os::LockMutex(data->m_ThreadStateCheckMutex);
                TutorialThreadState& currentState = data->m_ThreadState;
                if (currentState == TUTORIAL_THREAD_STATE_RUN)
                {
                    currentState = TUTORIAL_THREAD_STATE_WAIT;
                }
                nn::os::UnlockMutex(data->m_ThreadStateCheckMutex);
            }
            break;

        case TUTORIAL_THREAD_STATE_WAIT:
                /*
                 * Nothing to do so yield and wait until state changes
                 */
            nn::os::YieldThread();
            break;

        case TUTORIAL_THREAD_STATE_STOP:
                /* Rendering is stopped; break out of loop and clean up */
            run = false;
            break;

        default:
            NN_ASSERT(false, "Unknown thread state encountered in render thread pool\n");
        }
    }
}
