﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{MemoryPool.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a wrapper around the NVNmemoryPool
 *  object that keeps track of the memory allocated for the
 *  pool, the size of the pool, and the offset into the pool
 *  at which the next memory write could take place. This class
 *  also handles initializing the NVNmemoryPool and makes sure
 *  the memory alignment is handled properly.
 */

#include <nvn/nvn_FuncPtrInline.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nvntutorial/MemoryPool.h>
#include <nvntutorial/TutorialUtil.h>

MemoryPool::MemoryPool() : m_CurrentWriteOffset(0),
                           m_pMemory(NULL),
                           m_Size(0)
{
}

MemoryPool::~MemoryPool()
{
    if (m_pMemory != NULL)
    {
        Shutdown();
    }
}

void MemoryPool::Init(void* pMemory, size_t size, int flags, NVNdevice* pDevice)
{
    m_pMemory = pMemory;

    if (size < g_MinimumPoolSize)
    {
        size = g_MinimumPoolSize;
    }

        /* Align the pool size to the proper alignment. */
    m_Size = Align(size, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

        /* Allocate memory of the aligned size at the correct address alignment. */
    if (m_pMemory == NULL)
    {
        m_pMemory = AlignedAllocate(m_Size, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    }

        /*
        * NVN Memory Pool
        * ---------------
        * The NVNmemoryPool manages a block of GPU memory that is used to allocate
        * NVNbuffer, NVNtexture, etc. objects. A memory pool must specify an access
        * flag for the CPU and GPU. If the memory pool will be used to hold executable
        * shader code, the SHADER_CODE flag must also be set.
        *
        * CPU_NO_ACCESS     - Memory won't be accessed by the CPU.
        *
        * CPU_UNCACHED      - Memory is accessible by the CPU, but won't be cached. Useful
        *                     for sequential writes, slow for reading an non-sequential
        *                     writes. Memory access will be coherent with the GPU.
        *
        * CPU_CACHED        - Memory is accessible by the CPU and will be cached by the CPU.
        *                     Reads/writes will be fast, but caching needs to be managed
        *                     manually. Reading from the pool may not reflect previous GPU
        *                     commands until the cached CPU memory has been invalidated.
        *                     Similarly, values written may not be available to the GPU
        *                     until the writes have been flushed.
        *
        * GPU_NO_ACCESS     - Memory won't be accessedby the GPU.
        *
        * GPU_UNCACHED      - Memory is accessible by the GPU, but not stored in GPU caches.
        *                     Useful for single use command buffers or one time writes, slow
        *                     for repeated/random access by the CPU.
        *
        * GPU_CACHED        - Memory is accessible by the GPU and stored in GPU caches.
        *
        * COMPRESSIBLE_BIT  - The memory pool can be used to allocate compressible textures.
        *                     Attempting to allocate compressed textures in a memory pool
        *                     WITHOUT this bit causes an error.
        *
        * PHYSICAL_BIT      - The pool can only be used as physical storage for memory mappings
        *                     into virtual memory pools
        *
        * VIRTUAL_BIT       - The pool can be used for sparse operations like buffer and
        *                     and textures.  This pool type must be backed by physical memory
        *                     from a memory pool with the above bit.
        *
        * The base address of the memory used by the pool needs to be aligned to the value
        * defined in the macro NVN_MEMORY_POOL_STORAGE_ALIGNMENT.  Its size must be a
        * multiple of the value defined in NVN_MEMORY_POOL_STORAGE_GRANULARITY. Different
        * resources require various different alignments within the pool itself. These
        * requirements are outlined in the memory pool section of the NVN programming guide.
        */
    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, pDevice);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder, flags);
    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    if (nvnMemoryPoolInitialize(&m_MemoryPool, &poolBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to initialize buffer memory pool");
    }

    m_Flags = flags;
}

void MemoryPool::Shutdown()
{
    if (m_pMemory != NULL)
    {
        m_Size = 0;
        m_CurrentWriteOffset.store(0);

        nvnMemoryPoolFinalize(&m_MemoryPool);

        AlignedDeallocate(m_pMemory);
        m_pMemory = NULL;
    }
}

ptrdiff_t MemoryPool::GetNewMemoryChunkOffset(size_t size, size_t alignment)
{
    if (static_cast<size_t>(m_CurrentWriteOffset.load()) >= m_Size)
    {
        NN_ASSERT(0, "Memory pool out of memory.");
    }

    m_CurrentWriteOffset.store(Align(m_CurrentWriteOffset.load(), alignment));

    ptrdiff_t dataOffset = m_CurrentWriteOffset.load();

    m_CurrentWriteOffset.fetch_add(size);

    return dataOffset;
}

NVNmemoryPool* MemoryPool::GetMemoryPool()
{
    return &m_MemoryPool;
}
