﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AlignedAllocator.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a simple aligned allocator
 *  that takes care of keeping track of the original
 *  allocation while returning the aligned pointer.
 */

#include <nvntutorial/AlignedAllocator.h>
#include <nvntutorial/TutorialUtil.h>

/*
 * AlignedAllocator Constructor
 * ----------------------------
 * Empty default constructor.
 */
AlignedAllocator::AlignedAllocator()
    : m_Memory(NULL)
{
    m_Memory = std::malloc(64 * 1024 * 1024);
    NN_ASSERT_NOT_NULL(m_Memory);
    m_Allocator.Initialize(m_Memory, 64 * 1024 * 1024);
}

/*
 * AlignedAllocator Destructor
 * ---------------------------
 * Asserts if any allocations haven't been freed.
 */
AlignedAllocator::~AlignedAllocator()
{
    m_Allocator.Finalize();
    std::free(m_Memory);
    m_Memory = NULL;
}

/*
 * AlignedAllocator::Allocate
 * ---------------------------
 * Allocates a block of memory and returns a pointer
 * aligned to the given alignment.
 */
void* AlignedAllocator::Allocate(size_t size, size_t alignment)
{
    return m_Allocator.Allocate(size, alignment);
}

/*
 * AlignedAllocator::Deallocate
 * ---------------------------
 * Frees a block of aligned memory allocated by the
 * above function.
 */
void AlignedAllocator::Deallocate(void* pData)
{
    m_Allocator.Free(pData);
}
