﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;
using System.Runtime.InteropServices;

using Nintendo.Htcs;

// cs の動作を確かめる簡易的なツールです
// あらかじめ TargetManager を起動して Target に接続しておく必要があります

namespace CsClient
{
    internal class Program
    {
        private const string TargetServerPortName = "iywys@$csForRunnerTools";

        private static HtcsCommunicator m_HtcsCommunicator = new HtcsCommunicator();
        private static TargetList m_TargetList;
        private static PortMap m_PortMap;
        private static AutoResetEvent m_FindServerEvent = new AutoResetEvent(false);

        private static void CsClientSession(CsClientParams param)
        {
            // HTCS ポート名が TargetServerPortName で表されるサーバが開くまで待機
            m_FindServerEvent.WaitOne();

            // HTCS ポート名が TargetServerPortName のもののうち、最初に見つかったものを取得
            //   複数のターゲットが同名の HTCS ポートを開いている場合は、
            //   PortMapItem.HtcsPortDescriptor.HtcsPortName で取得できるターゲット名で識別してください
            PortMapItem portMapItem = m_PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == TargetServerPortName);

            // TCP ソケットを作成、EndPoint 情報を基に Connect
            TcpClient client = new TcpClient();
            client.Connect(portMapItem.EndPoint);
            Console.WriteLine("Connected to server on target.");

            using (var writer = new BinaryWriter(client.GetStream()))
            using (var reader = new BinaryReader(client.GetStream()))
            {
                // コマンドを送信
                Console.WriteLine("--  Command --");
                CommandUtils.Send(writer, param);
                Console.WriteLine("---------------");

                // レスポンスを受信・表示
                do
                {
                    Console.WriteLine("-- Response --");
                    ResponseUtils.Receive(reader);
                    Console.WriteLine("---------------");
                } while (param.loop);
            }

            // ソケットをクローズ
            client.Close();
            Console.WriteLine("Closed.");
        }

        private static void CheckHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            // HtcsInfo のキャッシュを更新
            m_TargetList = e.TargetList;
            m_PortMap = e.PortMap;

            // ポート名が TargetServerPortName のポートが開かれていたら通知を投げる
            if (e.PortMap.Any(x => x.HtcsPortDescriptor.HtcsPortName == TargetServerPortName))
            {
                m_FindServerEvent.Set();
            }
        }

        private static void PrintHtcsInfo(object sender, HtcsInfoUpdatedEventArgs e)
        {
            Console.WriteLine("-  TargetList -");
            foreach (Target item in e.TargetList)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");

            Console.WriteLine("--- PortMap ---");
            foreach (PortMapItem item in e.PortMap)
            {
                Console.WriteLine(item.ToString());
            }
            Console.WriteLine("---------------");
        }

        private static void Main(string[] args)
        {
            CsClientArgs opt = new CsClientArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            // HtcsInfo が更新されたときに呼ばれる関数を登録
            m_HtcsCommunicator.HtcsInfoUpdated += PrintHtcsInfo;
            m_HtcsCommunicator.HtcsInfoUpdated += CheckHtcsInfo;

            // TargetManager への接続が確立されるまでループ
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.WriteLine("Failed connecting to TargetManager.");
                Console.WriteLine("Press Enter to retry.");
                Console.ReadLine();
            }

            m_HtcsCommunicator.RequestSystemPort();

            CsClientSession(opt.Params);

            m_HtcsCommunicator.Stop();
        }
    }
}
