﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

// テスト対象
#include "testsf_IHipcMapTransferAttributeTests.h"

// hipc
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_HipcClient.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <new>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/svc/svc_Base.h>

namespace {

class MyHipcServerManager
    : public nn::sf::HipcSimpleAllInOneServerManager<100, 1>
{
public:

    static void Loop(MyHipcServerManager* p) NN_NOEXCEPT
    {
        p->LoopAuto();
    }

};

bool g_ImplInvoked;

class HipcMapTransferAttributedObjectImpl
{
private:

    static nn::svc::MemoryInfo GetMemoryInfo(const void* p) NN_NOEXCEPT
    {
        nn::svc::MemoryInfo memoryInfo;
        nn::svc::PageInfo pageInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::QueryMemory(&memoryInfo, &pageInfo, reinterpret_cast<uintptr_t>(p)));
        return memoryInfo;
    }

public:

    nn::Result Secure(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        g_ImplInvoked = true;
        EXPECT_EQ(nn::svc::MemoryState_Ipc, GetMemoryInfo(inBuffer.GetPointerUnsafe()).state);
        EXPECT_EQ(nn::svc::MemoryState_Ipc, GetMemoryInfo(outBuffer.GetPointerUnsafe()).state);
        NN_RESULT_SUCCESS;
    }

    nn::Result NonSecure(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        g_ImplInvoked = true;
        EXPECT_EQ(nn::svc::MemoryState_NonSecureIpc, GetMemoryInfo(inBuffer.GetPointerUnsafe()).state);
        EXPECT_EQ(nn::svc::MemoryState_NonSecureIpc, GetMemoryInfo(outBuffer.GetPointerUnsafe()).state);
        NN_RESULT_SUCCESS;
    }

    nn::Result NonDevice(const nn::sf::InBuffer& inBuffer, const nn::sf::OutBuffer& outBuffer) NN_NOEXCEPT
    {
        g_ImplInvoked = true;
        EXPECT_EQ(nn::svc::MemoryState_NonDeviceIpc, GetMemoryInfo(inBuffer.GetPointerUnsafe()).state);
        EXPECT_EQ(nn::svc::MemoryState_NonDeviceIpc, GetMemoryInfo(outBuffer.GetPointerUnsafe()).state);
        NN_RESULT_SUCCESS;
    }

};

NN_OS_ALIGNAS_THREAD_STACK char g_ServerStack[nn::os::StackRegionAlignment * 4];
nnt::sfutil::TestThread g_ServerThread;
const char ServiceName[] = "sf_test1";
std::aligned_storage<sizeof(MyHipcServerManager), NN_ALIGNOF(MyHipcServerManager)>::type g_HipcServerManagerStorage;

nn::sf::SharedPointer<nnt::testsf::IHipcMapTransferAttributedObject> GetHipcProxyObjectByName(const char* serviceName) NN_NOEXCEPT
{
    NN_SF_RESULT_AND_TRY(ret, (nn::sf::CreateHipcProxyByName<nnt::testsf::IHipcMapTransferAttributedObject, nn::sf::DefaultAllocationPolicy>(serviceName)))
        NN_RESULT_CATCH_ALL
        {
            EXPECT_TRUE(false);
        }
    NN_SF_RESULT_AND_END_TRY
    return ret;
}

nn::sf::SharedPointer<nnt::testsf::IWrongHipcMapTransferAttributedObject> GetWrongHipcProxyObjectByName(const char* serviceName) NN_NOEXCEPT
{
    NN_SF_RESULT_AND_TRY(ret, (nn::sf::CreateHipcProxyByName<nnt::testsf::IWrongHipcMapTransferAttributedObject, nn::sf::DefaultAllocationPolicy>(serviceName)))
        NN_RESULT_CATCH_ALL
        {
            EXPECT_TRUE(false);
        }
    NN_SF_RESULT_AND_END_TRY
    return ret;
}

TEST(sf_AllInOne, HipcMapTransferAttribute)
{
    nn::sf::UnmanagedServiceObject<nnt::testsf::IHipcMapTransferAttributedObject, HipcMapTransferAttributedObjectImpl> o;
    auto pServer = new (&g_HipcServerManagerStorage) MyHipcServerManager;
    NN_UTIL_SCOPE_EXIT
    {
        pServer->~MyHipcServerManager();
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(pServer->RegisterObjectForPort(o.GetShared(), 10, ServiceName));
    pServer->Start();
    g_ServerThread.Initialize(MyHipcServerManager::Loop, pServer, g_ServerStack, sizeof(g_ServerStack), 10);
    g_ServerThread.Start();

    // 呼び出し先で正しく属性がセットされていることのテスト
    {
        auto p = GetHipcProxyObjectByName(ServiceName);
        char inBuffer[1];
        char outBuffer[1];
        {
            g_ImplInvoked = false;
            EXPECT_TRUE(p->Secure(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(g_ImplInvoked);
        }
        {
            g_ImplInvoked = false;
            EXPECT_TRUE(p->NonSecure(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(g_ImplInvoked);
        }
        {
            g_ImplInvoked = false;
            EXPECT_TRUE(p->NonDevice(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(g_ImplInvoked);
        }
    }

    // サーバ側と齟齬のある属性で呼んだ時に、実装が実行されずに、エラーが返ることのテスト
    {
        char inBuffer[1];
        char outBuffer[1];
        {
            auto p = GetWrongHipcProxyObjectByName(ServiceName);
            g_ImplInvoked = false;
            EXPECT_TRUE(!p->Secure(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(!g_ImplInvoked);
        }
        {
            auto p = GetWrongHipcProxyObjectByName(ServiceName);
            g_ImplInvoked = false;
            EXPECT_TRUE(!p->NonSecure(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(!g_ImplInvoked);
        }
        {
            auto p = GetWrongHipcProxyObjectByName(ServiceName);
            g_ImplInvoked = false;
            EXPECT_TRUE(!p->NonDevice(nn::sf::InBuffer(inBuffer, 1), nn::sf::OutBuffer(outBuffer, 1)).IsSuccess());
            EXPECT_TRUE(!g_ImplInvoked);
        }
    }

    pServer->RequestStop();
    g_ServerThread.Finalize();
}

}
