﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testSf_ContextControlCommon.h"

#include <nnt/nntest.h>

// テスト対象
#include "testSf_IBlockingMap.h"

// hipc
#include <nn/sf/sf_HipcClient.h>

// 実装
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nnt/sfutil/sfutil_ThreadUtils.h>
#include <nnt/sfutil/sfutil_CounterSynchronizer.h>

namespace nnt { namespace testsf {

namespace {

struct MyAllocatorTag
{
};

typedef nn::sf::ExpHeapStaticAllocator<1024 * 1024, MyAllocatorTag> MyAllocator;

class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyAllocatorInitializer;

NN_OS_ALIGNAS_THREAD_STACK char g_SetterStack[nn::os::StackRegionAlignment * 4];
NN_OS_ALIGNAS_THREAD_STACK char g_GetterStack[nn::os::StackRegionAlignment * 4];

nnt::sfutil::TestThread g_SetterThread;
nnt::sfutil::TestThread g_GetterThread;

enum class TestState
{
    Initial,

    NotYetSetK1,
    AfterTryGetK1Failed,

    BeforeSetK1V1,
    AfterSetK1V1,

    BeforeGetK1V1,
    AfterGetK1V1,

    NotYetSetK2,

    BeforeSetK2V2,
    AfterSetK2V2,

    AfterGetK2V2,

    End,
};

nnt::sfutil::CounterSynchronizer<TestState> g_CounterSynchronizer;

void SetterThreadFunction(const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nnt::testsf::IBlockingMap> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS((nn::sf::CreateHipcProxyByName<nnt::testsf::IBlockingMap, MyAllocator::Policy>(&p, serviceName)));

    g_CounterSynchronizer.Synchronize(TestState::BeforeSetK1V1);
    {
        auto result = p->SetKeyValue(1, 1);
        EXPECT_TRUE(result.IsSuccess());
    }
    g_CounterSynchronizer.Synchronize(TestState::AfterSetK1V1);

    g_CounterSynchronizer.Synchronize(TestState::BeforeSetK2V2);
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    {
        // getter が待っているのとは別のキーを登録しても問題ないことのテスト
        // この場合 Resume されたリクエストが、再度 Suspend されるはず
        auto result = p->SetKeyValue(3, 3);
        EXPECT_TRUE(result.IsSuccess());
    }
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
    {
        auto result = p->SetKeyValue(2, 2);
        EXPECT_TRUE(result.IsSuccess());
    }
    g_CounterSynchronizer.Synchronize(TestState::AfterSetK2V2);
}

void GetterThreadFunction(const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nnt::testsf::IBlockingMap> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS((nn::sf::CreateHipcProxyByName<nnt::testsf::IBlockingMap, MyAllocator::Policy>(&p, serviceName)));

    g_CounterSynchronizer.Synchronize(TestState::NotYetSetK1);
    {
        // 未登録のキーに対して TryGet が失敗することをテスト
        bool valid = false;
        int32_t value = 0;
        auto result = p->TryGetValue(&valid, &value, 1);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_TRUE(!valid);
    }
    g_CounterSynchronizer.Synchronize(TestState::AfterTryGetK1Failed);

    g_CounterSynchronizer.Synchronize(TestState::BeforeGetK1V1);
    {
        // 登録されているキーに対して TryGet が成功することのテスト
        bool valid = false;
        int32_t value = 0;
        auto result = p->TryGetValue(&valid, &value, 1);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_TRUE(valid);
        EXPECT_EQ(1, value);
    }
    {
        // 登録されているキーに対して Get が成功することのテスト
        int32_t value = 0;
        auto result = p->GetValue(&value, 1);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(1, value);
    }
    g_CounterSynchronizer.Synchronize(TestState::AfterGetK1V1);

    g_CounterSynchronizer.Synchronize(TestState::NotYetSetK2);
    {
        // 登録されていないキーに対して Get が登録されるまでブロックし成功することのテスト
        int32_t value = 0;
        auto result = p->GetValue(&value, 2);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_EQ(2, value);
    }
    {
        // key=2 が取得できていたら key=3 も無条件で取得できる
        bool valid = false;
        int32_t value = 0;
        auto result = p->TryGetValue(&valid, &value, 3);
        EXPECT_TRUE(result.IsSuccess());
        EXPECT_TRUE(valid);
        EXPECT_EQ(3, value);
    }
    g_CounterSynchronizer.Synchronize(TestState::AfterGetK2V2);
}

} // namespace

nn::sf::SharedPointer<nnt::testsf::IBlockingMap> MakeBlockingMapViaHipc(const char* serviceName) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nnt::testsf::IBlockingMap> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS((nn::sf::CreateHipcProxyByName<nnt::testsf::IBlockingMap, MyAllocator::Policy>(&p, serviceName)));
    return p;
}

void RunContextControlClients(const char* serviceName) NN_NOEXCEPT
{
    g_SetterThread.Initialize(&SetterThreadFunction, serviceName, g_SetterStack, sizeof(g_SetterStack), 10);
    g_GetterThread.Initialize(&GetterThreadFunction, serviceName, g_GetterStack, sizeof(g_GetterStack), 10);
    g_SetterThread.Start();
    g_GetterThread.Start();

    for (auto i = 1; i < static_cast<int>(TestState::End); ++i)
    {
        g_CounterSynchronizer.Set(static_cast<TestState>(i));
    }

    g_GetterThread.Finalize();
    g_SetterThread.Finalize();
}

}}
