﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nn/settings/system/settings_Account.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/system/settings_Backlight.h>
#include <nn/settings/system/settings_BluetoothDevices.h>
#include <nn/settings/system/settings_Capture.h>
#include <nn/settings/system/settings_Clock.h>
#include <nn/settings/system/settings_DataDeletion.h>
#include <nn/settings/system/settings_ErrorReport.h>
#include <nn/settings/system/settings_Eula.h>
#include <nn/settings/system/settings_Hid.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Ldn.h>
#include <nn/settings/system/settings_News.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/settings/system/settings_NetworkService.h>
#include <nn/settings/system/settings_Nfc.h>
#include <nn/settings/system/settings_Region.h>
#include <nn/settings/system/settings_Sleep.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_Telemetry.h>
#include <nn/settings/system/settings_TimeZone.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/system/settings_Usb.h>
#include <nn/settings/system/settings_WirelessLan.h>
#include <nn/time/time_LocationName.h>
#include <nnt/nntest.h>

namespace {

//!< システムイベント
::nn::os::SystemEventType g_Event = {};

} // namespace

//!< 初期状態においてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite1, SignalTest1)
{
    ::nn::settings::system::BindTelemetryDirtyFlagEvent(
        &g_Event, ::nn::os::EventClearMode_ManualClear);

    // バインド直後はシグナル状態にある
    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    // 初期状態においては全てのフラグが true
    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(128, flags.CountPopulation());

    // 状態の取得によって全てのフラグが false に
    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(0, flags.CountPopulation());
}

//!< 言語設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest001)
{
    ::nn::settings::LanguageCode value = {};
    ::nn::settings::GetLanguageCode(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetLanguageCode(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::LanguageCode>());
}

//!< ネットワーク設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest002)
{
    ::nn::settings::system::NetworkSettings value = {};
    const int count = ::nn::settings::system::GetNetworkSettings(&value, 1);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetNetworkSettings(&value, count);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::NetworkSettings>());
}

//!< リージョンコードについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest003)
{
    auto value = ::nn::settings::system::RegionCode();
    ::nn::settings::system::GetRegionCode(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetRegionCode(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::RegionCode>());
}

//!< 誤操作防止画面の有無についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest004)
{
    const bool value = ::nn::settings::system::GetLockScreenFlag();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetLockScreenFlag(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::LockScreenFlag>());
}

//!< 画面の明るさの設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest005)
{
    ::nn::settings::system::BacklightSettings value = {};
    ::nn::settings::system::GetBacklightSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetBacklightSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::BacklightSettings>());
}

//!< Bluetooth デバイス設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest006)
{
    ::nn::settings::system::BluetoothDevicesSettings value = {};
    const int count = ::nn::settings::system::GetBluetoothDevicesSettings(
        &value, 1);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetBluetoothDevicesSettings(&value, count);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            BluetoothDevicesSettings>());
}

//!< ExternalSteadyClock のソース ID についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest007)
{
    ::nn::settings::system::ClockSourceId value = {};
    ::nn::settings::system::GetExternalSteadyClockSourceId(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetExternalSteadyClockSourceId(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::ExternalSteadyClockSourceId
        >());
}

//!< UserSystemClock のコンテキストについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest008)
{
    ::nn::settings::system::SystemClockContext value = {};
    ::nn::settings::system::GetUserSystemClockContext(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetUserSystemClockContext(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::UserSystemClockContext>());
}

//!< アカウント設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest009)
{
    ::nn::settings::system::AccountSettings value = {};
    ::nn::settings::system::GetAccountSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAccountSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AccountSettings>());
}

//!< 本体音量設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest010)
{
    ::nn::settings::system::AudioVolume value1 = {};
    ::nn::settings::system::GetAudioVolume(
        &value1, ::nn::settings::system::AudioVolumeTarget_Speaker);

    ::nn::settings::system::AudioVolume value2 = {};
    ::nn::settings::system::GetAudioVolume(
        &value2, ::nn::settings::system::AudioVolumeTarget_Headphone);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAudioVolume(
        value1, ::nn::settings::system::AudioVolumeTarget_Speaker);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AudioVolume>());

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(0, flags.CountPopulation());

    EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAudioVolume(
        value2, ::nn::settings::system::AudioVolumeTarget_Headphone);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AudioVolume>());
}

//!< 本体 EULA の同意バージョンについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest011)
{
    ::nn::settings::system::EulaVersion value = {};
    const int count = ::nn::settings::system::GetEulaVersions(&value, 1);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetEulaVersions(&value, count);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::EulaVersions>());
}

//!< カラーセット ID についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest012)
{
    const int32_t value = ::nn::settings::system::GetColorSetId();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetColorSetId(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::ColorSetId>());
}

//!< 本体情報アップロードフラグについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest013)
{
    const bool value =
        ::nn::settings::system::GetConsoleInformationUploadFlag();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetConsoleInformationUploadFlag(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            ConsoleInformationUploadFlag>());
}

//!< 自動配信フラグについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest014)
{
    const bool value =
        ::nn::settings::system::GetAutomaticApplicationDownloadFlag();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAutomaticApplicationDownloadFlag(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            AutomaticApplicationDownloadFlag>());
}

//!< 通知設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest015)
{
    ::nn::settings::system::NotificationSettings value = {};
    ::nn::settings::system::GetNotificationSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetNotificationSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::NotificationSettings>());
}

//!< アカウント毎の通知設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest016)
{
    ::nn::settings::system::AccountNotificationSettings value = {};
    const int count = ::nn::settings::system::GetAccountNotificationSettings(
        &value, 1);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAccountNotificationSettings(&value, count);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AccountNotificationSettings
        >());
}

//!< 振動のマスターボリュームについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest017)
{
    const float value = ::nn::settings::system::GetVibrationMasterVolume();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetVibrationMasterVolume(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::VibrationMasterVolume>());
}

//!< TV 設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest018)
{
    ::nn::settings::system::TvSettings value = {};
    ::nn::settings::system::GetTvSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetTvSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::TvSettings>());
}

//!< Edid についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest019)
{
    ::nn::settings::system::Edid value = {};
    ::nn::settings::system::GetEdid(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetEdid(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<::nn::settings::system::TelemetryDirtyFlag::Edid>());
}

//!< 本体オーディオ出力設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest020)
{
    const ::nn::settings::system::AudioOutputMode value1 =
        ::nn::settings::system::GetAudioOutputMode(
            ::nn::settings::system::AudioOutputModeTarget_Hdmi);

    const ::nn::settings::system::AudioOutputMode value2 =
        ::nn::settings::system::GetAudioOutputMode(
            ::nn::settings::system::AudioOutputModeTarget_Speaker);

    const ::nn::settings::system::AudioOutputMode value3 =
        ::nn::settings::system::GetAudioOutputMode(
            ::nn::settings::system::AudioOutputModeTarget_Headphone);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAudioOutputMode(
        value1,
        ::nn::settings::system::AudioOutputModeTarget_Hdmi);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode>());

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(0, flags.CountPopulation());

    EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAudioOutputMode(
        value2,
        ::nn::settings::system::AudioOutputModeTarget_Speaker);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode>());

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(0, flags.CountPopulation());

    EXPECT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetAudioOutputMode(
        value3,
        ::nn::settings::system::AudioOutputModeTarget_Headphone);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::AudioOutputMode>());
}

//!< 強制ミュート設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest021)
{
    const bool value = ::nn::settings::system::IsForceMuteOnHeadphoneRemoved();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetForceMuteOnHeadphoneRemoved(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::ForceMute>());
}

//!< Quest メニュー起動フラグについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest022)
{
    const bool value = ::nn::settings::system::GetQuestFlag();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetQuestFlag(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::QuestFlag>());
}

//!< おすすめ削除の設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest023)
{
    ::nn::settings::system::DataDeletionSettings value = {};
    ::nn::settings::system::GetDataDeletionSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetDataDeletionSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::DataDeletionSettings>());
}

//!< デバイスの地域名についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest024)
{
    ::nn::time::LocationName value = {};
    ::nn::settings::system::GetDeviceTimeZoneLocationName(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetDeviceTimeZoneLocationName(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            DeviceTimeZoneLocationName>());
}

//!< NetworkSystemClock のコンテキストについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest025)
{
    ::nn::settings::system::SystemClockContext value = {};
    ::nn::settings::system::GetNetworkSystemClockContext(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetNetworkSystemClockContext(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::NetworkSystemClockContext
        >());
}

//!< UserSystemClock の自動補正についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest026)
{
    const bool value =
        ::nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionEnabled(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            UserSystemClockAutomaticCorrection>());
}

//!< 画面写真を保存するストレージについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest027)
{
    const ::nn::settings::system::PrimaryAlbumStorage value =
        ::nn::settings::system::GetPrimaryAlbumStorage();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetPrimaryAlbumStorage(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::PrimaryAlbumStorage>());
}

//!< USB 3.0 が有効か否かについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest028)
{
    const bool value = ::nn::settings::system::IsUsb30Enabled();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetUsb30Enabled(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::Usb30EnableFlag>());
}

//!< NFC が有効か否かについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest029)
{
    const bool value = ::nn::settings::system::IsNfcEnabled();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetNfcEnabled(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::NfcEnableFlag>());
}

//!< スリープ設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest030)
{
    ::nn::settings::system::SleepSettings value = {};
    ::nn::settings::system::GetSleepSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetSleepSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::SleepSettings>());
}

//!< 無線 LAN が有効か否かについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest031)
{
    const bool value = ::nn::settings::system::IsWirelessLanEnabled();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetWirelessLanEnabled(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::WirelessLanEnableFlag>());
}

//!< 初回起動設定についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest032)
{
    ::nn::settings::system::InitialLaunchSettings value = {};
    ::nn::settings::system::GetInitialLaunchSettings(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetInitialLaunchSettings(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::InitialLaunchSettings>());
}

//!< 本体の識別名についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest033)
{
    ::nn::settings::system::DeviceNickName value = {};
    ::nn::settings::system::GetDeviceNickName(&value);

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetDeviceNickName(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::DeviceNickName>());
}

//!< ローカル通信のチャンネルについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest034)
{
    const int32_t value = ::nn::settings::system::GetLdnChannel();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetLdnChannel(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::LdnChannel>());
}

//!< Bluetooth が有効か否かについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest035)
{
    const bool value = ::nn::settings::system::IsBluetoothEnabled();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetBluetoothEnabled(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::BluetoothEnableFlag>());
}

//!< エラーレポートを他社に提供してよいかの許可状態についてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite2, SignalTest036)
{
    const ::nn::settings::system::ErrorReportSharePermission
        value = ::nn::settings::system::GetErrorReportSharePermission();

    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetErrorReportSharePermission(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::TelemetryDirtyFlag::
            ErrorReportSharePermission>());
}

//!< 設定マネージャについてダーティーフラグの変更通知が正しく行われるか
TEST(TelemetrySuite3, SignalTest1)
{
    ::nn::settings::system::TelemetryDirtyFlagSet flags = {};

    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    // システム設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    // リセット直後はシグナル状態にある
    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    // リセット直後は全てのフラグが true
    ::nn::settings::system::GetTelemetryDirtyFlags(&flags);
    EXPECT_EQ(128, flags.CountPopulation());
}
