﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/settings/system/settings_Clock.h>
#include <nnt/nntest.h>

namespace {

//!< ExternalSteadyClock のソース ID A
const ::nn::settings::system::ClockSourceId ExternalSteadyClockSourceIdA = {
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xEF, 0xCD, 0xAB, 0x89, 0x67, 0x45, 0x23, 0x01,
    }
};

//!< ExternalSteadyClock のソース ID B
const ::nn::settings::system::ClockSourceId ExternalSteadyClockSourceIdB = {
    {
        0xEF, 0xCD, 0xAB, 0x89, 0x67, 0x45, 0x23, 0x01,
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
    }
};

//!< UserSystemClock のコンテキスト A
const ::nn::settings::system::SystemClockContext UserSystemClockContextA = {
    1,
    {
        2,
        {
            {
                0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
                0xEF, 0xCD, 0xAB, 0x89, 0x67, 0x45, 0x23, 0x01,
            }
        }
    }
};

//!< UserSystemClock のコンテキスト B
const ::nn::settings::system::SystemClockContext UserSystemClockContextB = {
    3,
    {
        4,
        {
            {
                0xEF, 0xCD, 0xAB, 0x89, 0x67, 0x45, 0x23, 0x01,
                0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
            }
        }
    }
};

//!< NetowrkSystemClock のコンテキスト A
const ::nn::settings::system::SystemClockContext NetworkSystemClockContextA = {
    5,
    {
        6,
        {
            {
                0x01, 0x23, 0x45, 0x67, 0x67, 0x45, 0x23, 0x01,
                0xEF, 0xCD, 0xAB, 0x89, 0x89, 0xAB, 0xCD, 0xEF,
            }
        }
    }
};

//!< NetowrkSystemClock のコンテキスト B
const ::nn::settings::system::SystemClockContext NetworkSystemClockContextB = {
    7,
    {
        8,
        {
            {
                0xEF, 0xCD, 0xAB, 0x89, 0x89, 0xAB, 0xCD, 0xEF,
                0x01, 0x23, 0x45, 0x67, 0x67, 0x45, 0x23, 0x01,
            }
        }
    }
};

::nn::time::SteadyClockTimePoint g_UserSystemClockAutomaticCorrectionUpdatedTime;

} // namespace

//!< ExternalSteadyClock のソース ID を取得可能か
TEST(ClockSuite, ExternalSteadyClockSourceIdTest1)
{
    auto sourceId = ::nn::settings::system::ClockSourceId();

    ::nn::settings::system::GetExternalSteadyClockSourceId(&sourceId);
}

//!< ExternalSteadyClock のソース ID を設定可能か
TEST(ClockSuite, ExternalSteadyClockSourceIdTest2)
{
    ::nn::settings::system::SetExternalSteadyClockSourceId(
        ExternalSteadyClockSourceIdA);
    {
        auto sourceId = ::nn::settings::system::ClockSourceId();
        ::nn::settings::system::GetExternalSteadyClockSourceId(&sourceId);
        EXPECT_EQ(ExternalSteadyClockSourceIdA, sourceId);
    }

    ::nn::settings::system::SetExternalSteadyClockSourceId(
        ExternalSteadyClockSourceIdB);
    {
        auto sourceId = ::nn::settings::system::ClockSourceId();
        ::nn::settings::system::GetExternalSteadyClockSourceId(&sourceId);
        EXPECT_EQ(ExternalSteadyClockSourceIdB, sourceId);
    }
}

//!< UserSystemClock のコンテキストを取得可能か
TEST(ClockSuite, UserSystemClockContextTest1)
{
    auto context = ::nn::settings::system::SystemClockContext();

    ::nn::settings::system::GetUserSystemClockContext(&context);
}

//!< UserSystemClock のコンテキストを設定可能か
TEST(ClockSuite, UserSystemClockContextTest2)
{
    ::nn::settings::system::SetUserSystemClockContext(UserSystemClockContextA);
    {
        auto context = ::nn::settings::system::SystemClockContext();
        ::nn::settings::system::GetUserSystemClockContext(&context);
        EXPECT_EQ(UserSystemClockContextA.offset, context.offset);
        EXPECT_EQ(UserSystemClockContextA.timeStamp.value,
                  context.timeStamp.value);
        EXPECT_EQ(UserSystemClockContextA.timeStamp.sourceId,
                  context.timeStamp.sourceId);
    }

    ::nn::settings::system::SetUserSystemClockContext(UserSystemClockContextB);
    {
        auto context = ::nn::settings::system::SystemClockContext();
        ::nn::settings::system::GetUserSystemClockContext(&context);
        EXPECT_EQ(UserSystemClockContextB.offset, context.offset);
        EXPECT_EQ(UserSystemClockContextB.timeStamp.value,
                  context.timeStamp.value);
        EXPECT_EQ(UserSystemClockContextB.timeStamp.sourceId,
                  context.timeStamp.sourceId);
    }
}

//!< NetowrkSystemClock のコンテキストを取得可能か
TEST(ClockSuite, NetworkSystemClockContextTest1)
{
    auto context = ::nn::settings::system::SystemClockContext();

    ::nn::settings::system::GetNetworkSystemClockContext(&context);
}

//!< NetowrkSystemClock のコンテキストを設定可能か
TEST(ClockSuite, NetworkSystemClockContextTest2)
{
    ::nn::settings::system::SetNetworkSystemClockContext(NetworkSystemClockContextA);
    {
        auto context = ::nn::settings::system::SystemClockContext();
        ::nn::settings::system::GetNetworkSystemClockContext(&context);
        EXPECT_EQ(NetworkSystemClockContextA.offset, context.offset);
        EXPECT_EQ(NetworkSystemClockContextA.timeStamp.value,
                  context.timeStamp.value);
        EXPECT_EQ(NetworkSystemClockContextA.timeStamp.sourceId,
                  context.timeStamp.sourceId);
    }

    ::nn::settings::system::SetNetworkSystemClockContext(NetworkSystemClockContextB);
    {
        auto context = ::nn::settings::system::SystemClockContext();
        ::nn::settings::system::GetNetworkSystemClockContext(&context);
        EXPECT_EQ(NetworkSystemClockContextB.offset, context.offset);
        EXPECT_EQ(NetworkSystemClockContextB.timeStamp.value,
                  context.timeStamp.value);
        EXPECT_EQ(NetworkSystemClockContextB.timeStamp.sourceId,
                  context.timeStamp.sourceId);
    }
}

//!< UserSystemClock の自動補正フラグを取得可能か
TEST(ClockSuite, UserSystemClockAutomaticCorrectionEnabledTest1)
{
    ::nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled();
}

//!< UserSystemClock の自動補正フラグを設定可能か
TEST(ClockSuite, UserSystemClockAutomaticCorrectionEnabledTest2)
{
    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionEnabled(true);
    {
        EXPECT_TRUE(::nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled());
    }

    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionEnabled(false);
    {
        EXPECT_FALSE(::nn::settings::system::IsUserSystemClockAutomaticCorrectionEnabled());
    }
}

//!< シャットダウン時の RTC 値を取得可能か
TEST(ClockSuite, ShutdownRtcValueTest1)
{
    ::nn::settings::system::GetShutdownRtcValue();
}

//!< シャットダウン時の RTC 値を設定可能か
TEST(ClockSuite, ShutdownRtcValueTest2)
{
    const int64_t BackupValue = ::nn::settings::system::GetShutdownRtcValue();

    ::nn::settings::system::SetShutdownRtcValue(123456789LL);
    EXPECT_EQ(123456789LL, ::nn::settings::system::GetShutdownRtcValue());

    ::nn::settings::system::SetShutdownRtcValue(987654321LL);
    EXPECT_EQ(987654321LL, ::nn::settings::system::GetShutdownRtcValue());

    ::nn::settings::system::SetShutdownRtcValue(BackupValue);
    EXPECT_EQ(BackupValue, ::nn::settings::system::GetShutdownRtcValue());
}

//!< ExternalSteadyClock の内部オフセットを取得可能か
TEST(ClockSuite, ExternalSteadyClockInternalOffset1)
{
    ::nn::settings::system::GetExternalSteadyClockInternalOffset();
}

//!< ExternalSteadyClock の内部オフセットを設定可能か
TEST(ClockSuite, ExternalSteadyClockInternalOffset2)
{
    const int64_t BackupValue = ::nn::settings::system::GetExternalSteadyClockInternalOffset();

    ::nn::settings::system::SetExternalSteadyClockInternalOffset(123456789LL);
    EXPECT_EQ(123456789LL, ::nn::settings::system::GetExternalSteadyClockInternalOffset());

    ::nn::settings::system::SetExternalSteadyClockInternalOffset(987654321LL);
    EXPECT_EQ(987654321LL, ::nn::settings::system::GetExternalSteadyClockInternalOffset());

    ::nn::settings::system::SetExternalSteadyClockInternalOffset(BackupValue);
    EXPECT_EQ(BackupValue, ::nn::settings::system::GetExternalSteadyClockInternalOffset());
}

//!< UserSystemClock の自動補正フラグ更新時間を取得可能か
TEST(ClockSuite, UserSystemClockAutomaticCorrectionUpdatedTime1)
{
    ::nn::settings::system::GetUserSystemClockAutomaticCorrectionUpdatedTime(&g_UserSystemClockAutomaticCorrectionUpdatedTime);
}

//!< UserSystemClock の自動補正フラグ更新時間を設定可能か
TEST(ClockSuite, UserSystemClockAutomaticCorrectionUpdatedTime2)
{
    ::nn::time::SteadyClockTimePoint value =
    {
        123456LL,
        ::nn::util::GenerateUuid()
    };

    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionUpdatedTime(value);
    {
        ::nn::time::SteadyClockTimePoint tp;
        ::nn::settings::system::GetUserSystemClockAutomaticCorrectionUpdatedTime(&tp);
        EXPECT_EQ(value, tp);
    }

    // 念のため元に戻す
    ::nn::settings::system::SetUserSystemClockAutomaticCorrectionUpdatedTime(g_UserSystemClockAutomaticCorrectionUpdatedTime);
    {
        ::nn::time::SteadyClockTimePoint tp;
        ::nn::settings::system::GetUserSystemClockAutomaticCorrectionUpdatedTime(&tp);
        EXPECT_EQ(g_UserSystemClockAutomaticCorrectionUpdatedTime, tp);
    }
}
