﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/settings/settings_DebugPad.h>
#include <nnt/nntest.h>

namespace {


//!< DebugPad に対する汎用コントローラの割り当てが初期値であることを期待します。
void ExpectInitialDebugPadGenericPadMap(
    const ::nn::settings::DebugPadGenericPadMap& map) NN_NOEXCEPT;

//!< DebugPad に対するキーボードの割り当てが初期値であることを期待します。
void ExpectInitialDebugPadKeyboardMap(
    const ::nn::settings::DebugPadKeyboardMap& map) NN_NOEXCEPT;

//!< 割り当てをファジングします。
template<typename T>
void FuzzMap(T* outValue) NN_NOEXCEPT;

//!< 割り当てがファジングされた状態であることを期待します。
template<typename T>
void ExpectFuzzedMap(const T& map) NN_NOEXCEPT;

} // namespace

//!< DebugPad に対する汎用コントローラの割り当ては正しく初期化されるか
TEST(DebugPadSuite, DebugPadGenericPadMapTest)
{
    ::nn::settings::DebugPadGenericPadMap map;

    // 初期状態の割り当ては初期値
    ::nn::settings::GetDebugPadGenericPadMap(&map);
    {
        SCOPED_TRACE("");
        ExpectInitialDebugPadGenericPadMap(map);
    }

    // 割り当てを設定
    FuzzMap(&map);
    map.genericPadId = 10;
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>(true);
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>(false);
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>(true);
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>(false);
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>(true);
    map.attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>(true);
    ::nn::settings::SetDebugPadGenericPadMap(map);

    // 設定値の反映を確認
    ::nn::settings::GetDebugPadGenericPadMap(&map);
    EXPECT_EQ(10, map.genericPadId);
    {
        SCOPED_TRACE("");
        ExpectFuzzedMap(map);
    }
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>());

    // リセット後の割り当ては初期値
    ::nn::settings::ResetDebugPadGenericPadMap();
    ::nn::settings::GetDebugPadGenericPadMap(&map);
    EXPECT_EQ(10, map.genericPadId); // 汎用コントローラ ID の値は維持
    {
        SCOPED_TRACE("");
        ExpectInitialDebugPadGenericPadMap(map);
    }
}

//!< DebugPad に対するキーボードの割り当ては正しく初期化されるか
TEST(DebugPadSuite, DebugPadKeyboardMapTest)
{
    ::nn::settings::DebugPadKeyboardMap map;

    // 初期状態の割り当ては初期値
    ::nn::settings::GetDebugPadKeyboardMap(&map);
    {
        SCOPED_TRACE("");
        ExpectInitialDebugPadKeyboardMap(map);
    }

    // 割り当てを設定
    FuzzMap(&map);
    ::nn::settings::SetDebugPadKeyboardMap(map);

    // 設定値の反映を確認
    ::nn::settings::GetDebugPadKeyboardMap(&map);
    {
        SCOPED_TRACE("");
        ExpectFuzzedMap(map);
    }

    // リセット後の割り当ては初期値
    ::nn::settings::ResetDebugPadKeyboardMap();
    ::nn::settings::GetDebugPadKeyboardMap(&map);
    {
        SCOPED_TRACE("");
        ExpectInitialDebugPadKeyboardMap(map);
    }
}

namespace {

void ExpectInitialDebugPadGenericPadMap(
    const ::nn::settings::DebugPadGenericPadMap& map) NN_NOEXCEPT
{
    EXPECT_EQ(::nn::settings::GenericPadButton_4, map.buttonA);
    EXPECT_EQ(::nn::settings::GenericPadButton_3, map.buttonB);
    EXPECT_EQ(::nn::settings::GenericPadButton_2, map.buttonX);
    EXPECT_EQ(::nn::settings::GenericPadButton_1, map.buttonY);
    EXPECT_EQ(::nn::settings::GenericPadButton_5, map.buttonL);
    EXPECT_EQ(::nn::settings::GenericPadButton_6, map.buttonR);
    EXPECT_EQ(::nn::settings::GenericPadButton_7, map.buttonZL);
    EXPECT_EQ(::nn::settings::GenericPadButton_8, map.buttonZR);
    EXPECT_EQ(::nn::settings::GenericPadButton_10, map.buttonStart);
    EXPECT_EQ(::nn::settings::GenericPadButton_9, map.buttonSelect);
    EXPECT_EQ(::nn::settings::GenericPadButton_Left, map.buttonLeft);
    EXPECT_EQ(::nn::settings::GenericPadButton_Up, map.buttonUp);
    EXPECT_EQ(::nn::settings::GenericPadButton_Right, map.buttonRight);
    EXPECT_EQ(::nn::settings::GenericPadButton_Down, map.buttonDown);
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>());
}

void ExpectInitialDebugPadKeyboardMap(
    const ::nn::settings::DebugPadKeyboardMap& map) NN_NOEXCEPT
{
    EXPECT_EQ(0, map.buttonA);
    EXPECT_EQ(0, map.buttonB);
    EXPECT_EQ(0, map.buttonX);
    EXPECT_EQ(0, map.buttonY);
    EXPECT_EQ(0, map.buttonL);
    EXPECT_EQ(0, map.buttonR);
    EXPECT_EQ(0, map.buttonZL);
    EXPECT_EQ(0, map.buttonZR);
    EXPECT_EQ(0, map.buttonStart);
    EXPECT_EQ(0, map.buttonSelect);
    EXPECT_EQ(0, map.buttonLeft);
    EXPECT_EQ(0, map.buttonUp);
    EXPECT_EQ(0, map.buttonRight);
    EXPECT_EQ(0, map.buttonDown);
}

template<typename T>
void FuzzMap(T* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    outValue->buttonA = 'A';
    outValue->buttonB = 'B';
    outValue->buttonX = 'X';
    outValue->buttonY = 'Y';
    outValue->buttonL = 'L';
    outValue->buttonR = 'R';
    outValue->buttonZL = '1';
    outValue->buttonZR = '2';
    outValue->buttonStart = '3';
    outValue->buttonSelect = '4';
    outValue->buttonLeft = '5';
    outValue->buttonUp = '6';
    outValue->buttonRight = '7';
    outValue->buttonDown = '8';
}

template<typename T>
void ExpectFuzzedMap(const T& map) NN_NOEXCEPT
{
    EXPECT_EQ('A', map.buttonA);
    EXPECT_EQ('B', map.buttonB);
    EXPECT_EQ('X', map.buttonX);
    EXPECT_EQ('Y', map.buttonY);
    EXPECT_EQ('L', map.buttonL);
    EXPECT_EQ('R', map.buttonR);
    EXPECT_EQ('1', map.buttonZL);
    EXPECT_EQ('2', map.buttonZR);
    EXPECT_EQ('3', map.buttonStart);
    EXPECT_EQ('4', map.buttonSelect);
    EXPECT_EQ('5', map.buttonLeft);
    EXPECT_EQ('6', map.buttonUp);
    EXPECT_EQ('7', map.buttonRight);
    EXPECT_EQ('8', map.buttonDown);
}

} // namespace
