﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <iterator>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nnt/nntest.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

#include "testSettings_Utility.h"

namespace {

//!< ファームウェアデバッグ設定（セーブ）の期待値
const ::nn::Bit8 SystemSaveFirmwareDebugData[] =
{
    0x3C, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x66, 0x6f, 0x6f, 0x21, 0x62, 0x61, 0x72, 0x00,
    0x01, 0x05, 0x00, 0x00, 0x00, 0x62, 0x65, 0x65,
    0x66, 0x00, 0x08, 0x00, 0x00, 0x00, 0x66, 0x6f,
    0x6f, 0x21, 0x62, 0x61, 0x7a, 0x00, 0x01, 0x11,
    0x00, 0x00, 0x00, 0x64, 0x65, 0x61, 0x64, 0x66,
    0x65, 0x65, 0x64, 0x62, 0x65, 0x65, 0x66, 0x66,
    0x61, 0x63, 0x65, 0x00,
};

//!< ファームウェアデバッグ設定（データ）の期待値
const ::nn::Bit8 SystemDataFirmwareDebugData[] =
{
    0x38, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
    0x66, 0x6f, 0x6f, 0x21, 0x62, 0x61, 0x72, 0x00,
    0x01, 0x09, 0x00, 0x00, 0x00, 0x64, 0x65, 0x61,
    0x64, 0x62, 0x65, 0x65, 0x66, 0x00, 0x08, 0x00,
    0x00, 0x00, 0x66, 0x6f, 0x6f, 0x21, 0x62, 0x61,
    0x7a, 0x00, 0x01, 0x09, 0x00, 0x00, 0x00, 0x66,
    0x65, 0x65, 0x64, 0x66, 0x61, 0x63, 0x65, 0x00,
};

} // namespace

class SettingsManagementSuite : public testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        ::nnt::settings::fwdbg::InitializeSettings();
    }
};

//!< ファームウェアデバッグ設定の読み出しが正常に動作するか
TEST_F(SettingsManagementSuite, ReadSettingsTest1)
{
    const char name[] = "foo";
    const char key1[] = "bar";
    const char key2[] = "baz";
    const char value1_1[] = "deadbeef";
    const char value1_2[] = "beef";
    const char value2_1[] = "feedface";
    const char value2_2[] = "deadfeedbeefface";
    ::nn::Bit8 buffer[512] = {};

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key1, value1_1, NN_ARRAY_SIZE(value1_1));
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key2, value2_1, NN_ARRAY_SIZE(value2_1));

    // テスト用の設定を書き換え
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key1, value1_2, NN_ARRAY_SIZE(value1_2));
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key2, value2_2, NN_ARRAY_SIZE(value2_2));

    // ファームウェアデバッグ設定（セーブ）を読み出し
    ::std::fill(::std::begin(buffer), ::std::end(buffer), ::nn::Bit8());
    EXPECT_EQ(
        NN_ARRAY_SIZE(SystemSaveFirmwareDebugData),
        ::nn::settings::fwdbg::ReadSettings(
            buffer, NN_ARRAY_SIZE(buffer),
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug));
    for (size_t i = 0; i < NN_ARRAY_SIZE(SystemSaveFirmwareDebugData); ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "SystemSaveFirmwareDebugData[" << i << "]");
        EXPECT_EQ(SystemSaveFirmwareDebugData[i], buffer[i]);
    }

    // ファームウェアデバッグ設定（データ）を読み出し
    ::std::fill(::std::begin(buffer), ::std::end(buffer), ::nn::Bit8());
    EXPECT_EQ(
        NN_ARRAY_SIZE(SystemDataFirmwareDebugData),
        ::nn::settings::fwdbg::ReadSettings(
            buffer, NN_ARRAY_SIZE(buffer),
            ::nn::settings::fwdbg::SettingsTarget_SystemDataFirmwareDebug));
    for (size_t i = 0; i < NN_ARRAY_SIZE(SystemDataFirmwareDebugData); ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "SystemDataFirmwareDebugData[" << i << "]");
        EXPECT_EQ(SystemDataFirmwareDebugData[i], buffer[i]);
    }

    // プラットフォーム構成を読み出し
    EXPECT_EQ(
        0u,
        ::nn::settings::fwdbg::ReadSettings(
            buffer, NN_ARRAY_SIZE(buffer),
            ::nn::settings::fwdbg::
                SettingsTarget_SystemDataPlatformConfigration));
}

//!< バッファが不足した状態で設定の読み出しが正常に動作するか
TEST_F(SettingsManagementSuite, ReadSettingsTest2)
{
    const char name[] = "foo";
    const char key1[] = "bar";
    const char key2[] = "baz";
    const char value1_1[] = "deadbeef";
    const char value1_2[] = "beef";
    const char value2_1[] = "feedface";
    const char value2_2[] = "deadfeedbeefface";
    ::nn::Bit8 buffer[512] = {};
    const size_t bufferSize = 16;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key1, value1_1, NN_ARRAY_SIZE(value1_1));
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key2, value2_1, NN_ARRAY_SIZE(value2_1));

    // テスト用の設定を書き換え
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key1, value1_2, NN_ARRAY_SIZE(value1_2));
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key2, value2_2, NN_ARRAY_SIZE(value2_2));

    // ファームウェアデバッグ設定（セーブ）を読み出し
    ::std::fill(::std::begin(buffer), ::std::end(buffer), ::nn::Bit8());
    EXPECT_EQ(
        bufferSize,
        ::nn::settings::fwdbg::ReadSettings(
            buffer, bufferSize,
            ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug));
    for (size_t i = 0; i < NN_ARRAY_SIZE(buffer); ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "SystemSaveFirmwareDebugData[" << i << "]");
        EXPECT_EQ(
            bufferSize <= i ? 0 : SystemSaveFirmwareDebugData[i], buffer[i]);
    }

    // ファームウェアデバッグ設定（データ）を読み出し
    ::std::fill(::std::begin(buffer), ::std::end(buffer), ::nn::Bit8());
    EXPECT_EQ(
        bufferSize,
        ::nn::settings::fwdbg::ReadSettings(
            buffer, bufferSize,
            ::nn::settings::fwdbg::SettingsTarget_SystemDataFirmwareDebug));
    for (size_t i = 0; i < NN_ARRAY_SIZE(buffer); ++i)
    {
        SCOPED_TRACE(
            ::testing::Message() << "SystemDataFirmwareDebugData[" << i << "]");
        EXPECT_EQ(
            bufferSize <= i ? 0 : SystemDataFirmwareDebugData[i], buffer[i]);
    }
}

//!< ファームウェアデバッグ設定のリセットが正常に動作するか
TEST_F(SettingsManagementSuite, ResetSettingsTest1)
{
    const char name[] = "foo";
    const char key1[] = "bar";
    const char key2[] = "baz";
    const char value1_1[] = "deadbeef";
    const char value1_2[] = "beef";
    const char value2_1[] = "feedface";
    const char value2_2[] = "deadfeedbeefface";
    char buffer[32];

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key1, value1_1, NN_ARRAY_SIZE(value1_1));
    ::nnt::settings::fwdbg::CreateSettingsItem(
        name, key2, value2_1, NN_ARRAY_SIZE(value2_1));

    // テスト用の設定を書き換え
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key1, value1_2, NN_ARRAY_SIZE(value1_2));
    ::nn::settings::fwdbg::SetSettingsItemValue(
        name, key2, value2_2, NN_ARRAY_SIZE(value2_2));

    // テスト用の設定の書き換えを確認
    EXPECT_EQ(
        NN_ARRAY_SIZE(value1_2),
        ::nn::settings::fwdbg::GetSettingsItemValue(
            buffer, NN_ARRAY_SIZE(buffer), name, key1));
    EXPECT_STREQ(value1_2, buffer);
    EXPECT_EQ(
        NN_ARRAY_SIZE(value2_2),
        ::nn::settings::fwdbg::GetSettingsItemValue(
            buffer, NN_ARRAY_SIZE(buffer), name, key2));
    EXPECT_STREQ(value2_2, buffer);

    // ファームウェアデバッグ設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveFirmwareDebug);

    // テスト用の設定のリセットを確認
    EXPECT_EQ(
        NN_ARRAY_SIZE(value1_1),
        ::nn::settings::fwdbg::GetSettingsItemValue(
            buffer, NN_ARRAY_SIZE(buffer), name, key1));
    EXPECT_STREQ(value1_1, buffer);
    EXPECT_EQ(
        NN_ARRAY_SIZE(value2_1),
        ::nn::settings::fwdbg::GetSettingsItemValue(
            buffer, NN_ARRAY_SIZE(buffer), name, key2));
    EXPECT_STREQ(value2_1, buffer);
}
