﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "testSettings_Utility.h"

class SettingsItemKeyIteratorSuite : public testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        ::nnt::settings::fwdbg::InitializeSettings();
    }
};

//!< キーイテレータを通して設定項目の操作ができるか
TEST_F(SettingsItemKeyIteratorSuite, IterationTest1)
{
    const char* const name = "foo";
    const size_t ItemCount = 3;
    const char* keys[ItemCount] = { "bar", "baz", "qux" };
    const char* values[ItemCount] = { "dead", "beef", "face" };

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    for (int i = 0; i < ItemCount; ++i)
    {
        ::nnt::settings::fwdbg::CreateSettingsItem(
            name,
            keys[i],
            values[i],
            ::std::strlen(values[i]) + 1);
    }

    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);

    for (int i = 0; i < ItemCount; ++i)
    {
        const size_t keySize = ::std::strlen(keys[i]) + 1;
        EXPECT_EQ(keySize, ::nn::settings::fwdbg::GetSettingsItemKeySize(iter));

        char key[16];
        EXPECT_EQ(
            keySize,
            ::nn::settings::fwdbg::GetSettingsItemKey(key, sizeof(key), iter));
        EXPECT_STREQ(keys[i], key);

        // キーイテレータを通して項目のサイズを取得可能
        const size_t valueSize = ::std::strlen(values[i]) + 1;
        EXPECT_EQ(valueSize,
                  ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key));

        // キーイテレータを通して項目の値を取得可能
        char value[16];
        EXPECT_EQ(valueSize,
                  ::nn::settings::fwdbg::GetSettingsItemValue(value,
                                                              sizeof(value),
                                                              name,
                                                              key));
        EXPECT_STREQ(values[i], value);

        const char* const newValue = "beadfeed";
        const size_t newSize = ::std::strlen(newValue) + 1;

        // キーイテレータを通して項目の値を設定可能
        ::nn::settings::fwdbg::SetSettingsItemValue(name,
                                                    key,
                                                    newValue,
                                                    newSize);
        EXPECT_EQ(newSize,
                  ::nn::settings::fwdbg::GetSettingsItemValue(value,
                                                              sizeof(value),
                                                              name,
                                                              key));
        EXPECT_STREQ(newValue, value);

        // キーイテレータを通して項目の値をリセット可能
        ::nn::settings::fwdbg::ResetSettingsItemValue(name, key);
        EXPECT_EQ(valueSize,
                  ::nn::settings::fwdbg::GetSettingsItemValue(value,
                                                              sizeof(value),
                                                              name,
                                                              key));
        EXPECT_STREQ(values[i], value);

        if (i < ItemCount - 1)
        {
            NNT_EXPECT_RESULT_SUCCESS(
                ::nn::settings::fwdbg::AdvanceSettingsItemKeyIterator(&iter));
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(
                ::nn::settings::ResultStopIteration,
                ::nn::settings::fwdbg::AdvanceSettingsItemKeyIterator(&iter));
        }
    }

    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
}
