﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <limits>
#include <nn/nn_Macro.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nnt/nntest.h>

#include "testSettings_Utility.h"

class SettingsItemSuite : public testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        ::nnt::settings::fwdbg::InitializeSettings();
    }
};

//!< バッファ長が不足している際に正常に動作するか
TEST_F(SettingsItemSuite, BufferTest1)
{
    const char* const name = "foo";
    const char* const key = "barbaz";

    const char* const value = "deadbeef";
    const size_t size = ::std::strlen(value) + 1;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, value, size);

    const size_t bufferSize = 4;
    char buffer[bufferSize];

    // バッファ長が不足している際は読み込めるだけ設定項目値を取得
    EXPECT_EQ(bufferSize,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          bufferSize,
                                                          name,
                                                          key));
    for (int i = 0; i < bufferSize; ++i)
    {
        EXPECT_EQ(value[i], buffer[i]) << "index = " << i;
    }

    // バッファ長が 0 の際は設定項目値を読み込まない
    EXPECT_EQ(0,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          0,
                                                          name,
                                                          key));

    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(name);

    // バッファ長が不足している際は読み込めるだけキーの値を取得
    EXPECT_EQ(
        bufferSize,
        ::nn::settings::fwdbg::GetSettingsItemKey(buffer, bufferSize, iter));
    EXPECT_EQ('\0', buffer[bufferSize - 1]);
    for (int i = 0; i < bufferSize - 1; ++i)
    {
        EXPECT_EQ(key[i], buffer[i]) << "index = " << i;
    }

    // バッファ長が 0 の際はキーの値を読み込まない
    EXPECT_EQ(0, ::nn::settings::fwdbg::GetSettingsItemKey(buffer, 0, iter));

    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
}

//!< 長い名前に対して正常に動作するか
TEST_F(SettingsItemSuite, NameTest1)
{
    // 長い設定名を作成
    static char s_Name[::nn::settings::fwdbg::SettingsNameLengthMax + 1];
    s_Name[::nn::settings::fwdbg::SettingsNameLengthMax] = '\0';
    for (int i = 0; i < ::nn::settings::fwdbg::SettingsNameLengthMax; ++i)
    {
        s_Name[i] = 'a' + (i % ('z' - 'a' + 1));
    }

    // 長い項目名を作成
    static char s_Key[::nn::settings::fwdbg::SettingsItemKeyLengthMax + 1];
    s_Key[::nn::settings::fwdbg::SettingsItemKeyLengthMax] = '\0';
    for (int i = 0; i < ::nn::settings::fwdbg::SettingsItemKeyLengthMax; ++i)
    {
        s_Key[i] = 'a' + (i % ('z' - 'a' + 1));
    }

    const char* const value1 = "deadbeef";
    const size_t size1 = ::std::strlen(value1) + 1;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(s_Name);
    ::nnt::settings::fwdbg::CreateSettingsItem(s_Name, s_Key, value1, size1);

    // 名前が長い場合でも項目のバイト数は取得可能
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValueSize(s_Name, s_Key));

    char buffer[16];

    // 名前が長い場合でも項目の値は取得可能
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          s_Name,
                                                          s_Key));
    EXPECT_STREQ(value1, buffer);

    const char* const value2 = "cafe";
    const size_t size2 = ::std::strlen(value2) + 1;

    // 名前が長い場合でも項目の値は設定可能
    ::nn::settings::fwdbg::SetSettingsItemValue(s_Name, s_Key, value2, size2);
    EXPECT_EQ(size2,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          s_Name,
                                                          s_Key));
    EXPECT_STREQ(value2, buffer);

    // 名前が長い場合でも項目の値はリセット可能
    ::nn::settings::fwdbg::ResetSettingsItemValue(s_Name, s_Key);
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          s_Name,
                                                          s_Key));
    EXPECT_STREQ(value1, buffer);

    // 名前が長い場合でもキーイテレータは作成可能
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(s_Name);

    // 名前が長い場合でもキーのサイズは取得可能
    const size_t keySize = sizeof(s_Key);
    EXPECT_EQ(keySize, ::nn::settings::fwdbg::GetSettingsItemKeySize(iter));

    // 名前が長い場合でもキーの値は取得可能
    static char s_KeyBuffer[keySize];
    EXPECT_EQ(
        keySize,
        ::nn::settings::fwdbg::GetSettingsItemKey(s_KeyBuffer, keySize, iter));
    EXPECT_STREQ(s_Key, s_KeyBuffer);

    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
}

//!< 有効な文字種に対して正常に動作するか
TEST_F(SettingsItemSuite, NameTest2)
{
    const char* const ValidChars = "-._"
                                   "abcdefghijklmnopqrstuvwxyz"
                                   "0123456789";

    const char* const value1 = "cafe";
    const size_t size1 = ::std::strlen(value1) + 1;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(ValidChars);
    ::nnt::settings::fwdbg::CreateSettingsItem(ValidChars,
                                               ValidChars,
                                               value1,
                                               size1);

    // 文字種が有効ならば項目のバイト数は取得可能
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValueSize(ValidChars,
                                                              ValidChars));

    char buffer[16];

    // 文字種が有効ならば項目の値は取得可能
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          ValidChars,
                                                          ValidChars));
    EXPECT_STREQ(value1, buffer);

    const char* const value2 = "deadbeef";
    const size_t size2 = ::std::strlen(value2) + 1;

    // 文字種が有効ならば項目の値は設定可能
    ::nn::settings::fwdbg::SetSettingsItemValue(ValidChars,
                                                ValidChars,
                                                value2,
                                                size2);
    EXPECT_EQ(size2,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          ValidChars,
                                                          ValidChars));
    EXPECT_STREQ(value2, buffer);

    // 文字種が有効ならば項目の値はリセット可能
    ::nn::settings::fwdbg::ResetSettingsItemValue(ValidChars, ValidChars);
    EXPECT_EQ(size1,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          ValidChars,
                                                          ValidChars));
    EXPECT_STREQ(value1, buffer);

    // 文字種が有効ならばキーイテレータは作成可能
    ::nn::settings::fwdbg::SettingsItemKeyIterator iter =
        ::nn::settings::fwdbg::CreateSettingsItemKeyIterator(ValidChars);

    // 文字種が有効ならばキーのサイズは取得可能
    const size_t keySize = ::std::strlen(ValidChars) + 1;;
    EXPECT_EQ(keySize, ::nn::settings::fwdbg::GetSettingsItemKeySize(iter));

    // 文字種が有効ならばキーの値は取得可能
    static char keyBuffer[64];
    EXPECT_EQ(
        keySize,
        ::nn::settings::fwdbg::GetSettingsItemKey(keyBuffer,
                                                  sizeof(keyBuffer),
                                                  iter));
    EXPECT_STREQ(ValidChars, keyBuffer);

    ::nn::settings::fwdbg::DestroySettingsItemKeyIterator(&iter);
}

//!< 設定項目値のバイト数の増減に対して正しく動作するか
TEST_F(SettingsItemSuite, RewriteTest1)
{
    const char* const name = "foo";
    const char* const key = "bar";

    const char* const value = "deadbeef";
    const size_t size = ::std::strlen(value) + 1;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, value, size);

    char buffer[16];

    // 文字種が有効ならば項目の値は取得可能
    EXPECT_EQ(size,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          name,
                                                          key));
    EXPECT_STREQ(value, buffer);

    const char* const shortValue = "cafe";
    const size_t shortSize = ::std::strlen(shortValue) + 1;

    // 設定項目値のバイト数が減少する方向に書き換え可能
    ::nn::settings::fwdbg::SetSettingsItemValue(name,
                                                key,
                                                shortValue,
                                                shortSize);
    EXPECT_EQ(shortSize,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          name,
                                                          key));
    EXPECT_STREQ(shortValue, buffer);

    const char* const longValue = "beadfeedface";
    const size_t longSize = ::std::strlen(longValue) + 1;

    // 設定項目値のバイト数が増加する方向に書き換え可能
    ::nn::settings::fwdbg::SetSettingsItemValue(name,
                                                key,
                                                longValue,
                                                longSize);
    EXPECT_EQ(longSize,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          sizeof(buffer),
                                                          name,
                                                          key));
    EXPECT_STREQ(longValue, buffer);
}

//!< 大きな設定項目値に対して正しく動作するか
TEST_F(SettingsItemSuite, LargeDataTest1)
{
    const char* const name = "foo";
    const char* const key = "bar";

    const size_t size = 128 * 1024;

    static char s_Value[size];
    for (int i = 0; i < size; ++i)
    {
        s_Value[i] =
            static_cast<char>(i % (::std::numeric_limits<char>::max() + 1));
    }

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key, s_Value, size);

    // 設定項目値が大きい場合であってもバイト数は取得可能
    EXPECT_EQ(size,
              ::nn::settings::fwdbg::GetSettingsItemValueSize(name, key));

    char s_Buffer[size];

    // 設定項目値が大きい場合であっても項目の値は取得可能
    EXPECT_EQ(size,
              ::nn::settings::fwdbg::GetSettingsItemValue(s_Buffer,
                                                          size,
                                                          name,
                                                          key));
    for (int i = 0; i < size; ++i)
    {
        EXPECT_EQ(s_Value[i], s_Buffer[i]) << "index = " << i;
    }

    const size_t offset = 100;

    // 設定項目値が大きい場合であっても項目の値は設定可能
    ::nn::settings::fwdbg::SetSettingsItemValue(name,
                                                key,
                                                &s_Value[offset],
                                                size - offset);
    EXPECT_EQ(size - offset,
              ::nn::settings::fwdbg::GetSettingsItemValue(s_Buffer,
                                                          size,
                                                          name,
                                                          key));
    for (int i = 0; i < size - offset; ++i)
    {
        EXPECT_EQ(s_Value[i + offset], s_Buffer[i]) << "index = " << i;
    }

    // 設定項目値が大きい場合であっても項目の値はリセット可能
    ::nn::settings::fwdbg::ResetSettingsItemValue(name, key);
    EXPECT_EQ(size,
              ::nn::settings::fwdbg::GetSettingsItemValue(s_Buffer,
                                                          size,
                                                          name,
                                                          key));
    for (int i = 0; i < size; ++i)
    {
        EXPECT_EQ(s_Value[i], s_Buffer[i]) << "index = " << i;
    }
}

//!< 空の設定項目値に対して正しく動作するか
TEST_F(SettingsItemSuite, ZeroDataTest1)
{
    const char* const name = "foo";
    const char* const key1 = "bar";
    const char* const key2 = "baz";
    const char* const value1 = "deadbeef";
    const size_t valueSize1 = ::std::strlen(value1) + 1;
    const char* const value2 = "feedface";
    const size_t valueSize2 = ::std::strlen(value2) + 1;

    // テスト用の設定を作成
    ::nnt::settings::fwdbg::CreateSettings(name);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key1, "", 0);
    ::nnt::settings::fwdbg::CreateSettingsItem(name, key2, value2, valueSize2);

    const size_t size = 16;

    char buffer[size];

    // 設定項目値が空の場合は値は取得されない
    EXPECT_EQ(0,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          size,
                                                          name,
                                                          key1));

    ::nn::settings::fwdbg::SetSettingsItemValue(name, key1, value1, valueSize1);
    EXPECT_EQ(valueSize1,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          size,
                                                          name,
                                                          key1));
    EXPECT_STREQ(value1, buffer);

    // 設定項目値の初期値が空の場合であっても項目の値はリセット可能
    ::nn::settings::fwdbg::ResetSettingsItemValue(name, key1);
    EXPECT_EQ(0,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          size,
                                                          name,
                                                          key1));

    EXPECT_EQ(valueSize2,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          size,
                                                          name,
                                                          key2));
    EXPECT_STREQ(value2, buffer);

    // 設定項目値を空に設定可能
    ::nn::settings::fwdbg::SetSettingsItemValue(name, key2, "", 0);
    EXPECT_EQ(0,
              ::nn::settings::fwdbg::GetSettingsItemValue(buffer,
                                                          size,
                                                          name,
                                                          key2));
}
