﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_SystemEvent.h>
#include <nn/settings/system/settings_Fatal.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/fwdbg/settings_SettingsManagement.h>
#include <nnt/nntest.h>

namespace {

//!< システムイベント
::nn::os::SystemEventType g_Event = {};

} // namespace

//!< 初期状態においてダーティーフラグの変更通知が正しく行われるか
TEST(FatalSuite1, SignalTest1)
{
    ::nn::settings::system::BindFatalDirtyFlagEvent(
        &g_Event, ::nn::os::EventClearMode_ManualClear);

    // バインド直後はシグナル状態にある
    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::FatalDirtyFlagSet flags = {};

    // 初期状態においては全てのフラグが true
    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    EXPECT_EQ(128, flags.CountPopulation());

    // 状態の取得によって全てのフラグが false に
    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    EXPECT_EQ(0, flags.CountPopulation());
}

//!< 言語設定についてダーティーフラグの変更通知が正しく行われるか
TEST(FatalSuite2, SignalTest001)
{
    ::nn::settings::LanguageCode value = {};
    ::nn::settings::GetLanguageCode(&value);

    ::nn::settings::system::FatalDirtyFlagSet flags = {};

    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    ::nn::settings::system::SetLanguageCode(value);

    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    EXPECT_EQ(1, flags.CountPopulation());
    EXPECT_TRUE(flags.Test<
        ::nn::settings::system::FatalDirtyFlag::LanguageCode>());
}

//!< 設定マネージャについてダーティーフラグの変更通知が正しく行われるか
TEST(FatalSuite3, SignalTest1)
{
    ::nn::settings::system::FatalDirtyFlagSet flags = {};

    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    ASSERT_EQ(0, flags.CountPopulation());

    ASSERT_FALSE(::nn::os::TryWaitSystemEvent(&g_Event));

    // システム設定をリセット
    ::nn::settings::fwdbg::ResetSettings(
        ::nn::settings::fwdbg::SettingsTarget_SystemSaveSystem);

    // リセット直後はシグナル状態にある
    EXPECT_TRUE(::nn::os::TryWaitSystemEvent(&g_Event));
    ::nn::os::ClearSystemEvent(&g_Event);

    // リセット直後においては全てのフラグが true
    ::nn::settings::system::GetFatalDirtyFlags(&flags);
    EXPECT_EQ(128, flags.CountPopulation());
}
