﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <string>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/vi.h>

#include "Util.h"
#include "DataStructure.h"
#include "OperationInterpreter.h"
#include "OperationExecuter.h"
#include "FileAccessor.h"
#include "StorageInterface.h"

#include "sdmmcCommon.h"
#include "GraphicMain.h"
#include "GraphicsSystem.h"
#include "FileManager.h"
#include "DisplayData.h"

#include <nn/gc/detail/gc_Util.h>

#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_DebugPad.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

char g_VersionInfo[] =
#include "version.txt"
;

#define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
#define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

const size_t g_SequenceDataBufferSize = 512 * 1024;
u8 g_SequenceDataBuffer[g_SequenceDataBufferSize];

const unsigned int g_RingBufferSize = 128 * 1024;
char g_RingBuffer[g_RingBufferSize];

const u32 g_ExecuterWorkBufferSize = 512 * 1024;
u8 g_ExecuterWorkBuffer[g_ExecuterWorkBufferSize];

const u32 MAX_DIRECTORY_ENTRY_COUNT = 256;
nn::fs::DirectoryEntry directoryEntryList[MAX_DIRECTORY_ENTRY_COUNT];

uintptr_t g_HeapPtrForMalloc;
const size_t MemoryHeapSize          = 420 * 1024 * 1024;
const size_t HeapSizeForMalloc       = 256 * 1024 * 1024;
const size_t ApplicationHeapSize     = 128 * 1024 * 1024;

// スレッド用
nn::os::ThreadType g_Thread;
static const int ThreadPriority = 26;
static const size_t ThreadStackSize = 128 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];
char g_ThreadName[] = "Test";
nn::os::EventType g_ThreadEndEvent;
nn::os::EventType g_ExecuteEvent;

// 画面描画用
FontSystem* g_pFontSystem;
GraphicsSystem* g_pGraphicsSystem;

DisplayData* g_pDisplayData = DisplayData::GetInstance();
GraphicMain g_GraphicMain;

struct CardState
{
    bool isCardInserted;
    bool isCardActivated;
    bool isCardAccessing;
};

CardState g_CardState;


void PrintDebugArray(const char* buffer, const size_t bufferLength)
{
    for(size_t i=0; i<bufferLength; i++)
    {
        if(i!=0 && i % 16 == 0)
        {
            NN_SC_LOG_S("\n");
        }

        NN_SC_LOG_S("%02X ", buffer[i]);

        if(i % 8 == 7)
        {
            NN_SC_LOG_S(" ");
        }
    }
    NN_SC_LOG_S("\n");
}

/** enum -> テキスト **/
void GetOperationBuffer(char* pOutBuffer, Operation operationId)
{
    switch(operationId)
    {
    case Operation_Power        : strcpy(pOutBuffer, "Power");break;
    case Operation_Select       : strcpy(pOutBuffer, "Select");break;
    case Operation_Init         : strcpy(pOutBuffer, "Init");break;
    case Operation_Enable       : strcpy(pOutBuffer, "Enable");break;
    case Operation_Seed         : strcpy(pOutBuffer, "Seed");break;
    case Operation_Skip         : strcpy(pOutBuffer, "Skip");break;
    case Operation_Write        : strcpy(pOutBuffer, "Write");break;
    case Operation_Read         : strcpy(pOutBuffer, "Read");break;
    case Operation_WriteRead    : strcpy(pOutBuffer, "WriteRead");break;
    case Operation_SecureErase  : strcpy(pOutBuffer, "SecureErase");break;
    case Operation_Load         : strcpy(pOutBuffer, "Load");break;
    case Operation_Dump         : strcpy(pOutBuffer, "Dump");break;
    case Operation_Insert       : strcpy(pOutBuffer, "Insert");break;
    case Operation_Wait         : strcpy(pOutBuffer, "Wait");break;
    case Operation_Pause        : strcpy(pOutBuffer, "Pause");break;
    case Operation_Register     : strcpy(pOutBuffer, "Register");break;
    case Operation_MeasureBegin : strcpy(pOutBuffer, "Measure Begin");break;
    case Operation_MeasureEnd   : strcpy(pOutBuffer, "Measure End");break;
    case Operation_Identify     : strcpy(pOutBuffer, "Identify");break;
    case Operation_Log          : strcpy(pOutBuffer, "Log");break;
    case Operation_Erase        : strcpy(pOutBuffer, "Erase");break;
    case Operation_Speed        : strcpy(pOutBuffer, "Speed");break;
    case Operation_Partition    : strcpy(pOutBuffer, "Partition");break;
    case Operation_Reg          : strcpy(pOutBuffer, "Reg");break;
    case Operation_WriteParam   : strcpy(pOutBuffer, "WriteParam");break;
    case Operation_Command      : strcpy(pOutBuffer, "Command");break;
    case Operation_GameCardCommand : strcpy(pOutBuffer, "GcCommand");break;
    default:                      strcpy(pOutBuffer, "None");
    }
}


#if defined(NN_BUILD_TARGET_PLATFORM_NX)
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

//------------------------------------------------------------------------------
// 画面描画用スレッド
//------------------------------------------------------------------------------
nn::hid::DebugPadState g_PrevDebugPadState;
nn::hid::TouchScreenState<1> g_PrevTouchScreenState;


template<typename T>
bool IsTrigger(nn::hid::DebugPadState debugPadState)
{
    return !g_PrevDebugPadState.buttons.Test<T>() && debugPadState.buttons.Test<T>();
}

bool IsTouchTrigger()
{
    if(g_PrevTouchScreenState.touches[0].x == 0 && g_PrevTouchScreenState.touches[0].y == 0 )
    {
        return true;
    }
//    NN_LOG("prev X = %d\n", g_PrevTouchScreenState.touches[0].x);
//    NN_LOG("prev Y = %d\n", g_PrevTouchScreenState.touches[0].y);
    return false;
}

/* ボタン別の処理 */
void SetByButton(nn::hid::DebugPadState debugPadState)
{
    if(IsTrigger<nn::hid::DebugPadButton::A>(debugPadState) || (g_GraphicMain.IsExecuteButtonPushed() && IsTouchTrigger()))
    {
        nn::os::SignalEvent(&g_ExecuteEvent);
        g_GraphicMain.TransitToScriptMode();
    }
    else if(IsTrigger<nn::hid::DebugPadButton::B>(debugPadState)|| (g_GraphicMain.IsBackButtonPushed()  && IsTouchTrigger()))
    {
        g_GraphicMain.TransitToMenuMode();
    }
    else if(IsTrigger<nn::hid::DebugPadButton::R>(debugPadState)|| (g_GraphicMain.IsIncreasePageCountButtonPushed()  && IsTouchTrigger()))
    {
        g_GraphicMain.IncreasePageCounter();
    }
    else if(IsTrigger<nn::hid::DebugPadButton::L>(debugPadState)|| (g_GraphicMain.IsDecreasePageCountButtonPushed()  && IsTouchTrigger()))
    {
        g_GraphicMain.DecreasePageCounter();
    }
    else if(IsTrigger<nn::hid::DebugPadButton::Down>(debugPadState))
    {
        g_GraphicMain.IncreaseCursorPos();
    }
    else if(IsTrigger<nn::hid::DebugPadButton::Up>(debugPadState))
    {
        g_GraphicMain.DecreaseCursorPos();
    }
}

/* 画面描画スレッド */
void RunThreadFunction(void* pThreadNum) NN_NOEXCEPT
{
    NN_LOG("Thread Start\n");
    nn::hid::TouchScreenState<1> touchScreenState;
    nn::hid::DebugPadState debugPadState;
    size_t counter = 0;
    while(1)
    {
        if(nn::os::TimedWaitEvent(&g_ThreadEndEvent, nn::TimeSpan::FromMilliSeconds(10)))
        {
            break;
        }
        if(counter % 50 == 0)
        {
            counter = 0;
            if(g_GraphicMain.IsMenuMode())
            {
                FileManager::GetInstance()->Initialize();
            }
        }
        nn::hid::GetTouchScreenState(&touchScreenState);
        nn::hid::GetDebugPadState(&debugPadState);
        g_GraphicMain.SetTouchScreenState(touchScreenState.touches[0].x, touchScreenState.touches[0].y);
        // 画面表示実行
        g_GraphicMain.Execute();
        // ボタンごとの処理
        SetByButton(debugPadState);
        g_PrevDebugPadState = debugPadState;
        g_PrevTouchScreenState = touchScreenState;
        counter++;
    }
}

//------------------------------------------------------------------------------
//  スレッド生成関連の処理
//------------------------------------------------------------------------------
void InitializeThread()
{
    NN_LOG("Initialize Thread\n");
    nn::os::InitializeEvent(&g_ThreadEndEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_ExecuteEvent, false, nn::os::EventClearMode_AutoClear);
    // 待ちイベントの初期設定
    nn::Result result = nn::os::CreateThread( &g_Thread, RunThreadFunction, &g_Thread, g_ThreadStack, ThreadStackSize, ThreadPriority );
    if ( !result.IsSuccess() )
    {
        NN_ABORT("failed to create a thread\n");
    }
    nn::os::SetThreadNamePointer(&g_Thread, g_ThreadName);
    nn::os::StartThread( &g_Thread );
}

void FinalizeThread()
{
    // スレッドの終了
    nn::os::WaitThread( &g_Thread );
    nn::os::DestroyThread( &g_Thread );
    nn::os::FinalizeEvent(&g_ThreadEndEvent);
}

//------------------------------------------------------------------------------
//  アプリの初期化関連の処理
//------------------------------------------------------------------------------

/* 初期設定 */
extern "C" void nninitStartup()
{
    NN_LOG("NNINIT START\n");
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    uintptr_t dataBufferAddress;

    // メモリヒープから DataBuffer で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock(&dataBufferAddress, GetDataBufferSize());
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory for NAND data buffer.");
    StorageInterface::SetDataBufferAddress(reinterpret_cast<uint8_t*>(dataBufferAddress));

    // メモリヒープから malloc で使用するメモリ領域を確保
    result = nn::os::AllocateMemoryBlock( &g_HeapPtrForMalloc, HeapSizeForMalloc );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_HeapPtrForMalloc), HeapSizeForMalloc );
}

void InitializeGraphic()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif
}

void InitializeHid()
{
    nn::hid::InitializeTouchScreen();
    nn::hid::InitializeDebugPad();
}

//------------------------------------------------------------------------------
//  スクリプト実行部分
//------------------------------------------------------------------------------
void ExecuteImpl(u8* sequenceBuffer, size_t bufferSize)
{
    // シーケンスが記述されたバイナリファイルをパースする
    NN_SC_DETAIL_LOG("Parsing sequence binary file ...\n");
    OperationDataManager* dataManager = new OperationDataManager(sequenceBuffer, bufferSize);
    dataManager->ParseDataStructure();

    // インタプリタクラスを準備し、先ほど得られたシーケンスデータを元に実処理を行う
    NN_SC_DETAIL_LOG("Processing given operations ...\n");
    OperationInterpreter& interpreter = OperationInterpreter::GetInstance();
    interpreter.InterpretOperationDataList(dataManager->operationDataArray);

    // 終了処理
    NN_SC_DETAIL_LOG2("Done.\n");
    delete dataManager;
}

void ExecuteScript()
{
    // データを読む
    NN_SC_DETAIL_LOG("Reading sequence binary file ...\n");
    FileManager* pFileManager = FileManager::GetInstance();
    size_t index = g_GraphicMain.GetCursorPos();
    // 画面表示用のスクリプト名設定
    char fileNameBuffer[64];
    pFileManager->GetFileName(fileNameBuffer, sizeof(fileNameBuffer), index);
    g_pDisplayData->SetScriptName(fileNameBuffer, strlen(fileNameBuffer) + 1);

    // スクリプト読み込み
    size_t readDataSize = pFileManager->ReadFile(g_SequenceDataBuffer, g_SequenceDataBufferSize, index);
    ExecuteImpl(g_SequenceDataBuffer, readDataSize);
}

//------------------------------------------------------------------------------
//  我らがメイン
//------------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    NN_LOG("NNMAIN Start\n");
    FileManager::GetInstance()->Mount();
    OperationExecuter::GetInstance().SetBuffer(g_ExecuterWorkBuffer, sizeof(g_ExecuterWorkBuffer));
    g_pDisplayData->SetBuffer(g_RingBuffer, g_RingBufferSize);

    if(argc < 2)
    {
        InitializeGraphic();
        NN_LOG("Graphic main Init\n");
        g_GraphicMain.Initialize(ApplicationHeapSize);
        InitializeHid();
        InitializeThread();
        NN_SC_DETAIL_LOG2("Current System Tick: %ld\n", nn::os::GetSystemTick().GetInt64Value());

        while(1)
        {
            nn::os::WaitEvent(&g_ExecuteEvent);
            ExecuteScript();
            DisplayData::GetInstance()->errorTick  = nn::os::GetSystemTick();
            g_GraphicMain.TransitToScriptEndMode();
        }

        nn::os::SignalEvent(&g_ThreadEndEvent);
        FinalizeThread();
    }
    else
    {
        NN_LOG("\n--- %d sequence execute ---\n", argc - 1);
        for(int i = 1; i < argc; i++)
        {
            NN_LOG("\n-- File name : %s --\n", argv[i]);
            FileAccessor& fileAccessor = FileAccessor::GetInstance();
            size_t readDataSize = 0;
            g_pDisplayData->SetScriptName(argv[i], strlen(argv[i]) + 1);
            fileAccessor.Read(g_SequenceDataBuffer, &readDataSize, g_SequenceDataBufferSize, argv[i]);
            ExecuteImpl(g_SequenceDataBuffer, readDataSize);
        }
        NN_LOG("\n--- Sequences end ---\n");
    }


    // メモリの解放
//    OperationExecuter::GetInstance().Free();
    nn::os::FreeMemoryBlock(reinterpret_cast<uintptr_t>(StorageInterface::GetDataBufferAddress()), GetDataBufferSize());
    nn::os::FreeMemoryBlock(g_HeapPtrForMalloc, HeapSizeForMalloc);
}
