﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>
#include <nn/nn_TimeSpan.h>
#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#include "Util.h"
#include "DataStructure.h"
#include "StorageInterface.h"
#include "OperationInterpreter.h"
#include "OperationExecuter.h"
#include "DisplayData.h"
#include <nn/gc/detail/gc_Util.h>

#if !defined ( NN_BUILD_CONFIG_OS_WIN )
#include "../../../../Iris/Include/nn/hid/system/hid_InputDetection.h"
#endif

OperationInterpreter::OperationInterpreter()
{
    m_LoopCountStackCurrentIndex = -1;
    memset(m_LoopCountStack, 0, maxLoopNestCount);
    logTimer.Start();
}

size_t g_CurrentLineNum = 0;

void AddBuffer(char* buffer, size_t bufferLength)
{
    DisplayData* pDisplayData = DisplayData::GetInstance();
    pDisplayData->SetLog(buffer, bufferLength, g_CurrentLineNum);
    NN_LOG("%s", buffer);
}

OperationData* SearchLoopEndNextOperation(OperationData* pOperation)
{
    int loopOperationCount = -1;
    OperationData* pOperationPoint = pOperation;
    while(loopOperationCount < 0)
    {
        if (pOperationPoint->operationId == Operation_Loop)
        {
            loopOperationCount--;
        }
        else if (pOperationPoint->operationId == Operation_LoopEnd)
        {
            loopOperationCount++;
        }
        pOperationPoint = pOperationPoint->GetNext();
    }
    return pOperationPoint;
}

// pOperation のリストを順々に見ながら InterpretOperationData を呼ぶ
OperationData* OperationInterpreter::InterpretOperationDataList(OperationData* pOperation)
{
    while(pOperation)
    {
        NN_SC_DETAIL_LOG("Index: %d, Operation: %d\n", pOperation->GetIndex(), pOperation->operationId);
        DisplayData::GetInstance()->lineIndex =  pOperation->GetIndex();
        // ループの際の再帰処理
        if (pOperation->operationId == Operation_Loop)
        {
            // ループ数を記録するインデックスを更新
            m_LoopCountStackCurrentIndex++;

            OperationData* nextOperation = NULL;
            u32 loopMaxCount = pOperation->ParseArgument<ArgumentTypeU32>().arg;
            NN_SC_DETAIL_LOG2("Loop begin (%d times)\n", loopMaxCount);
            // loopMaxCount が -1 の時は無限ループ
            for(u32 i=0; ((int)(loopMaxCount) < 0) || (i<loopMaxCount); i++)
            {
                // ループ数を記録する
                m_LoopCountStack[m_LoopCountStackCurrentIndex] = i;
                DisplayData::GetInstance()->loopNum = i;
                NN_SC_DETAIL_LOG("Loop index: %d\n", i);
                nextOperation = InterpretOperationDataList(pOperation->GetNext());
            }
            // ループ回数が 0 だったときは抜け先を調べる
            if(loopMaxCount == 0)
            {
                nextOperation = SearchLoopEndNextOperation(pOperation->GetNext());
            }
            pOperation = nextOperation;

            // ループ数記録のインデックスを元に戻す
            m_LoopCountStackCurrentIndex--;
            continue;
        }
        else if (pOperation->operationId == Operation_LoopEnd)
        {
            NN_SC_DETAIL_LOG("Loop end\n");
            return pOperation->GetNext();
        }

        // 実質的な処理はここ
        pOperation = InterpretOperationData(pOperation);
    }

    return NULL;
}

// InterpretOperationDataList から呼ばれる、各 pOperation の処理記述関数
OperationData* OperationInterpreter::InterpretOperationData(OperationData* pOperation)
{
#if !defined ( NN_BUILD_CONFIG_OS_WIN )
        // TODO : Sleep Disble 対応後削除
        // 無操作状態の解除のための通知
        nn::hid::system::NotifyInputDetector(nn::hid::system::InputSourceId::TouchScreen::Mask);
#endif
    OperationData* pNextOperationData = pOperation->GetNext();
    OperationExecuter& operationExecuter = OperationExecuter::GetInstance();
    nn::Result result = nn::ResultSuccess();
    NN_SC_DETAIL_LOG("Current System Tick: %ld\n", nn::os::GetSystemTick().GetInt64Value());
    DisplayData* pDisplayData = DisplayData::GetInstance();
    pDisplayData->operation = *pOperation;

    switch(pOperation->operationId)
    {
    // Storage Interface に関係するもの
    case Operation_Power:
        {
            NN_SC_DETAIL_LOG("[power]\n");
            result = operationExecuter.Power(pOperation);
        }
        break;
    case Operation_Select:
        {
            NN_SC_DETAIL_LOG("[select]\n");
            result = operationExecuter.Select(pOperation);
        }
        break;
    case Operation_Init:
        {
            NN_SC_DETAIL_LOG("[init]\n");
            result = operationExecuter.Init(pOperation);
        }
        break;
    case Operation_Enable:
        {
            NN_SC_DETAIL_LOG("[enable]\n");
            result = operationExecuter.Enable(pOperation);
        }
        break;
    case Operation_Skip:
        {
            NN_SC_DETAIL_LOG("[skip]\n");
            result = operationExecuter.Skip(pOperation);
        }
        break;
    case Operation_Write:
        {
            NN_SC_DETAIL_LOG("[write]\n");
            result = operationExecuter.WriteRead(pOperation);
        }
        break;
    case Operation_Read:
        {
            NN_SC_DETAIL_LOG("[read]\n");
            result = operationExecuter.WriteRead(pOperation);
        }
        break;
    case Operation_WriteRead:
        {
            NN_SC_DETAIL_LOG("[writeread]\n");
            result = operationExecuter.WriteRead(pOperation);
        }
        break;
    case Operation_SecureErase:
        {
            NN_SC_DETAIL_LOG("[secure_erase]\n");
            result = operationExecuter.SecureErase(pOperation);
        }
        break;
    case Operation_Load:
        {
            NN_SC_DETAIL_LOG("[load]\n");
            result = operationExecuter.Load(pOperation);
        }
        break;
    case Operation_Dump:
        {
            NN_SC_DETAIL_LOG("[dump]\n");
            result = operationExecuter.Dump(pOperation);
        }
        break;
    case Operation_Insert:
        {
            NN_SC_DETAIL_LOG("[insert]\n");
            result = operationExecuter.Insert(pOperation);
        }
        break;

    // StorageInterface に関係ないもの
    case Operation_Seed:
        {
            NN_SC_DETAIL_LOG("[seed]\n");
            result = operationExecuter.Seed(pOperation);
        }
        break;
    case Operation_Wait:
        {
            NN_SC_DETAIL_LOG("[wait]\n");
            result = operationExecuter.Wait(pOperation);
        }
        break;
    case Operation_Pause:
        {
            NN_SC_DETAIL_LOG("[pause]\n");
            result = operationExecuter.Pause(pOperation);
        }
        break;
    case Operation_MeasureBegin:
        {
            NN_SC_DETAIL_LOG("[measurebegin]\n");
            result = operationExecuter.MeasureBegin(pOperation);
        }
        break;
    case Operation_MeasureEnd:
        {
            NN_SC_DETAIL_LOG("[measureend]\n");
            result = operationExecuter.MeasureEnd(pOperation);
        }
        break;
    case Operation_Identify:
        {
            NN_LOG("Error: Operation Identify is not implemented!\n");
        }
        break;
    case Operation_Log:
        {
            NN_SC_DETAIL_LOG("[log]\n");
            result = operationExecuter.Log(pOperation);
        }
        break;
    case Operation_Erase:
        {
            NN_SC_DETAIL_LOG("[erase]\n");
            result = operationExecuter.Erase(pOperation);
        }
        break;
    case Operation_Speed:
        {
            NN_SC_DETAIL_LOG("[speed]\n");
            result = operationExecuter.Speed(pOperation);
        }
        break;
    case Operation_Partition:
        {
            NN_SC_DETAIL_LOG("[partition]\n");
            result = operationExecuter.Partition(pOperation);
        }
        break;
    case Operation_Reg:
        {
            NN_SC_DETAIL_LOG("[reg]\n");
            result = operationExecuter.Reg(pOperation);
        }
        break;
    case Operation_WriteParam:
        {
            NN_SC_DETAIL_LOG("[writeparam]\n");
            result = operationExecuter.WriteParam(pOperation);
        }
        break;
    case Operation_Command:
        {
            NN_SC_DETAIL_LOG("[command]\n");
            result = operationExecuter.Command(pOperation);
        }
        break;
    case Operation_GameCardCommand:
        {
            NN_SC_DETAIL_LOG("[gamecard]\n");
            result = operationExecuter.GameCardCommand(pOperation);
        }
    break;
    // その他：制御構文などここに来ることを想定しないもの
    case Operation_None:
    case Operation_Loop:
    case Operation_LoopEnd:
    default:
        {
            NN_ABORT("Unexpected Operation %d\n", pOperation->operationId);
        }
        break;
    }

    nn::gc::detail::MemoryLogger::GetInstance().OutputLog(AddBuffer);

    // 処理に失敗した場合は ABORT する
    if(result.IsFailure())
    {
        NN_LOG("- * Error occurred * -\n");
        NN_LOG("Result Failure - module: %d, description: %d\n", result.GetModule(), result.GetDescription());
        NN_LOG("-------- Error Info --------\n");
        PrintErrorInfos();
        NN_LOG("-------- DRIVER LOG --------\n");
        pDisplayData->isError = true;
        pDisplayData->result  = result;
        pDisplayData->errorTick  = nn::os::GetSystemTick();
        pDisplayData->isSetting = true;
        pDisplayData->isSetting = false;

        NN_SC_DETAIL_ERR_LOG("- * Error occurred * -\n");
        if(IsInLoop())
        {
            NN_SC_DETAIL_ERR_LOG("");
            PrintLoopIndex();
        }
        NN_SC_DETAIL_ERR_LOG("Current System Tick: %ld\n", nn::os::GetSystemTick().GetInt64Value());
        NN_SC_DETAIL_ERR_LOG("Operation Info: index: %d, %d\n", pOperation->GetIndex(), pOperation->operationId);

        NN_SC_DETAIL_ERR_LOG("Result Failure - module: %d, description: %d", result.GetModule(), result.GetDescription());
        nn::gc::detail::MemoryLogger::GetInstance().OutputLog(AddBuffer);
        while (true)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }


    operationExecuter.AddOperationCount();

    // 次のオペレーションを返す
    return pNextOperationData;
} // NOLINT (readability/fn_size)

void OperationInterpreter::PrintLoopIndex()
{
    NN_LOG("Loop index: ");
    for(int i = 0; i <= m_LoopCountStackCurrentIndex; i++)
    {
        NN_LOG("%d, ", m_LoopCountStack[i]);
    }
    NN_LOG("\n");
}

void OperationInterpreter::PrintErrorInfos()
{
    DisplayData* pDisplayData = DisplayData::GetInstance();

    // スクリプト名表示
    char buffer[64];
    pDisplayData->GetScriptName(buffer);
    NN_LOG("  Script name : %s\n", buffer);

    // 行数表示
    NN_LOG("  Line : %d\n", pDisplayData->lineIndex);

    // オペレーション名表示
    GetOperationBuffer(buffer, pDisplayData->operation.operationId);
    NN_LOG("  Operation : %s\n", buffer);

    // Loop 数表示
    NN_LOG("  Loop : %d\n", pDisplayData->loopNum);

    // セクタ範囲表示
    NN_LOG("  Sector : 0x%x->0x%x\n", pDisplayData->startSectorIndex, pDisplayData->startSectorIndex + pDisplayData->totalSectorNum);

    // 現在アクセス中のセクタ表示
    if(pDisplayData->operation.operationId == Operation_Read  ||
       pDisplayData->operation.operationId == Operation_Write ||
       pDisplayData->operation.operationId == Operation_WriteRead )
    {
        NN_LOG("CurrentSector : 0x%x\n", pDisplayData->currentSecoterIndex);
        NN_LOG("ChunkSize : 0x%x\n", pDisplayData->chunkSize);
    }
}

