﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Types.h"
#include <cstdio>
#include <cinttypes>
#include "StorageInterface.h"

struct MeasuredData
{
    u32 group; // sdmmc read/write はある一つの converter read/write から複数生み出されており、そのグルーピング
    u32 address;    // 読み出しアドレス
    int64_t size;   // 何 Byte 読みだしたか
    int64_t unitSize; // 一度の読み出しサイズ
    int64_t time;   // 読み出しに何usかかったか（注意：元は int64_t - int64_t）
    float speed;    // 読み出し速度
    int64_t mintime;    //かかった時間の最小値
    int64_t maxtime;    //かかった時間の最大値
    u32 operationType;  // write なのか read なのか
};

// 計測されたデータを格納するクラス
class MeasuredDataManager
{
    MeasuredData* m_DataBuffer;
    u32 m_CurrentIndex;

public:
    MeasuredDataManager()
    {
        // TEMP: とりあえず 1 MB 確保する
        NN_LOG("MeasuredDataManager: allocate memory %d bytes\n", 256 * 1024);
        m_DataBuffer = (MeasuredData *)std::malloc(1 * 1024 * 1024);
        m_CurrentIndex = 0;
    }
    ~MeasuredDataManager()
    {
        std::free(m_DataBuffer);
    }
    // データを追加する関数
    void AddData(u32 group, u32 address, int64_t size, int64_t unitSize, int64_t time, float speed, int64_t mintime, int64_t maxtime, u32 operationType)
    {
        MeasuredData& data = m_DataBuffer[m_CurrentIndex];
        data.group = group;
        data.address = address;
        data.size = size;
        data.unitSize = unitSize;
        data.time = time;
        data.speed = speed;
        data.mintime = mintime;
        data.maxtime = maxtime;
        data.operationType = operationType;

        m_CurrentIndex++;
    }
    // 出力データを取得する関数
    static size_t GetOutputBufferHeader(u8* outputBuffer)
    {
        return sprintf((char*)outputBuffer, "group,address,size,unitSize[Byte],totaltime[us],speed[MB/s],mintime[us],maxtime[us],type\n");
    }
    size_t GetOutputBuffer(u8* outputBuffer)
    {
        size_t totalSize = 0;
        for(u32 i = 0; i < m_CurrentIndex; i++)
        {
            MeasuredData& data = m_DataBuffer[i];
            totalSize += sprintf((char*)outputBuffer + totalSize, "%d,%d,%" PRId64 ",%" PRId64 ",%" PRId64 ",%f,%" PRId64 ",%" PRId64 ",%d\n", data.group, data.address, data.size, data.unitSize, data.time, data.speed, data.mintime, data.maxtime, data.operationType);
        }
        return totalSize;
    }
};

class OperationExecuter
{
public:
    void Free();
    static OperationExecuter &GetInstance()
    {
        static OperationExecuter m_Instance;
        return m_Instance;
    }

    void SetBuffer(void* workBuffer, size_t workBufferSize)
    {
        m_WorkBuffer = reinterpret_cast<u8*>(workBuffer);
        m_WorkBufferSize = workBufferSize;
    }

    nn::Result Power(OperationData* const pOperation);
    nn::Result Select(OperationData* const pOperation);
    nn::Result Init(OperationData* const pOperation);
    nn::Result Enable(OperationData* const pOperation);
    nn::Result Seed(OperationData* const pOperation);
    nn::Result Wait(OperationData* const pOperation);
    nn::Result Skip(OperationData* const pOperation);
    nn::Result WriteRead(OperationData* const pOperation);
    nn::Result SecureErase(OperationData* const pOperation);
    nn::Result Load(OperationData* const pOperation);
    nn::Result Dump(OperationData* const pOperation);
    nn::Result Insert(OperationData* const pOperation);
    nn::Result Pause(OperationData* const pOperation);
    nn::Result Register(OperationData* const pOperation);
    nn::Result MeasureBegin(OperationData* const pOperation);
    nn::Result MeasureEnd(OperationData* const pOperation);
    nn::Result Log(OperationData* const pOperation);
    nn::Result Erase(OperationData* const pOperation);
    nn::Result Speed(OperationData* const pOperation);
    nn::Result Partition(OperationData* const pOperation);
    nn::Result Reg(OperationData* const pOperation);
    nn::Result WriteParam(OperationData* const pOperation);
    nn::Result Command(OperationData* const pOperation);
    nn::Result GameCardCommand(OperationData* const pOperation);

    // 現在の StorageInterface を取得
    StorageInterface& GetStorage()
    {
        NN_ABORT_UNLESS(m_StorageInterfaceType < sizeof_StorageInterfaceType, "Invalid argument for Init\n");
        return *(m_pStorageInterfaceArray[m_StorageInterfaceType]);
    }

    void AddOperationCount()
    {
        m_OperationCount++;
    }

private:
    int64_t m_LogIntervalUsec;

    u8* m_WorkBuffer;
    size_t m_WorkBufferSize;

    u32 m_OperationCount;
    bool m_IsMeasure;

    OperationExecuter();

    u32 m_SkipPageCount;

    StorageInterfaceType m_StorageInterfaceType;
    StorageInterface* m_pStorageInterfaceArray[sizeof_StorageInterfaceType];

    MeasuredDataManager m_WriteMeasuredData;
    MeasuredDataManager m_ReadMeasuredData;

    // void WriteReadSimple(WriteReadDirection direction, ArgumentTypeWriteRead& arg, u32 pageNumber);
    void AddMeasuredData(bool isWrite, u32 pageCount, u32 pageUnit, int64_t time);
    u32 GetAccessPage(ArgumentTypeWriteRead& arg, size_t countIndex);
    void GenerateSectorData(u8* sectorBuffer, u32 pageAddress, WriteReadPattern pattern);
    nn::Result PrepareWriteData(u8* targetBuffer, WriteReadPattern pattern, size_t pageAddress, size_t size);
    bool IsReadDataMatch(u8* targetBuffer, WriteReadPattern pattern, size_t pageAddress, size_t size);

    nn::Result SendReadCardId1();
    nn::Result SendReadCardId2();
    nn::Result SendReadCardId3();
    nn::Result SendReadCardUid();
    nn::Result SendReadRefresh();
    nn::Result SendReadSetKey();
    nn::Result SendReadSelfRefresh();
    nn::Result SendReadRefreshStatus();
    nn::Result SendReadPage(u32 pageAddress, u32 pageCount);
};
