﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdio>
#include <string>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include "FontSystem.h"
#include "Color.h"
#include "GraphicsSystem.h"
#include "ApplicationHeap.h"
#include "DisplayData.h"

#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>

const int FrameRate = 60;

enum StorageCheckerMode
{
    StorageCheckerMode_Menu,
    StorageCheckerMode_Script,
    StorageCheckerMode_ScriptEnd,
    StorageCheckerMode_ScriptError,
};

class GraphicMain
{
private:
    ApplicationHeap m_ApplicationHeap;
    GraphicsSystem* m_pGraphicsSystem;
    FontSystem*     m_pFontSystem;
    DisplayData*    m_pDisplayData;
    StorageCheckerMode m_StorageCheckerMode;
    size_t m_CursorPos;
    nn::os::Tick m_StartTime;
    nn::os::Tick m_EndTime;
    int32_t m_TouchedX;
    int32_t m_TouchedY;
    size_t m_ErrorStartPos;
    size_t m_CurrentPagePos;
    size_t m_CurrentMenuPagePos;
    size_t m_DisplayableByteSize;
    size_t m_DisplayableFileNum;
    size_t m_FileNum;
    bool m_IsNextPageButtonPushed;
    bool m_IsBackPageButtonPushed;
    bool m_IsExecuteButtonPushed;
    bool m_IsBackButtonPushed;

private:
    bool SetTouchableTextAndDetectTouch(float x, float y, char* buffer);
    bool SetTouchableSquareAndDetectTouch(float x, float y, float width, float height);
    bool SetButtonAndBuffer(float x, float y, float width, float height, char* buffer);
    void SetLine(float startX, float startY, float endX, float endY);
    void SetAndDrawText();
    void SetTextForMenu();
    void SetTouchTextForMenu();
    void SetErrorLog();
    void SetFixedTextForMenu();
    void SetTextForScript();
    void SetTouchTextForScript();
    void SetErrorText();
    void SetLabelForMenu(char* buffer, size_t line);
    void SetTextForGameCardCommand();
    void DisplayCommand();
    void DrawText()
    {
        m_pGraphicsSystem->BeginDraw();
        m_pFontSystem->Draw();
        m_pGraphicsSystem->EndDraw();
        m_pGraphicsSystem->Synchronize(nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
    }
    size_t GetDisplayableFilePos()
    {
        size_t displayableLimitPos = m_DisplayableFileNum;
        if(m_CurrentMenuPagePos == m_FileNum / m_DisplayableFileNum)// 最終ページ
        {
            displayableLimitPos = m_FileNum % m_DisplayableFileNum;
        }
        return displayableLimitPos;
    }

    bool IsExistNextPage()
    {
        return m_FileNum > m_DisplayableFileNum * (m_CurrentMenuPagePos + 1);
    }

    bool IsExistPreviousPage()
    {
        return m_CurrentMenuPagePos > 0;
    }

public:
    GraphicMain();
    void Initialize(size_t applicationHeapSize);
    void Execute();
    void SetTouchScreenState(int32_t x, int32_t y)
    {
        m_TouchedX = x;
        m_TouchedY = y;
    }
    size_t GetCursorPos() {return m_DisplayableFileNum * m_CurrentMenuPagePos + m_CursorPos;}

    bool IsMenuMode()
    {
        return m_StorageCheckerMode == StorageCheckerMode_Menu;
    }
    void TransitToScriptMode()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_Menu)
        {
            m_StartTime = nn::os::GetSystemTick();
            m_ErrorStartPos = 0;
            m_CurrentPagePos = 0;
            m_StorageCheckerMode = StorageCheckerMode_Script;
        }
    }
    void TransitToScriptEndMode()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_Script)
        {
            m_StorageCheckerMode = StorageCheckerMode_ScriptEnd;
        }
    }
    void TransitToMenuMode()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_ScriptEnd || m_StorageCheckerMode == StorageCheckerMode_ScriptError)
        {
            m_pDisplayData->Reset();
            m_StorageCheckerMode = StorageCheckerMode_Menu;
        }
    }

    void IncreaseCursorPos()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_Menu)
        {
            m_CursorPos++;
            if(m_CursorPos == GetDisplayableFilePos())
            {
                m_CursorPos = 0;
            }
        }
    }

    void DecreaseCursorPos()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_Menu)
        {
            if(m_CursorPos == 0)
            {
                m_CursorPos = GetDisplayableFilePos();
            }
            m_CursorPos--;
        }
    }

    void IncreasePageCounter()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_ScriptEnd && m_pDisplayData->operation.operationId == Operation_GameCardCommand)
        {
            if(m_pDisplayData->bufferSizeForGameCardCommand > m_DisplayableByteSize * (m_CurrentPagePos + 1))
            {
                m_CurrentPagePos++;
                m_CursorPos = 0;
            }
        }
        else if(m_StorageCheckerMode == StorageCheckerMode_Menu)
        {
            if(IsExistNextPage())
            {
                m_CurrentMenuPagePos++;
                m_CursorPos = 0;
            }
        }
    }
    void DecreasePageCounter()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_ScriptEnd && m_pDisplayData->operation.operationId == Operation_GameCardCommand)
        {
            if(m_CurrentPagePos > 0)
            {
                m_CurrentPagePos--;
            }
        }
        else if(m_StorageCheckerMode == StorageCheckerMode_Menu)
        {
            if(IsExistPreviousPage())
            {
                m_CurrentMenuPagePos--;
            }
        }
    }
    bool IsExecuteButtonPushed()
    {
        if((m_StorageCheckerMode != StorageCheckerMode_Menu) || !m_IsExecuteButtonPushed)
        {
            return false;
        }
        m_IsExecuteButtonPushed = false;
        return true;
    }
    bool IsBackButtonPushed()
    {
        if(m_StorageCheckerMode == StorageCheckerMode_Menu || !m_IsBackButtonPushed)
        {
            return false;
        }
        m_IsBackButtonPushed = false;
        return true;
    }
    bool IsIncreasePageCountButtonPushed()
    {
        if(!m_IsNextPageButtonPushed)
        {
            return false;
        }
        m_IsNextPageButtonPushed = false;
        return true;
    }
    bool IsDecreasePageCountButtonPushed()
    {
        if(!m_IsBackPageButtonPushed)
        {
            return false;
        }
        m_IsBackPageButtonPushed = false;
        return true;
    }
};
