﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <string>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>

#include "DisplayData.h"
#include "FileManager.h"
#include "OperationExecuter.h"
#include "Types.h"
#include "Util.h"
#include "GraphicMain.h"
#include "StorageInterface.h"

namespace{
const float ButtonPosx   = 1000;
const float ButtonPosy   = 30;
const float ButtonWidth  = 250;
const float ButtonHeight = 100;
}

GraphicMain::GraphicMain()
{
    m_pDisplayData    = DisplayData::GetInstance();
    m_StorageCheckerMode = StorageCheckerMode_Menu;
    m_CursorPos = 0;
    m_TouchedX  = 0;
    m_TouchedY  = 0;
    m_ErrorStartPos = 0;
    m_CurrentPagePos = 0;
    m_CurrentMenuPagePos = 0;
    m_DisplayableByteSize = 256;
    m_DisplayableFileNum  = 6;
    m_FileNum = 0;
    m_IsNextPageButtonPushed = false;
    m_IsBackPageButtonPushed = false;
    m_IsExecuteButtonPushed = false;
    m_IsBackButtonPushed = false;
}

void GraphicMain::Initialize(size_t applicationHeapSize)
{
    m_ApplicationHeap.Initialize(applicationHeapSize);
    m_pGraphicsSystem = new GraphicsSystem();
    m_pGraphicsSystem->SetApplicationHeap(&m_ApplicationHeap);
    m_pGraphicsSystem->Initialize();
    m_pFontSystem = new FontSystem();
    m_pFontSystem->SetApplicationHeap(&m_ApplicationHeap);
    m_pFontSystem->SetGraphicsSystem(m_pGraphicsSystem);
    m_pFontSystem->Initialize();
}

//------------------------------------------------------------------------------
// 描画部品
//------------------------------------------------------------------------------


bool GraphicMain::SetTouchableTextAndDetectTouch(float x, float y, char* buffer)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetCursor(x, y);
    textWriter.Print(buffer);
    float width  = textWriter.CalculateStringWidth(buffer) + 10;
    float height = textWriter.CalculateStringHeight(buffer) + 20;

    if(m_TouchedX > x &&
       m_TouchedX < x + width &&
       m_TouchedY > y&&
       m_TouchedY < y + height)
    {
        return true;
    }
    return false;
}

void GraphicMain::SetLine(float startX, float startY, float endX, float endY)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    char buffer = '*';
    textWriter.SetScale(1.2, 1.2);

    float dx = endX - startX;
    float dy = endY - startY;

    if(dx == 0 && dy == 0)
    {
        textWriter.SetCursor(startX, startY);
        textWriter.Print(&buffer);
    }
    else
    {
        // 描画する点の個数を決定します
        float lineLength = 1.0f / nn::util::Rsqrt(dx * dx + dy * dy);
        int pointNum = lineLength / 2.5;

        for(int i = 0; i <= pointNum; i++)
        {
            float x = (startX * (pointNum - i) + endX * i) / pointNum;
            float y = (startY * (pointNum - i) + endY * i) / pointNum;
            textWriter.SetCursor(x, y);
            textWriter.Print(&buffer);
        }
    }
}

bool GraphicMain::SetTouchableSquareAndDetectTouch(float x, float y, float width, float height)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Green);
    SetLine(x, y, x + width, y);
    SetLine(x, y, x, y + height);
    SetLine(x + width, y, x + width, y + height);
    SetLine(x, y + height, x + width, y + height);
    if(m_TouchedX > x &&
       m_TouchedX < x + width &&
       m_TouchedY > y&&
       m_TouchedY < y + height)
    {
        return true;
    }
    return false;
}

bool GraphicMain::SetButtonAndBuffer(float x, float y, float width, float height, char* buffer)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    auto color = textWriter.GetTextColor();
    bool ret = false;
    ret = SetTouchableSquareAndDetectTouch(x, y, width, height);
    textWriter.SetScale(2.4, 2.4);
    float textWidth  = textWriter.CalculateStringWidth(buffer) - 10;
    float textHeigth = textWriter.CalculateStringHeight(buffer) - 20;
    textWriter.SetTextColor(color);
    textWriter.SetCursor(x + (width - textWidth) / 2, y + (height - textHeigth) / 2);
    textWriter.Print(buffer);
    return ret;
}


//------------------------------------------------------------------------------
// 描画メイン
//------------------------------------------------------------------------------

/* モード別テキスト描画 */
void GraphicMain::SetAndDrawText()
{
    switch(m_StorageCheckerMode)
    {
    case StorageCheckerMode_Menu:
        SetTouchTextForMenu();
        SetTextForMenu();
        break;
    case StorageCheckerMode_Script:
        SetTextForScript();
        break;
    case StorageCheckerMode_ScriptEnd:
        SetTextForScript();
        SetTouchTextForScript();
        break;
    case StorageCheckerMode_ScriptError:
        SetTextForScript();
        SetTouchTextForScript();
        SetErrorLog();
        break;
    default:
        break;
    }
    DrawText();
}

/* メインでループする描画 API*/
void GraphicMain::Execute()
{
    SetAndDrawText();
}

//------------------------------------------------------------------------------
// メニュー画面用描画
//------------------------------------------------------------------------------

/* 選択画面でラベル設定*/
void GraphicMain::SetLabelForMenu(char* buffer, size_t line)
{
    const size_t StartPosX    = 50;
    const size_t StartPosY    = 250;
    const size_t HeightBase   = 60;
    const size_t WidthBase    = 40;

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    // 選択内容表示
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.0, 2.0);

    if(m_CursorPos == line)
    {
        textWriter.SetTextColor(Color::Orange);
        textWriter.SetCursor(StartPosX, StartPosY + HeightBase * line);
        textWriter.Print(">");
    }
    if(SetTouchableTextAndDetectTouch(StartPosX + WidthBase, StartPosY + HeightBase * line, buffer))
    {
        m_CursorPos = line;
    }
}

/* 選択画面での描画 */
void GraphicMain::SetTouchTextForMenu()
{
    const float LabelY = 650;
    const float BackPageStartX = 0;
    const float NextPageStartX = 300;
    char buffer[64];

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Green);
    textWriter.SetScale(2.0, 2.0);

    // ページ送り 表示
    if(IsExistPreviousPage())
    {
        strcpy(buffer, "<< Before");
        if(SetTouchableTextAndDetectTouch(BackPageStartX, LabelY, buffer))
        {
            m_IsBackPageButtonPushed = true;
        }
    }
    if(IsExistNextPage())
    {
        strcpy(buffer, "Next >>");
        if(SetTouchableTextAndDetectTouch(NextPageStartX, LabelY, buffer))
        {
            m_IsNextPageButtonPushed = true;
        }
    }
    strcpy(buffer,  "EXEC");
    textWriter.SetTextColor(Color::White);
    if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
    {
        m_IsExecuteButtonPushed = true;
    }
}

void GraphicMain::SetFixedTextForMenu()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    // 選択内容表示
    textWriter.SetTextColor(Color::Gray);
    textWriter.SetScale(2.4, 2.4);

    // フォルダ表示
    textWriter.SetCursor(0, 40);
    textWriter.Print("Folder : C:/ref/hostIO/StorageChecker");

    // Definition files 表示
    textWriter.SetCursor(20, 140);
    textWriter.Print("Sequence files");
}

void GraphicMain::SetTextForMenu()
{
    SetTouchTextForMenu();
    SetFixedTextForMenu();

    FileManager* pFileManager = FileManager::GetInstance();
    m_FileNum = pFileManager->GetFileNum();
    char buffer[64];
    for(int64_t i = 0 ; i < m_DisplayableFileNum; i++)
    {
        int64_t curPos = m_DisplayableFileNum * m_CurrentMenuPagePos + i;
        if(curPos < m_FileNum)
        {
            pFileManager->GetFileName(buffer, sizeof(buffer), curPos);
            SetLabelForMenu(buffer, i);
        }
    }
}

//------------------------------------------------------------------------------
// スクリプト描画用スレッド
//------------------------------------------------------------------------------

/** ドライバのログを画面表示 **/
void GraphicMain::SetErrorLog()
{
    const size_t HeightBase = 40;
    const size_t WidthBase = 300;
    size_t lineNum = 0;

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(1.5, 1.6);
    textWriter.SetCursor(WidthBase - 30, 60);
    textWriter.Print("[ERROR LOG]");

    // ログ表示
    textWriter.SetScale(0.85, 1.3);

//    NN_LOG("y = %d\n",y);
    if(m_TouchedY <= 300 && m_TouchedY > 0)
    {
        m_ErrorStartPos += 1;
    }
    if(m_TouchedY > 400 && m_TouchedY < 700)
    {
        if(m_ErrorStartPos >= 1)
        {
            m_ErrorStartPos -= 1;
        }
    }
//    NN_LOG("Start Pos = %d\n", m_ErrorStartPos);
    for(size_t i = 0; i < LogLineMaxNum; i++)
    {
        textWriter.SetCursor(WidthBase, 100 + HeightBase * lineNum++);
        char buffer[128];
        memset(buffer, 0x00, sizeof(buffer));
        m_pDisplayData->GetLog(buffer, sizeof(buffer), m_ErrorStartPos + LogLineMaxNum - (i + 1));
        textWriter.Print(buffer);
    }
}

/** コマンドバラ打ち画面表示 **/
void GraphicMain::DisplayCommand()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    OperationExecuter& executer = OperationExecuter::GetInstance();
    char columBuffer[128];
    const size_t ColumnSize   = 16;
    const size_t StartPosX    = 450;
    const size_t StartPosY    = 180;
    const size_t AddressWidth = 120;
    const size_t LineHeight   = 28;
    const size_t LetterWidth  = 38;

    textWriter.SetCursor(StartPosX - 40, StartPosY);
    textWriter.SetScale(2.0, 2.0);
    textWriter.Print("[Result]");
    textWriter.SetScale(1.3, 1.3);

    auto arg = m_pDisplayData->operation.ParseArgument<ArgumentTypeU32_3>();
    size_t basePos  = arg.arg2;
    size_t startPos = m_CurrentPagePos * m_DisplayableByteSize;
    strcpy(columBuffer,"");

    // アドレスオフセットのみ描画
    for(size_t i = 0; i < ColumnSize; i++)
    {
        textWriter.SetCursor(StartPosX + AddressWidth + LetterWidth * i, StartPosY);
        textWriter.Print("%02zX", i);
    }

    // 線のみ描画
    strcpy(columBuffer, "------------------------------------------------------------");
    textWriter.SetCursor(StartPosX + AddressWidth, StartPosY + LineHeight);
    textWriter.Print(columBuffer);

    // アドレスのみ描画
    for(size_t i = 0; i < m_DisplayableByteSize / 16; i++)
    {
        textWriter.SetCursor(StartPosX, StartPosY + LineHeight * (i + 2));
        textWriter.Print("%06zX |", basePos + startPos + i * 16);
    }

    // バッファ描画
    for(size_t i = 0; i < m_DisplayableByteSize / 16; i++)
    {
        strcpy(columBuffer, "");
        for(size_t j = 0; j < 16; j++)
        {
            size_t curPos = startPos + i * 16 + j;
            if(curPos < m_pDisplayData->bufferSizeForGameCardCommand)
            {
                textWriter.SetCursor(StartPosX + AddressWidth + LetterWidth * j, StartPosY + LineHeight * (i + 2));
                textWriter.Print("%02X", executer.GetStorage().dataBuffer[curPos]);
            }
        }
    }

    const float LabelY = 60;
    const float BackPageStartX = 500;
    const float NextPageStartX = 800;
    char buffer[64];

    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.0, 2.0);

    // ページ送り 表示
    strcpy(buffer, "<< Before");
    if(SetTouchableTextAndDetectTouch(BackPageStartX, LabelY, buffer))
    {
        m_IsBackPageButtonPushed = true;
    }
    strcpy(buffer, "Next >>");
    if(SetTouchableTextAndDetectTouch(NextPageStartX, LabelY, buffer))
    {
        m_IsNextPageButtonPushed = true;
    }
}

/** スクリプト画面のタッチ用テキスト表示 **/
void GraphicMain::SetTouchTextForScript()
{
    char buffer[64];
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Gray);
    textWriter.SetScale(1.8, 1.8);
    strcpy(buffer, "Menu");
    textWriter.SetTextColor(Color::White);
    if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
    {
        m_IsBackButtonPushed = true;
    }

}

/** スクリプト画面表示 **/
void GraphicMain::SetTextForScript()
{
    const size_t HeightBase = 50;
    size_t lineNum = 0;
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);

    nn::os::Tick targetTick = nn::os::GetSystemTick();
    if(m_pDisplayData->isError)
    {
        m_StorageCheckerMode = StorageCheckerMode_ScriptError;
        targetTick = m_pDisplayData->errorTick;
        textWriter.SetTextColor(Color::Red);
    }
    else if(m_StorageCheckerMode == StorageCheckerMode_ScriptEnd)
    {
        targetTick = m_pDisplayData->errorTick;
        textWriter.SetTextColor(Color::Blue);
    }
    textWriter.SetScale(1.5, 2);

    char buffer[64];

    // 時刻表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    nn::TimeSpan timeSpan = (targetTick - m_StartTime).ToTimeSpan();
    textWriter.Print("Time : %ld日%ld時%ld分%ld秒%ldミリ秒", timeSpan.GetDays(), timeSpan.GetHours() % 24, timeSpan.GetMinutes() % 60, timeSpan.GetSeconds() % 60, timeSpan.GetMilliSeconds() % 1000);

    // Script 名表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    m_pDisplayData->GetScriptName(buffer);
    textWriter.Print("Script Name :");
    textWriter.SetCursor(0, HeightBase * lineNum++);
    textWriter.Print(buffer);

    // 行数表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    textWriter.Print("Line : %d", m_pDisplayData->lineIndex);

    // オペレーション名表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    GetOperationBuffer(buffer, m_pDisplayData->operation.operationId);
    char bufferTmp[64];
    strcpy(bufferTmp,"Operation : ");
    strncat(bufferTmp, buffer, strlen(buffer));
    textWriter.Print(bufferTmp);

    // Loop 数表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    textWriter.Print("Loop : %d", m_pDisplayData->loopNum);

    // セクタ範囲表示
    textWriter.SetCursor(0, HeightBase * lineNum++);
    textWriter.Print("Sector : ");
    textWriter.SetCursor(0, HeightBase * lineNum++);
    textWriter.Print("0x%x->0x%x", m_pDisplayData->startSectorIndex, m_pDisplayData->startSectorIndex + m_pDisplayData->totalSectorNum);

    // 現在アクセス中のセクタ表示
    if(m_pDisplayData->operation.operationId == Operation_Read  ||
       m_pDisplayData->operation.operationId == Operation_Write ||
       m_pDisplayData->operation.operationId == Operation_WriteRead )
    {
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("CurrentSector", m_pDisplayData->currentSecoterIndex);
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("0x%x", m_pDisplayData->currentSecoterIndex);
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("ChunkSize", m_pDisplayData->chunkSize);
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("0x%x", m_pDisplayData->chunkSize);
    }
    // コマンドバラ打ち時の出力
    if(m_pDisplayData->operation.operationId == Operation_GameCardCommand)
    {
        DisplayCommand();
    }

    // エラー表示
    if(m_StorageCheckerMode == StorageCheckerMode_ScriptError)
    {
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("Module : %d", m_pDisplayData->result.GetModule());
        textWriter.SetCursor(0, HeightBase * lineNum++);
        textWriter.Print("Description : %d", m_pDisplayData->result.GetDescription());
    }
}

