﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <iostream>
#include <fstream>

#include "AccessPatternManager.h"

#define LOG_ENABLE

using namespace std;

enum AgingPageType
{
    AgingPageType_DEC    = 0x00,
    AgingPageType_INC    = 0x01,
    AgingPageType_ALL00  = 0x02,
    AgingPageType_00FF   = 0x03,
    AgingPageType_55AA   = 0x04,
    AgingPageType_ALLFF  = 0x05,
    AgingPageType_NUM
};

#ifdef LOG_ENABLE
void PrintOnePage(char* pInBuffer)
{
    NN_LOG("Print Register\n");
    for(int i = 0; i < GcMemoryPageSizeForAccessPattern / 16; i++)
    {
        NN_LOG("  ");
        for(int j = 0; j < 16; j++)
        {
            NN_LOG("0x%02x ",pInBuffer[i * 16 + j]);
        }
        NN_LOG("\n");
    }
}
#endif


void CreatePatternData(char* pOutBuffer, u32 currentPageAddress)
{
    u32 agingType = currentPageAddress % AgingPageType_NUM;
    switch(agingType)
    {
    case AgingPageType_INC:
    {
        for(int i = 0; i < GcMemoryPageSizeForAccessPattern; i++)
        {
            pOutBuffer[i]   = i;
        }
    }
    break;
    case AgingPageType_DEC:
    {
        for(int i = 0; i < GcMemoryPageSizeForAccessPattern; i++)
        {
            pOutBuffer[i]   = GcMemoryPageSizeForAccessPattern - 1 - i;
        }
    }
    break;
    case AgingPageType_ALL00:
    {
        std::memset(pOutBuffer, 0, GcMemoryPageSizeForAccessPattern);
    }
    break;
    case AgingPageType_ALLFF:
    {
        std::memset(pOutBuffer, 0xFF, GcMemoryPageSizeForAccessPattern);
    }
    break;
    case AgingPageType_00FF:
    {
        for(int i = 0; i < GcMemoryPageSizeForAccessPattern / 2; i++)
        {
            pOutBuffer[2 * i]   = 0x00;
            pOutBuffer[2 * i + 1] = 0xFF;
        }
    }
    break;
    case AgingPageType_55AA:
    {
        for(int i = 0; i < GcMemoryPageSizeForAccessPattern / 2; i++)
        {
            pOutBuffer[2 * i]   = 0x55;
            pOutBuffer[2 * i + 1] = 0xAA;
        }
    }
    break;
    default:
    {
        // default は 0x33 埋め(ありえない)
        std::memset(pOutBuffer, 0x33, GcMemoryPageSizeForAccessPattern);
    }
    break;
    }
}

void AddPageAddress(char* pOutBuffer, u32 currentPageAddress)
{
    for(size_t i = 0; i < sizeof(u32); i++)
    {
        pOutBuffer[i] = static_cast<char>( (currentPageAddress >> (8 * (sizeof(u32) - 1 - i))) & 0xFF );
    }
}

void SetPageData(char* pOutBuffer, u32 currentPageAddress)
{
    CreatePatternData(pOutBuffer, currentPageAddress);
    AddPageAddress(pOutBuffer, currentPageAddress);
}

void AccessPatternManager::GetAccessPattern(char* pOutBuffer, u32 pageAddress, u32 pageSize)
{
    for(auto counter = 0U; counter < pageSize; counter++)
    {
        u32 currentPageAddress = pageAddress + counter;
        char* pCurrentOutBufferPointer = pOutBuffer + counter * GcMemoryPageSizeForAccessPattern;
        if(currentPageAddress < m_ReservedAreaStartPageAddress)
        {
            SetPageData(pCurrentOutBufferPointer, currentPageAddress);
        }
        else
        {
            std::memset(pCurrentOutBufferPointer, 0xFF, GcMemoryPageSizeForAccessPattern);
        }
    }
}

bool AccessPatternManager::CompAccessPattern(char* pInBuffer, u32 pageAddress, u32 pageSize)
{
    for(auto counter = 0U; counter < pageSize; counter++)
    {
        char temp[GcMemoryPageSizeForAccessPattern];
        u32 currentPageAddress = pageAddress + counter;
        char* pCurrentInBufferPointer = pInBuffer + counter * GcMemoryPageSizeForAccessPattern;
        SetPageData(temp, currentPageAddress);
        if(::std::memcmp(pCurrentInBufferPointer, temp, GcMemoryPageSizeForAccessPattern) != 0)
        {
#ifdef LOG_ENABLE
            NN_LOG("ERROR : PA = 0x%x\n",pageAddress);
            NN_LOG("CUR\n");
            PrintOnePage(pInBuffer);
            NN_LOG("EXP\n");
            PrintOnePage(temp);
#endif
            return false;
        }
    }
    return true;
}

void AccessPatternManager::SetCardSize(u32 binSize)
{
    m_ReservedAreaStartPageAddress = binSize * DefaultReservedAreaStartPageAddress;
}
