﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    #if 0   // MMC
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else   // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
            #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2
                    // NN_BUILD_CONFIG_HARDWARE_NX
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
            #endif
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];

    const uint32_t TestIntervalSeconds = 5;
    const uint32_t TestSectorIndex = 1;
    const uint32_t TestSize = 4 * 1024 * 1024;
    const uint32_t TestNumSectors = TestSize / nn::sdmmc::SectorSize;
    const size_t DataBufferSize = nn::util::align_up(TestSize, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    void SdCardDetectionEventCallback(void* pParameter)
    {
        NN_UNUSED(pParameter);
        NN_LOG("%s\n", nn::sdmmc::IsSdCardInserted(nn::sdmmc::Port_SdCard0) ? "I" : "R");
    }

    void PutToSleepAndWait()
    {
        if (TestPort == nn::sdmmc::Port_SdCard0)
        {
            NN_LOG("nn::sdmmc::PutSdCardToSleep\n");
            nn::sdmmc::PutSdCardToSleep(TestPort);
        }
        else
        {
            NN_LOG("nn::sdmmc::PutMmcToSleep\n");
            nn::sdmmc::PutMmcToSleep(TestPort);
        }

        NN_LOG("Wait %u(s)...\n", TestIntervalSeconds);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(TestIntervalSeconds));
    }

    void Awaken()
    {
        if (TestPort == nn::sdmmc::Port_SdCard0)
        {
            NN_LOG("nn::sdmmc::AwakenSdCard\n");
            nn::sdmmc::AwakenSdCard(TestPort);
        }
        else
        {
            NN_LOG("nn::sdmmc::AwakenMmc\n");
            nn::sdmmc::AwakenMmc(TestPort);
        }
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    if (TestPort == nn::sdmmc::Port_SdCard0)
    {
        nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
        nn::sdmmc::RegisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0, SdCardDetectionEventCallback, nullptr);
    }
    else
    {
        nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }

    uintptr_t dataBufferAddress;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize));

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif

    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    // Activate 前の Sleep 制御
    PutToSleepAndWait();
    Awaken();

    NN_LOG("nn::sdmmc::Activate\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Activate(TestPort));

    // バス状態のダンプ
    LogBusStatus(TestPort);

    // Activate 後の Sleep 制御
    PutToSleepAndWait();
    Awaken();

    NN_LOG("nn::sdmmc::Read\n");
    nn::Result result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
    if (result.IsFailure())
    {
        if (nn::sdmmc::ResultDeviceRemoved::Includes(result))
        {
            NN_LOG("nn::sdmmc::ResultDeviceRemoved\n");
            NN_LOG("nn::sdmmc::Deactivate, Activate\n");
            nn::sdmmc::Deactivate(TestPort);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Activate(TestPort));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    // Read 後の Sleep 制御
    PutToSleepAndWait();
    Awaken();

    NN_LOG("nn::sdmmc::Read\n");
    result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
    if (result.IsFailure())
    {
        if (nn::sdmmc::ResultDeviceRemoved::Includes(result))
        {
            NN_LOG("nn::sdmmc::ResultDeviceRemoved\n");
            NN_LOG("nn::sdmmc::Deactivate, Activate\n");
            nn::sdmmc::Deactivate(TestPort);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::sdmmc::Activate(TestPort));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    NN_LOG("nn::sdmmc::Deactivate\n");
    nn::sdmmc::Deactivate(TestPort);

    // Deactivate 後の Sleep 制御
    PutToSleepAndWait();
    Awaken();

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    if (TestPort == nn::sdmmc::Port_SdCard0)
    {
        nn::sdmmc::UnregisterSdCardDetectionEventCallback(nn::sdmmc::Port_SdCard0);
    }
    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);

    NN_LOG("Done\n");
}
