﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    #if 0   // MMC (NAND データが壊れる可能性があるので要注意)
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else   // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
            #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2
                    // NN_BUILD_CONFIG_HARDWARE_NX
                const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
            #endif
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif
    const bool isAttachMmcToPortSdCard0ForDebug = false;

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBufferForHostController[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::WorkBufferSizeForHostController)];
    #endif

    const uint32_t TestSectorIndex = 1;
    const uint32_t TestNumSectors = 9;
    const size_t DataBufferSize = nn::util::align_up(nn::sdmmc::SectorSize * TestNumSectors, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    const char* GetPortName(nn::sdmmc::Port port)
    {
        switch (port)
        {
        case nn::sdmmc::Port_Mmc0:
            return "Port_Mmc0";
        case nn::sdmmc::Port_SdCard0:
            return "Port_SdCard0";
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void LogData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint8_t* pCurrentData =  reinterpret_cast<uint8_t*>(pData);
        uint8_t* pDataEnd = pCurrentData + dataSize;
        while (true)
        {
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    return;
                }
            }
            NN_LOG("\n");
        }
    }

    void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            pCurrentData[i] = i;
        }
    }

    void CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            NN_ABORT_UNLESS(pCurrentData[i] == i);
        }
    }

    void GetRegisters() NN_NOEXCEPT
    {
        uint32_t ocr;
        nn::Result result = nn::sdmmc::GetDeviceOcr(&ocr, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("OCR:\n");
        NN_LOG("0x%08X\n", ocr);

        uint16_t rca;
        result = nn::sdmmc::GetDeviceRca(&rca, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("RCA:\n");
        NN_LOG("0x%04X\n", rca);

        uint8_t cid[nn::sdmmc::DeviceCidSize];
        result = nn::sdmmc::GetDeviceCid(cid, sizeof(cid), TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("CID:\n");
        LogData(cid, nn::sdmmc::DeviceCidSize);

        uint8_t csd[nn::sdmmc::DeviceCsdSize];
        result = nn::sdmmc::GetDeviceCsd(csd, sizeof(csd), TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("CSD:\n");
        LogData(csd, nn::sdmmc::DeviceCsdSize);

        if ((TestPort == nn::sdmmc::Port_Mmc0) || ((TestPort == nn::sdmmc::Port_SdCard0) && isAttachMmcToPortSdCard0ForDebug))
        {
            result = nn::sdmmc::GetMmcExtendedCsd(g_pDataBuffer, DataBufferSize, TestPort);
            if (nn::sdmmc::ResultMmcNotSupportExtendedCsd::Includes(result))
            {
                NN_LOG("MMC doesn't support Extended CSD\n");
            }
            else
            {
                NN_ABORT_UNLESS(result.IsSuccess());
                NN_LOG("MMC Extended CSD:\n");
                LogData(g_pDataBuffer, nn::sdmmc::MmcExtendedCsdSize);
            }
        }

        if ((TestPort == nn::sdmmc::Port_SdCard0) && (!isAttachMmcToPortSdCard0ForDebug))
        {
            result = nn::sdmmc::GetSdCardScr(g_pDataBuffer, DataBufferSize, TestPort);
            if (result.IsSuccess())
            {
                NN_LOG("SD Card SCR:\n");
                LogData(g_pDataBuffer, nn::sdmmc::SdCardScrSize);
            }
            else
            {
                // 初期化後に SCR を再取得できない SD カードが存在する
                NN_LOG("nn::sdmmc::GetSdCardScr is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
            }

            result = nn::sdmmc::GetSdCardSwitchFunctionStatus(g_pDataBuffer, DataBufferSize,
                TestPort, nn::sdmmc::SdCardSwitchFunction_CheckSupportedFunction);
            if (nn::sdmmc::ResultSdCardNotSupportSwitchFunctionStatus::Includes(result))
            {
                NN_LOG("SD Card doesn't support Switch Function Status\n");
            }
            else
            {
                NN_ABORT_UNLESS(result.IsSuccess());
                NN_LOG("SD Card Switch Function Status (Check Supported Function):\n");
                LogData(g_pDataBuffer, nn::sdmmc::SdCardSwitchFunctionStatusSize);

                result = nn::sdmmc::GetSdCardSwitchFunctionStatus(g_pDataBuffer, DataBufferSize,
                    TestPort, nn::sdmmc::SdCardSwitchFunction_CheckHighSpeed);
                NN_ABORT_UNLESS(result.IsSuccess());
                NN_LOG("SD Card Switch Function Status (Check High Speed):\n");
                LogData(g_pDataBuffer, nn::sdmmc::SdCardSwitchFunctionStatusSize);
            }

            result = nn::sdmmc::GetSdCardSdStatus(g_pDataBuffer, DataBufferSize, TestPort);
            NN_ABORT_UNLESS(result.IsSuccess());
            NN_LOG("SD Card SD Status:\n");
            LogData(g_pDataBuffer, nn::sdmmc::SdCardSdStatusSize);
        }
    }

    void ReadWrite() NN_NOEXCEPT
    {
        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];

        SetData(g_pDataBuffer, nn::sdmmc::SectorSize * TestNumSectors);
        NN_LOG("Write %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), TestSectorIndex, TestNumSectors);
        nn::Result result = nn::sdmmc::Write(TestPort, TestSectorIndex, TestNumSectors, g_pDataBuffer, DataBufferSize);
        if (result.IsFailure())
        {
            NN_LOG("nn::sdmmc::Write is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
            return;
        }
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_LOG("Write had hidden error. %s\n", logBuffer);
            // 以降のアクセスを続ける
        }
        NN_LOG("Done.\n");
        std::memset(g_pDataBuffer, 0xA5, DataBufferSize);
        NN_LOG("Read  %s, sectorIndex=%u, numSectors=%u ...\n", GetPortName(TestPort), TestSectorIndex, TestNumSectors);
        result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
        if (result.IsFailure())
        {
            NN_LOG("nn::sdmmc::Read is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
            return;
        }
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_LOG("Read had hidden error. %s\n", logBuffer);
            // 以降のアクセスを続ける
        }
        NN_LOG("Done.\n");
#if 0
        LogData(g_pDataBuffer, nn::sdmmc::SectorSize * TestNumSectors);
#endif
        NN_LOG("Verify ...\n");
        CheckData(g_pDataBuffer, nn::sdmmc::SectorSize * TestNumSectors);
        NN_LOG("Done.\n");
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    if ((TestPort == nn::sdmmc::Port_SdCard0) && (!isAttachMmcToPortSdCard0ForDebug))
    {
        nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }
    else
    {
        nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
    }

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceAddressSpaceMapInfo infoForHostController;
            uintptr_t hostControllerBufferAddress = reinterpret_cast<uintptr_t>(g_WorkBufferForHostController);
            size_t hostControllerBufferSize = sizeof(g_WorkBufferForHostController);
            nn::dd::DeviceVirtualAddress hostControllerBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &infoForHostController,
                hostControllerBufferAddress, hostControllerBufferSize, deviceVirtualAddressOffset);
            deviceVirtualAddressOffset = hostControllerBufferDeviceVirtualAddress + hostControllerBufferSize;
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
        #endif

        nn::sdmmc::SetWorkBufferForHostController(TestPort, g_WorkBufferForHostController, sizeof(g_WorkBufferForHostController));
    #endif

    uintptr_t dataBufferAddress;
    nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif

    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    // ResultNotActivated チェック
    result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
    NN_ABORT_UNLESS(nn::sdmmc::ResultNotActivated::Includes(result));

    result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        // Activate 重複チェック
        result = nn::sdmmc::Activate(TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numActivationFailures != 0) || (errorInfo.numActivationErrorCorrections != 0))
        {
            NN_LOG("Activate had hidden error. %s\n", logBuffer);
            // 以降のアクセスを続ける
        }

        // 容量取得
        uint32_t numSectorsOfMemoryCapacity = 0;
        result = nn::sdmmc::GetDeviceMemoryCapacity(&numSectorsOfMemoryCapacity, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        if ((TestPort == nn::sdmmc::Port_SdCard0) && (!isAttachMmcToPortSdCard0ForDebug))
        {
            uint32_t numSectorsOfProtectedAreaCapacity = 0;
            result = nn::sdmmc::GetSdCardProtectedAreaCapacity(&numSectorsOfProtectedAreaCapacity, TestPort);
            NN_ABORT_UNLESS(result.IsSuccess());
            NN_LOG("SD Card Protected Area %u (MB)\n", numSectorsOfProtectedAreaCapacity / 2 / 1024);
        }

        // バス状態のダンプ
        LogBusStatus(TestPort);

        // レジスタ取得
        GetRegisters();

        // リード・ライトチェック
        ReadWrite();
    }
    else
    {
        NN_LOG("nn::sdmmc::Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    nn::sdmmc::Deactivate(TestPort);

    // ResultNotActivated チェック
    result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, TestSectorIndex, TestNumSectors);
    NN_ABORT_UNLESS(nn::sdmmc::ResultNotActivated::Includes(result));

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&infoForHostController);
        #endif
    #endif

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
