﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/result/testResult_Assert.h>

namespace
{
    #define ABORT_IF_FAILURE(result, ...) \
        if (result.IsFailure()) { NN_LOG(__VA_ARGS__); NN_LOG(" Result: Module %d, Description %d\n", result.GetModule(), result.GetDescription()); NN_ABORT_UNLESS(result.IsSuccess()); }

    const size_t BufferSize = (8 * 1024 * 1024) + 256;
    char g_Buffer[BufferSize];

    int g_InsertRemoveMaxWaitTime = 30;

    class Time
    {
    public:
        Time() : m_StartTime(0), m_EndTime(0) {}
        ~Time() {}

        void Start()
        {
            m_StartTime = nn::os::GetSystemTick();
        }

        int64_t GetElapsedTimeMsec()
        {
            m_EndTime = nn::os::GetSystemTick();
            return (m_EndTime - m_StartTime).ToTimeSpan().GetMilliSeconds();
        }

        int64_t GetElapsedTimeSec()
        {
            return GetElapsedTimeMsec() / 1000;
        }

    private:
        nn::os::Tick m_StartTime;
        nn::os::Tick m_EndTime;
    };

    void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            pCurrentData[i] = i;
        }
    }

    void CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            NN_ABORT_UNLESS(pCurrentData[i] == i);
        }
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }

    void WaitAttach()
    {
        if( nn::fs::IsSdCardInserted() == false )
        {
            NN_LOG("waiting for attach...\n");
            Time waitTimer;
            waitTimer.Start();
            while( nn::fs::IsSdCardInserted() == false )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                if(waitTimer.GetElapsedTimeSec() > g_InsertRemoveMaxWaitTime)
                {
                    NN_ABORT("wait attach timeout\n");
                }
            }
            NN_LOG("done waiting\n");
        }
        else
        {
            NN_LOG("already attached\n");
        }
    }

    void WaitDetach()
    {
        if( nn::fs::IsSdCardInserted() == true )
        {
            NN_LOG("waiting for detach...\n");
            Time waitTimer;
            waitTimer.Start();
            while( nn::fs::IsSdCardInserted() == true )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
                if(waitTimer.GetElapsedTimeSec() > g_InsertRemoveMaxWaitTime)
                {
                    NN_ABORT("wait detach timeout\n");
                }
            }
            NN_LOG("done waiting\n");
        }
        else
        {
            NN_LOG("already detached\n");
        }
    }

    // result が IsFailure なら返り値は false
    nn::Result CheckResult(nn::Result result, bool isMountApi, bool isFormatApi)
    {
        if(result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        if (nn::fs::ResultPortSdCardDeviceRemoved::Includes(result))
        {
            NN_LOG("# Caught nn::fs::ResultPortSdCardDeviceRemoved\n");
        }
        else if (nn::fs::ResultSdCardFileSystemInvalidatedByRemoved::Includes(result))
        {
            NN_LOG("# Caught nn::fs::ResultSdCardFileSystemInvalidatedByRemoved\n");
        }
        else if((isMountApi || isFormatApi) && nn::fs::ResultPortSdCardNoDevice::Includes(result))
        {
            NN_LOG("# Caught nn::fs::ResultPortSdCardNoDevice\n");
        }
        else if (isFormatApi && nn::fs::ResultSdCardAccessFailed::Includes(result)) // FIXME: SIGLO-48473
        {
            NN_LOG("# Caught nn::fs::ResultSdCardAccessFailed when nn::fs::FormatSdCard (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            // TORIAEZU: SIGLO-48557
            if (nn::fs::ResultSdCardAccessFailed::Includes(result))
            {
                NN_LOG("Warning: ### Unexpected Result (isMount: %d, isFormat: %d) (Module:%d, Description:%d) ###\n", isMountApi, isFormatApi, result.GetModule(), result.GetDescription());
            }
            else
            {
                NN_ABORT("Error: Unexpected Result (isMount: %d, isFormat: %d) (Module:%d, Description:%d)\n", isMountApi, isFormatApi, result.GetModule(), result.GetDescription());
            }
        }
        return result;
    }

    nn::Result CheckResult(nn::Result result, bool isMountApi)
    {
        return CheckResult(result, isMountApi, false);
    }

    nn::Result ReadWriteVerify()
    {
        bool isFailed = true;
        bool isClosed = false;
        NN_UTIL_SCOPE_EXIT
        {
            // isClosed == false の時は CloseFile をするところでログを出す
            if(isFailed && isClosed == true)
            {
                NN_LOG("Failed fs API (scope exit)\n");
            }
        };

        NN_LOG("nn::fs::DeleteFile\n");
        nn::Result result = nn::fs::DeleteFile("sdcard:/data.bin");
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::DeleteFile is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        }
        NN_LOG("nn::fs::CreateFile\n");
        NN_RESULT_DO(nn::fs::CreateFile("sdcard:/data.bin", 0));

        NN_LOG("Write %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
        nn::fs::FileHandle fileHandle;
        NN_LOG("nn::fs::OpenFile\n");
        NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT
        {
            if(isClosed == false)
            {
                NN_LOG("failed (fs API). nn::fs::CloseFile (scope exit)\n");
                nn::fs::CloseFile(fileHandle);
            }
        };
        SetData(g_Buffer, BufferSize);
        NN_LOG("nn::fs::WriteFile\n");
        NN_RESULT_DO(nn::fs::WriteFile(fileHandle, 0, g_Buffer, BufferSize, nn::fs::WriteOption()));
        NN_LOG("nn::fs::FlushFile\n");
        NN_RESULT_DO(nn::fs::FlushFile(fileHandle));
        NN_LOG("nn::fs::CloseFile\n");
        nn::fs::CloseFile(fileHandle);
        isClosed = true;

        NN_LOG("Read %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
        NN_LOG("nn::fs::OpenFile\n");
        NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Read));
        isClosed = false;
        std::memset(g_Buffer, 0xA5, sizeof(g_Buffer));
        NN_LOG("nn::fs::ReadFile\n");
        NN_RESULT_DO(nn::fs::ReadFile(fileHandle, 0, g_Buffer, BufferSize));
        NN_LOG("nn::fs::CloseFile\n");
        nn::fs::CloseFile(fileHandle);
        isClosed = true;

        NN_LOG("Verify ...\n");
        CheckData(g_Buffer, BufferSize);

        isFailed = false;
        NN_RESULT_SUCCESS;
    }
}


#define NN_DETAIL_SDMMC_RESULT_CONTINUE(r) \
    { \
        const auto& _nn_result_do_temporary((r)); \
        if(_nn_result_do_temporary.IsFailure()) \
        { \
            NN_LOG("Detected Result Failure. continue\n"); \
            continue; \
        } \
    }


extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(Allocate, Deallocate);

    bool isMounted = false;

    // 挿抜の試行回数
    int maxTryNum = 3;

    // 引数取得
    {
        int argc = nn::os::GetHostArgc();
        char** argv = nn::os::GetHostArgv();
        if(argc == 1)
        {
            NN_LOG("usage: FsSdCardReadWriteInsertRemove <InsertRemoveCount> <TimeOutSec>\n");
        }
        if(argc > 1)
        {
            maxTryNum = atoi(argv[1]);
        }
        if(argc > 2)
        {
            g_InsertRemoveMaxWaitTime = atoi(argv[2]);
        }
    }
    NN_LOG("## SD card insert/remove test for [%d] times\n", maxTryNum);
    NN_LOG("timeout: %d sec\n", g_InsertRemoveMaxWaitTime);

    bool isVerifyCompleted = false;
    for( int i = 0; i < maxTryNum; i++ )
    {
        WaitDetach();

        NN_LOG("Test loop: %d\n", i);

        WaitAttach();

        if(isMounted)
        {
            NN_LOG("nn::fs::Unmount\n");
            nn::fs::Unmount("sdcard");
            isMounted = false;
        }

        NN_LOG("Test SD Card Access\n");

        // フォーマットは時間がかかるので、 maxTryNum の半分にあてる
        if(i > (maxTryNum / 2))
        {
            NN_LOG("nn::fs::FormatSdCard\n");
            NN_DETAIL_SDMMC_RESULT_CONTINUE(CheckResult(nn::fs::FormatSdCard(), false, true));
        }

        NN_LOG("nn::fs::MoundSdCardForDebug\n");
        NN_DETAIL_SDMMC_RESULT_CONTINUE( CheckResult( nn::fs::MountSdCardForDebug("sdcard"), true));
        isMounted = true;
        NN_LOG("SD Card Mounted\n");

        NN_DETAIL_SDMMC_RESULT_CONTINUE( CheckResult( ReadWriteVerify(), false));
        isVerifyCompleted = true;

        NN_LOG("Done.\n");
    }

    NN_ABORT_UNLESS(isVerifyCompleted, "Error: read/write verification failed in all attempt\n");

    NN_LOG("Test Finished\n");
}




