﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

namespace
{
    #define ABORT_IF_FAILURE(result, ...) \
        if (result.IsFailure()) { NN_LOG(__VA_ARGS__); NN_LOG(" Result: Module %d, Description %d\n", result.GetModule(), result.GetDescription()); NN_ABORT_UNLESS(result.IsSuccess()); }

    const size_t BufferSize = (8 * 1024 * 1024) + 256;
    char g_Buffer[BufferSize];

    void SetData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            pCurrentData[i] = i;
        }
    }

    void CheckData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t* pCurrentData = reinterpret_cast<uint32_t*>(pData);
        for (size_t i = 0; i < (dataSize / sizeof(uint32_t)); i++)
        {
            NN_ABORT_UNLESS(pCurrentData[i] == i);
        }
    }

    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
}

extern "C" void nnMain()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    NN_LOG("Test SD Card Access\n");

    nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
    ABORT_IF_FAILURE(result, "nn::fs::MountSdCardForDebug");

    result = nn::fs::DeleteFile("sdcard:/data.bin");
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::DeleteFile is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }

    result = nn::fs::CreateFile("sdcard:/data.bin", 0);
    ABORT_IF_FAILURE(result, "nn::fs::CreateFile");

    NN_LOG("Write %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    ABORT_IF_FAILURE(result, "nn::fs::OpenFile");
    SetData(g_Buffer, BufferSize);
    result = nn::fs::WriteFile(fileHandle, 0, g_Buffer, BufferSize, nn::fs::WriteOption());
    ABORT_IF_FAILURE(result, "nn::fs::WriteFile");
    result = nn::fs::FlushFile(fileHandle);
    ABORT_IF_FAILURE(result, "nn::fs::FlushFile");
    nn::fs::CloseFile(fileHandle);

    NN_LOG("Read %.1f(KB) ...\n", sizeof(g_Buffer) / 1024.0);
    result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Read);
    ABORT_IF_FAILURE(result, "nn::fs::OpenFile");
    std::memset(g_Buffer, 0xA5, sizeof(g_Buffer));
    result = nn::fs::ReadFile(fileHandle, 0, g_Buffer, BufferSize);
    ABORT_IF_FAILURE(result, "nn::fs::ReadFile");
    nn::fs::CloseFile(fileHandle);

    NN_LOG("Verify ...\n");
    CheckData(g_Buffer, BufferSize);

    nn::fs::Unmount("sdcard");

    NN_LOG("Done.\n");
}
