﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>

// システムを壊すことに注意！
// あらかじめ以下のツールで本体固有情報をバックアップしておくこと。
// http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=115345321

namespace
{
    const size_t DumpSize = 4 * 1024;
    uint8_t g_Buffer[DumpSize];

    nn::Result LogAllSize()
    {
        int64_t mmcPartitionSize = 0;

        NN_RESULT_DO(nn::fs::GetMmcPartitionSize(&mmcPartitionSize, nn::fs::MmcPartition::UserData));
        NN_LOG("User Data Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);

        NN_RESULT_DO(nn::fs::GetMmcPartitionSize(&mmcPartitionSize, nn::fs::MmcPartition::BootPartition1));
        NN_LOG("Boot Partition 1 Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);

        NN_RESULT_DO(nn::fs::GetMmcPartitionSize(&mmcPartitionSize, nn::fs::MmcPartition::BootPartition2));
        NN_LOG("Boot Partition 2 Size: %d (MB)\n", mmcPartitionSize / 1024 / 1024);

        NN_RESULT_SUCCESS;
    }

    nn::Result DumpFirstArea(nn::fs::BisPartitionId partitionId)
    {
        switch (partitionId)
        {
        case nn::fs::BisPartitionId::UserDataRoot:
            NN_LOG("User Data Size\n");
            break;
        case nn::fs::BisPartitionId::BootPartition1Root:
            NN_LOG("Boot Partition 1\n");
            break;
        case nn::fs::BisPartitionId::BootPartition2Root:
            NN_LOG("Boot Partition 2\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        std::unique_ptr<nn::fs::IStorage> storage;
        NN_RESULT_DO(nn::fs::OpenBisPartition(&storage, partitionId));
        NN_RESULT_DO(storage->Read(0, g_Buffer, sizeof(g_Buffer)));

        uint32_t address = 0;
        uint8_t* pCurrentData =  g_Buffer;
        uint8_t* pDataEnd = g_Buffer + sizeof(g_Buffer);
        while (true)
        {
            NN_LOG("%08X: ", address);
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    NN_RESULT_SUCCESS;
                }
            }
            NN_LOG("\n");
            address += 0x10;
        }
    }

    nn::Result EraseAll()
    {
        NN_RESULT_DO(nn::fs::EraseMmc(nn::fs::MmcPartition::UserData));
        NN_RESULT_DO(nn::fs::EraseMmc(nn::fs::MmcPartition::BootPartition1));
        NN_RESULT_DO(nn::fs::EraseMmc(nn::fs::MmcPartition::BootPartition2));
        NN_RESULT_SUCCESS;
    }
}

extern "C" void nnMain()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(LogAllSize());

    NN_LOG("Before Erase:\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::UserDataRoot));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::BootPartition1Root));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::BootPartition2Root));

    NN_ABORT_UNLESS_RESULT_SUCCESS(EraseAll());

    NN_LOG("After Erase:\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::UserDataRoot));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::BootPartition1Root));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DumpFirstArea(nn::fs::BisPartitionId::BootPartition2Root));

    NN_LOG("Done.\n");
}
