﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
    @brief
    本体6軸センサードライバライブラリと加工処理クラスを使って6軸センサー入力を取得するためのサンプルプログラム
    (Sample 以下の本体六軸センサー Sample の互換を保ちつつ、 sasbus サービス経由で値を取得するためのコピペしたツールです)
*/

/**
    @page PageSampleLsm6ds3Simple 本体6軸センサードライバライブラリを使って6軸センサー入力を取得するためのサンプルプログラム
    @tableofcontents

    @brief
    本体6軸センサードライバライブラリを使って6軸センサー入力を取得するためのサンプルプログラムの解説です。

    @section PageSampleHidSixAxisSensor_SectionBrief 概要
    本体6軸センサー入力を取得する方法について説明します。

    @section PageSampleLsm6ds3Simple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/Lsm6ds3Simple @endlink 以下にあります。

    @section PageSampleLsm6ds3Simple_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、コンソール上に加工済みの加速度センサー値[G] が出力され続けます。

    @section PageSampleLsm6ds3Simple_SectionPrecaution 注意事項
    本サンプルプログラムでは画面描画は行いません。

    @section PageSampleLsm6ds3Simple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleLsm6ds3Simple_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - 本体6軸センサードライバライブラリ(以下、lsm6ds3 と呼びます)の初期化
    - lsm6ds3 による種々のセンサー設定操作
    - lsm6ds3 によるセンサ―生値の取得
    - センサー加工クラス SixAxisSensorProcessor への生値注入
    - SixAxisSensorProcessor から加工済みセンサー値の取得
*/

#include <cstdlib>

#include <nn/font.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/xcd/xcd_Input.h>

#include "../../../Tests/TargetDevices/lsm6ds3/detail/lsm6ds3.h"

#include "hid_SixAxisSensor/hid_SixAxisSensorProcessor.h"

namespace {

/**
 * @brief       本体6軸センサードライバ向けフルスケールレンジ(FSR)・出力データレート(ODR)・フィルタ設定です
 *
 * @details     本体6軸センサードライバ向けの設定値です。
 *              ドライバ側の設定変更を行うと、取得される生値の感度が変わります。
 */
const nnt::sasbus::lsm6ds3::AccelerometerFsr DriverAccFsr    = nnt::sasbus::lsm6ds3::AccelerometerFsr_2G;
const nnt::sasbus::lsm6ds3::AccelerometerOdr DriverAccOdr    = nnt::sasbus::lsm6ds3::AccelerometerOdr_1660Hz;
const nnt::sasbus::lsm6ds3::GyroscopeFsr     DriverGyroFsr   = nnt::sasbus::lsm6ds3::GyroscopeFsr_2000dps;
const nnt::sasbus::lsm6ds3::GyroscopeOdr     DriverGyroOdr   = nnt::sasbus::lsm6ds3::GyroscopeOdr_1660Hz;
const bool                           IsFilterEnabled = true;

/**
 * @brief       6軸センサーのキャリブレーション値 [count] です
 *
 * @details     本体6軸センサーのデバイス仕様で規定された値です。
 *              キャリブレーション値を利用者側での変更する必要はありませんが、
 *              SixAxisSensorProcessor クラスに以下のキャリブレーション値を注入する必要があります。
 */
const int16_t AccelerometerOrigin      = 0;
const int16_t AccelerometerSensitivity = 16384;
const int16_t GyroscopeOrigin          = 0;
const int16_t GyroscopeSensitivity     = 13371;

/**
 * @brief       SixAxisSensorProcessor 向けフルスケールレンジ(FSR)の設定です
 *
 * @details     内部演算でフルスケールレンジが必要になるため設定します。
 *              SixAxisSensorProcessor とドライバ側のFSR設定は同一にする必要があります。
 */
const nn::xcd::AccelerometerFsr ProcessorAccFsr  = nn::xcd::AccelerometerFsr_2G;
const nn::xcd::GyroscopeFsr     ProcessorGyroFsr = nn::xcd::GyroscopeFsr_2000dps;


const int64_t SamplingIntervalMilliSeconds = 1;                 //!< サンプリング周期の値 [ms]
const nn::util::Float3 InitialAngle = {{{ 0.0f, 0.0f, 0.0f }}}; //!< 回転角の初期値 [360度=1.0とする単位系]

static nn::hid::detail::SixAxisSensorProcessor s_Processor; //!< センサーの生値を加工するクラスです

/**
 * @brief       定期取得用のバッファです。
 *
 * @details     定期取得用のバッファです。nn::os::MemoryPageSize の整数倍である必要があります。
 */
NN_ALIGNAS(nn::os::MemoryPageSize) char PeriodicReceiveModeBuffer[nn::os::MemoryPageSize];

/**
 * @brief       ドライバから取得した生値の座標変換を行います
 *
 * @details     ドライバライブラリから取得した生値を SDK 標準の W3C座標系に変換します。
 *              https://www.w3.org/TR/orientation-event/start.png
 *
 */
void ConvertAxis(nn::xcd::SensorState* pOutAccelerationCount,
                 nn::xcd::SensorState* pOutAngularVelocityCount,
                 const nnt::sasbus::lsm6ds3::SixAxisSensorCountState& state)
{
    pOutAccelerationCount->x = state.acceleration.y;
    pOutAccelerationCount->y = - state.acceleration.x;
    pOutAccelerationCount->z = - state.acceleration.z;

    pOutAngularVelocityCount->x = - state.angularVelocity.y;
    pOutAngularVelocityCount->y = state.angularVelocity.x;
    pOutAngularVelocityCount->z = state.angularVelocity.z;
}

/**
 * @brief       SensorCalibrationValue を生成するユーティリティ関数です。
 */
void GenerateSensorCalibrationValue(nn::xcd::SensorCalibrationValue* pOutCalibrationValue,
                                    const int16_t& AccelerometerOrigin,
                                    const int16_t& AccelerometerSensitivity,
                                    const int16_t& GyroscopeOrigin,
                                    const int16_t& GyroscopeSensitivity)
{
    pOutCalibrationValue->accelerometerOrigin.x = AccelerometerOrigin;
    pOutCalibrationValue->accelerometerOrigin.y = AccelerometerOrigin;
    pOutCalibrationValue->accelerometerOrigin.z = AccelerometerOrigin;

    pOutCalibrationValue->accelerometerSensitivity.x = AccelerometerSensitivity;
    pOutCalibrationValue->accelerometerSensitivity.y = AccelerometerSensitivity;
    pOutCalibrationValue->accelerometerSensitivity.z = AccelerometerSensitivity;

    pOutCalibrationValue->gyroscopeOrigin.x = GyroscopeOrigin;
    pOutCalibrationValue->gyroscopeOrigin.y = GyroscopeOrigin;
    pOutCalibrationValue->gyroscopeOrigin.z = GyroscopeOrigin;

    pOutCalibrationValue->gyroscopeSensitivity.x = GyroscopeSensitivity;
    pOutCalibrationValue->gyroscopeSensitivity.y = GyroscopeSensitivity;
    pOutCalibrationValue->gyroscopeSensitivity.z = GyroscopeSensitivity;
}

/**
 * @brief       SixAxisSensorProcessor の初期化を行います。
 */
void InitializeSixAxisSensorProcessor(nn::hid::detail::SixAxisSensorProcessor* pOutProcessor) NN_NOEXCEPT
{
    // キャリブレーション値を SixAxisSensorProcessor クラスに注入
    nn::xcd::SensorCalibrationValue calibrationValue;
    GenerateSensorCalibrationValue(&calibrationValue,
                                   AccelerometerOrigin,
                                   AccelerometerSensitivity,
                                   GyroscopeOrigin,
                                   GyroscopeSensitivity);
    pOutProcessor->SetSensorCalibrationValue(calibrationValue);
    NN_ABORT_UNLESS(pOutProcessor->IsCalibrated(), "Calibration Failed.");

    // 回転角・姿勢の初期化
    pOutProcessor->SetDirection(nn::hid::detail::UnitDirection);
    pOutProcessor->SetAngle(InitialAngle);
    pOutProcessor->SetAccelerometerFsr(ProcessorAccFsr);
    pOutProcessor->SetGyroscopeFsr(ProcessorGyroFsr);
}

/**
 * @brief       SixAxisSensorProcessor に生値を注入します。
 */
void UpdateSixAxisSensorProcessor(nn::hid::detail::SixAxisSensorProcessor* pOutProcessor,
                                  const nnt::sasbus::lsm6ds3::SixAxisSensorCountState& state) NN_NOEXCEPT
{
    nn::xcd::SensorState accelerationCount;
    nn::xcd::SensorState angularVelocityCount;

    // 軸方向を修正
    ConvertAxis(&accelerationCount,
                &angularVelocityCount,
                state);

    // 生値を加工クラスに注入
    pOutProcessor->Sampling(accelerationCount,
                            angularVelocityCount,
                            ::nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds),
                            true);
}

/**
 * @brief       SixAxisSensorProcessor から加工済みセンサー値を取得します。
 */
void GetState(nn::hid::SixAxisSensorState* pOutStates,
              int count,
              const nn::hid::detail::SixAxisSensorProcessor& processor) NN_NOEXCEPT
{
    // 加工データを取得
    processor.GetSixAxisSensorStates(pOutStates, nn::hid::SixAxisSensorStateCountMax);
}

/**
 * @brief       加工済みセンサー値を出力します。
 */
void DumpState(const nn::hid::SixAxisSensorState& state)
{
    NN_LOG("[Acc] %f, %f, %f\n", state.acceleration.x,
                                 state.acceleration.y,
                                 state.acceleration.z);
    /*
    NN_LOG("%f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f, %f\n",
        state.acceleration.x,
        state.acceleration.y,
        state.acceleration.z,
        state.angularVelocity.x,
        state.angularVelocity.y,
        state.angularVelocity.z,
        state.angle.x,
        state.angle.y,
        state.angle.z,
        state.direction.x.x,
        state.direction.x.y,
        state.direction.x.z,
        state.direction.y.x,
        state.direction.y.y,
        state.direction.y.z,
        state.direction.z.x,
        state.direction.z.y,
        state.direction.z.z);
    */
}

}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    const size_t MallocHeapSize = 256 * 1024 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

extern "C" void nnMain()
{
    NN_LOG("/// START ///\n");

    nn::sasbus::Session                              session;

    nn::sasbus::Initialize();

    nn::sasbus::OpenSession(&session, nn::sasbus::SasbusDevice_Lsm6ds3);

    // Lsm6ds3 ドライバライブラリの初期化
    nnt::sasbus::lsm6ds3::Initialize(&session);

    // センサーの電源を入れる
    nnt::sasbus::lsm6ds3::StartAccelerometer(&session);
    nnt::sasbus::lsm6ds3::StartGyroscope(&session);

    // 各種設定
    nnt::sasbus::lsm6ds3::SetAccelerometerFsr(&session, DriverAccFsr);
    nnt::sasbus::lsm6ds3::SetAccelerometerOdr(&session, DriverAccOdr);
    nnt::sasbus::lsm6ds3::SetGyroscopeFsr(&session, DriverGyroFsr);
    nnt::sasbus::lsm6ds3::SetGyroscopeOdr(&session, DriverGyroOdr);
    nnt::sasbus::lsm6ds3::SetNoiseReductionFilter(&session, IsFilterEnabled);

    InitializeSixAxisSensorProcessor(&s_Processor);

    NN_LOG("/// SetIntervalRead ///\n");
    nnt::sasbus::lsm6ds3::StartPeriodicReceiveMode(&session, nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds), PeriodicReceiveModeBuffer, sizeof(PeriodicReceiveModeBuffer));

    nnt::sasbus::lsm6ds3::SixAxisSensorCountState state;
    nn::hid::SixAxisSensorState           states[nn::hid::SixAxisSensorStateCountMax];

    bool waits = true;
    int count = 0;
    while (count < 1000)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds));

        // デバイスドライバから生値を取得
        nnt::sasbus::lsm6ds3::GetStateForPeriodicReceiveMode(&session, &state);

        // 生値を SixAxisSensorProcessor クラスへ注入
        UpdateSixAxisSensorProcessor(&s_Processor, state);

        // SixAxisSensorProcessor から加工済みのセンサー値を取得
        GetState(states,
                 nn::hid::SixAxisSensorStateCountMax,
                 s_Processor);

        // 最新の加工済みセンサー値を出力
        DumpState(states[0]);
        count++;
    }

    // 試しに Stop してみる。
    nnt::sasbus::lsm6ds3::StopPeriodicReceiveMode(&session);
    NN_LOG("/// Stop ///\n");

    // もう一度開始
    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5000));

    NN_LOG("/// SetIntervalRead ///\n");
    nnt::sasbus::lsm6ds3::StartPeriodicReceiveMode(&session, nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds), PeriodicReceiveModeBuffer, sizeof(PeriodicReceiveModeBuffer));

    waits = true;
    while (waits)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(SamplingIntervalMilliSeconds));

        // デバイスドライバから生値を取得
        nnt::sasbus::lsm6ds3::GetStateForPeriodicReceiveMode(&session, &state);

        // 生値を SixAxisSensorProcessor クラスへ注入
        UpdateSixAxisSensorProcessor(&s_Processor, state);

        // SixAxisSensorProcessor から加工済みのセンサー値を取得
        GetState(states,
            nn::hid::SixAxisSensorStateCountMax,
            s_Processor);

        // 最新の加工済みセンサー値を出力
        DumpState(states[0]);
    }

    nnt::sasbus::lsm6ds3::StopAccelerometer(&session);
    nnt::sasbus::lsm6ds3::StopGyroscope(&session);
    nnt::sasbus::lsm6ds3::Finalize(&session);

    nn::sasbus::CloseSession(&session);
    nn::sasbus::Finalize();

    NN_LOG("/// END ///\n");
}
