﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ジャイロセンサーの処理に特化した API の宣言
 */

#pragma once

#include <nn/hid/hid_Xpad.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>
#include <nn/os/os_Tick.h>

namespace nn { namespace hid { namespace detail {

/**
* @brief       設定可能なゼロ点ドリフト設定です。
*/
enum ZeroDriftMode
{
    ZeroDriftMode_Loose = 0,     //!< センサの値がある程度変動していても、安定しているとみなして修正が行われるモードです。
    ZeroDriftMode_Standard,      //!< ZeroDriftMode_Loose と ZeroDriftMode_Tight の中間にあたるモードです。デフォルト設定されます。
    ZeroDriftMode_Tight,         //!< センサの値が定められた閾値以内の変動でないと、安定しているとみなさない厳密な修正が行われるモードです。
};

/**
 * @brief       3軸をもつセンサーの値を表す構造体です。
 */
struct ZeroDriftParameters
{
    float   radius; //!< ドリフトカウント許容半径 (1 = 360deg/sec)
    float   power;  //!< ドリフト修正パワー
    int32_t count;  //!< ドリフト修正カウント
};

const ZeroDriftParameters ZeroDrift[] =
{ //  radius  power   count
    { 0.020f, 0.040f, 200 }, // Loose
    { 0.010f, 0.020f, 200 }, // Standard
    { 0.005f, 0.010f, 200 }  // Tight
};

//!< ジャイロセンサーの処理を担うクラス
class Gyroscope final
{
    NN_DISALLOW_COPY(Gyroscope);
    NN_DISALLOW_MOVE(Gyroscope);

private:
    nn::util::Float3            m_ZeroCount;             //!< ゼロ点カウント[count]
    nn::util::Float3            m_SensitivityCount;      //!< 感度カウント[count]
    nn::util::Float3            m_Scale;                 //!< 角速度値のスケーリングパラメータ。[360dps を 1.0 とする単位系]

    nn::xcd::SensorState        m_LastAngularVelocityCount; //!< 前回算出した角速度カウント[G]

    nn::util::Float3            m_StableCountSum;        //!< 連続して静止状態となったカウント値の総和[count]
    nn::xcd::SensorState        m_StableCountTimes;      //!< 連続して静止状態となった回数

    bool                        m_EnableZeroDrift;       //!< ゼロ点自動補正するか
    bool                        m_EnableZeroPlay;        //!< ゼロ点に遊びを設けるか
    float                       m_ZeroPlayRadius;        //!< ゼロ点の遊び半径
    ZeroDriftMode               m_ZeroDriftMode;         //!< ゼロ点ドリフト設定

    nn::util::Float3            m_Magnification;          //!< 角速度を計算する際に掛ける倍率です
    nn::util::Float3            m_DirectionMagnification; //!< 姿勢値を計算する際に掛ける倍率です

    nn::xcd::GyroscopeFsr       m_Fsr;                   //!< ジャイロセンサーのフルスケールレンジ

public:
    Gyroscope()  NN_NOEXCEPT;
    ~Gyroscope() NN_NOEXCEPT {};

    //!< 角速度を取得します。
    void GetState(nn::util::Float3* pOutState, const nn::xcd::SensorState& count) NN_NOEXCEPT;

    //!< 無入力時の角速度カウントを取得します。
    void GetNoInputCount( nn::xcd::SensorState* pOutCount ) const NN_NOEXCEPT;

    //!< フルスケールレンジを設定します。
    void SetSensorFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT;

    //!< 6軸センサーのキャリブレーション値を設定します。
    void SetCalibrationValue( const nn::xcd::SensorState& gyroscopeOrigin, const nn::xcd::SensorState& gyroscopeSensitivity ) NN_NOEXCEPT;

    //!< 角速度を計算する際に掛ける倍率を各方向ごとに設定します。
    void SetMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT;

    //!< direction を内部で計算する際に使用する角速度の値にかける倍率を設定します。
    void SetDirectionMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT;

    //!< ジャイロセンサーのゼロ点ドリフト補正機能を制御/取得します。
    void ControlZeroDrift( bool enable ) NN_NOEXCEPT;
    bool IsZeroDriftEnabled() const NN_NOEXCEPT;

    //!< ジャイロセンサーのゼロ点ドリフト補正モードを設定/取得します。
    void SetZeroDriftMode( const ZeroDriftMode& mode ) NN_NOEXCEPT;
    void GetZeroDriftMode( ZeroDriftMode* pOutMode ) const NN_NOEXCEPT;
    void ResetZeroDriftMode() NN_NOEXCEPT;

    //!< ジャイロセンサーのゼロ点遊び補正機能を制御/取得します。
    void ControlZeroPlay( bool enable ) NN_NOEXCEPT;
    bool IsZeroPlayEnabled() const NN_NOEXCEPT;

    //!< ジャイロセンサーのゼロ点遊び半径を設定/取得します。
    void SetZeroPlayParameters( const float& radius ) NN_NOEXCEPT;
    void GetZeroPlayParameters( float* pOutRadius ) const NN_NOEXCEPT;
    void ResetZeroPlayParameters() NN_NOEXCEPT;

    //!< 現在の姿勢を角速度値を用いて更新します。
    void UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& angularVelocity, const float& samplingInterval ) const NN_NOEXCEPT;

    //!< 開発用途: ゼロ点を取得します。ゼロ点の動的補正が有効に働いているかの検証のため
    void GetZeroCountForDev(nn::util::Float3* pOutZeroCount ) const NN_NOEXCEPT;

private:

    //!< 角速度のスケーリングパラメータを設定します。
    void SetScaleValue() NN_NOEXCEPT;

    //!< ある軸における静止判定を行いパラメータを更新します。
    void UpdateGyroParameter(int32_t* pStableCountTimes,
                             float* pStableCountSum,
                             const int32_t& currentCount,
                             const int32_t& previousCount,
                             const float& scale) NN_NOEXCEPT;

    //!< 角速度カウントを内部パラメータを用いて更新します。
    void ReviseAngularVelocityCountByStability(nn::xcd::SensorState* pOut, const nn::xcd::SensorState& count) const NN_NOEXCEPT;

    //!< ゼロ点を入力角速度カウントに近づけます。
    void UpdateZeroCount(const nn::xcd::SensorState& count) NN_NOEXCEPT;

    //!< 内部的に保持している前回サンプルの角速度カウント値を更新します。
    void UpdateLastAngularVelocityCount(const nn::xcd::SensorState& count) NN_NOEXCEPT;

    //!< 角速度値[1.0=360dps] にカウント値を変換します。
    void ConvertToAngularVelocity(nn::util::Float3* pOut, const nn::xcd::SensorState& count) const NN_NOEXCEPT;

    //!< ゼロ点遊びにより角速度を更新します。
    void ReviseAngularVelocityByZeroPlay( nn::util::Float3* pOut ) const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
